
/*--------------------------------------------------------------------*/
/*--- begin                                      guest_mips_toIR.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, a dynamic binary instrumentation
   framework.

   Copyright (C) 2010-2017 RT-RK
      mips-valgrind@rt-rk.com

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

/* Translates MIPS code to IR. */

#include "libvex_basictypes.h"
#include "libvex_ir.h"
#include "libvex.h"
#include "libvex_guest_mips32.h"
#include "libvex_guest_mips64.h"

#include "main_util.h"
#include "main_globals.h"
#include "guest_generic_bb_to_IR.h"
#include "guest_mips_defs.h"

/*------------------------------------------------------------*/
/*---                      Globals                         ---*/
/*------------------------------------------------------------*/

/* These are set at the start of the translation of a instruction, so
   that we don't have to pass them around endlessly. CONST means does
   not change during translation of the instruction. */

/* CONST: what is the host's endianness?  This has to do with float vs
   double register accesses on VFP, but it's complex and not properly
   thought out. */
static VexEndness host_endness;

/* Pointer to the guest code area. */
static const UChar *guest_code;

/* CONST: The guest address for the instruction currently being
   translated. */
#if defined(VGP_mips32_linux)
static Addr32 guest_PC_curr_instr;
#else
static Addr64 guest_PC_curr_instr;
#endif

/* MOD: The IRSB* into which we're generating code. */
static IRSB *irsb;

/* Is our guest binary 32 or 64bit? Set at each call to
   disInstr_MIPS below. */
static Bool mode64 = False;

/* CPU has FPU and 32 dbl. prec. FP registers. */
static Bool fp_mode64 = False;

/* FPU works in FRE mode */
static Bool fp_mode64_fre = False;

/* CPU has MSA unit */
static Bool has_msa = False;

/* Define 1.0 in single and double precision. */
#define ONE_SINGLE 0x3F800000
#define ONE_DOUBLE 0x3FF0000000000000ULL

/*------------------------------------------------------------*/
/*---                  Debugging output                    ---*/
/*------------------------------------------------------------*/

#define DIP(format, args...)           \
   if (vex_traceflags & VEX_TRACE_FE)  \
      vex_printf(format, ## args)

/*------------------------------------------------------------*/
/*--- Helper bits and pieces for deconstructing the        ---*/
/*--- mips insn stream.                                    ---*/
/*------------------------------------------------------------*/

/* ---------------- Integer registers ---------------- */

static UInt integerGuestRegOffset(UInt iregNo)
{
   /* Do we care about endianness here?  We do if sub-parts of integer
      registers are accessed, but I don't think that ever happens on
      MIPS. */
   UInt ret;
   if (!mode64)
      switch (iregNo) {
         case 0:
            ret = offsetof(VexGuestMIPS32State, guest_r0); break;
         case 1:
            ret = offsetof(VexGuestMIPS32State, guest_r1); break;
         case 2:
            ret = offsetof(VexGuestMIPS32State, guest_r2); break;
         case 3:
            ret = offsetof(VexGuestMIPS32State, guest_r3); break;
         case 4:
            ret = offsetof(VexGuestMIPS32State, guest_r4); break;
         case 5:
            ret = offsetof(VexGuestMIPS32State, guest_r5); break;
         case 6:
            ret = offsetof(VexGuestMIPS32State, guest_r6); break;
         case 7:
            ret = offsetof(VexGuestMIPS32State, guest_r7); break;
         case 8:
            ret = offsetof(VexGuestMIPS32State, guest_r8); break;
         case 9:
            ret = offsetof(VexGuestMIPS32State, guest_r9); break;
         case 10:
            ret = offsetof(VexGuestMIPS32State, guest_r10); break;
         case 11:
            ret = offsetof(VexGuestMIPS32State, guest_r11); break;
         case 12:
            ret = offsetof(VexGuestMIPS32State, guest_r12); break;
         case 13:
            ret = offsetof(VexGuestMIPS32State, guest_r13); break;
         case 14:
            ret = offsetof(VexGuestMIPS32State, guest_r14); break;
         case 15:
            ret = offsetof(VexGuestMIPS32State, guest_r15); break;
         case 16:
            ret = offsetof(VexGuestMIPS32State, guest_r16); break;
         case 17:
            ret = offsetof(VexGuestMIPS32State, guest_r17); break;
         case 18:
            ret = offsetof(VexGuestMIPS32State, guest_r18); break;
         case 19:
            ret = offsetof(VexGuestMIPS32State, guest_r19); break;
         case 20:
            ret = offsetof(VexGuestMIPS32State, guest_r20); break;
         case 21:
            ret = offsetof(VexGuestMIPS32State, guest_r21); break;
         case 22:
            ret = offsetof(VexGuestMIPS32State, guest_r22); break;
         case 23:
            ret = offsetof(VexGuestMIPS32State, guest_r23); break;
         case 24:
            ret = offsetof(VexGuestMIPS32State, guest_r24); break;
         case 25:
            ret = offsetof(VexGuestMIPS32State, guest_r25); break;
         case 26:
            ret = offsetof(VexGuestMIPS32State, guest_r26); break;
         case 27:
            ret = offsetof(VexGuestMIPS32State, guest_r27); break;
         case 28:
            ret = offsetof(VexGuestMIPS32State, guest_r28); break;
         case 29:
            ret = offsetof(VexGuestMIPS32State, guest_r29); break;
         case 30:
            ret = offsetof(VexGuestMIPS32State, guest_r30); break;
         case 31:
            ret = offsetof(VexGuestMIPS32State, guest_r31); break;
         default:
            vassert(0);
            break;
      }
   else
      switch (iregNo) {
         case 0:
            ret = offsetof(VexGuestMIPS64State, guest_r0); break;
         case 1:
            ret = offsetof(VexGuestMIPS64State, guest_r1); break;
         case 2:
            ret = offsetof(VexGuestMIPS64State, guest_r2); break;
         case 3:
            ret = offsetof(VexGuestMIPS64State, guest_r3); break;
         case 4:
            ret = offsetof(VexGuestMIPS64State, guest_r4); break;
         case 5:
            ret = offsetof(VexGuestMIPS64State, guest_r5); break;
         case 6:
            ret = offsetof(VexGuestMIPS64State, guest_r6); break;
         case 7:
            ret = offsetof(VexGuestMIPS64State, guest_r7); break;
         case 8:
            ret = offsetof(VexGuestMIPS64State, guest_r8); break;
         case 9:
            ret = offsetof(VexGuestMIPS64State, guest_r9); break;
         case 10:
            ret = offsetof(VexGuestMIPS64State, guest_r10); break;
         case 11:
            ret = offsetof(VexGuestMIPS64State, guest_r11); break;
         case 12:
            ret = offsetof(VexGuestMIPS64State, guest_r12); break;
         case 13:
            ret = offsetof(VexGuestMIPS64State, guest_r13); break;
         case 14:
            ret = offsetof(VexGuestMIPS64State, guest_r14); break;
         case 15:
            ret = offsetof(VexGuestMIPS64State, guest_r15); break;
         case 16:
            ret = offsetof(VexGuestMIPS64State, guest_r16); break;
         case 17:
            ret = offsetof(VexGuestMIPS64State, guest_r17); break;
         case 18:
            ret = offsetof(VexGuestMIPS64State, guest_r18); break;
         case 19:
            ret = offsetof(VexGuestMIPS64State, guest_r19); break;
         case 20:
            ret = offsetof(VexGuestMIPS64State, guest_r20); break;
         case 21:
            ret = offsetof(VexGuestMIPS64State, guest_r21); break;
         case 22:
            ret = offsetof(VexGuestMIPS64State, guest_r22); break;
         case 23:
            ret = offsetof(VexGuestMIPS64State, guest_r23); break;
         case 24:
            ret = offsetof(VexGuestMIPS64State, guest_r24); break;
         case 25:
            ret = offsetof(VexGuestMIPS64State, guest_r25); break;
         case 26:
            ret = offsetof(VexGuestMIPS64State, guest_r26); break;
         case 27:
            ret = offsetof(VexGuestMIPS64State, guest_r27); break;
         case 28:
            ret = offsetof(VexGuestMIPS64State, guest_r28); break;
         case 29:
            ret = offsetof(VexGuestMIPS64State, guest_r29); break;
         case 30:
            ret = offsetof(VexGuestMIPS64State, guest_r30); break;
         case 31:
            ret = offsetof(VexGuestMIPS64State, guest_r31); break;
         default:
            vassert(0);
            break;
      }
   return ret;
}

#if defined(VGP_mips32_linux)
#define OFFB_PC     offsetof(VexGuestMIPS32State, guest_PC)
#else
#define OFFB_PC     offsetof(VexGuestMIPS64State, guest_PC)
#endif

/* ---------------- Floating point registers ---------------- */

static UInt floatGuestRegOffset(UInt fregNo)
{
   vassert(fregNo < 32);
   UInt ret;
   if (!mode64)
      switch (fregNo) {
         case 0:
            ret = offsetof(VexGuestMIPS32State, guest_f0); break;
         case 1:
            ret = offsetof(VexGuestMIPS32State, guest_f1); break;
         case 2:
            ret = offsetof(VexGuestMIPS32State, guest_f2); break;
         case 3:
            ret = offsetof(VexGuestMIPS32State, guest_f3); break;
         case 4:
            ret = offsetof(VexGuestMIPS32State, guest_f4); break;
         case 5:
            ret = offsetof(VexGuestMIPS32State, guest_f5); break;
         case 6:
            ret = offsetof(VexGuestMIPS32State, guest_f6); break;
         case 7:
            ret = offsetof(VexGuestMIPS32State, guest_f7); break;
         case 8:
            ret = offsetof(VexGuestMIPS32State, guest_f8); break;
         case 9:
            ret = offsetof(VexGuestMIPS32State, guest_f9); break;
         case 10:
            ret = offsetof(VexGuestMIPS32State, guest_f10); break;
         case 11:
            ret = offsetof(VexGuestMIPS32State, guest_f11); break;
         case 12:
            ret = offsetof(VexGuestMIPS32State, guest_f12); break;
         case 13:
            ret = offsetof(VexGuestMIPS32State, guest_f13); break;
         case 14:
            ret = offsetof(VexGuestMIPS32State, guest_f14); break;
         case 15:
            ret = offsetof(VexGuestMIPS32State, guest_f15); break;
         case 16:
            ret = offsetof(VexGuestMIPS32State, guest_f16); break;
         case 17:
            ret = offsetof(VexGuestMIPS32State, guest_f17); break;
         case 18:
            ret = offsetof(VexGuestMIPS32State, guest_f18); break;
         case 19:
            ret = offsetof(VexGuestMIPS32State, guest_f19); break;
         case 20:
            ret = offsetof(VexGuestMIPS32State, guest_f20); break;
         case 21:
            ret = offsetof(VexGuestMIPS32State, guest_f21); break;
         case 22:
            ret = offsetof(VexGuestMIPS32State, guest_f22); break;
         case 23:
            ret = offsetof(VexGuestMIPS32State, guest_f23); break;
         case 24:
            ret = offsetof(VexGuestMIPS32State, guest_f24); break;
         case 25:
            ret = offsetof(VexGuestMIPS32State, guest_f25); break;
         case 26:
            ret = offsetof(VexGuestMIPS32State, guest_f26); break;
         case 27:
            ret = offsetof(VexGuestMIPS32State, guest_f27); break;
         case 28:
            ret = offsetof(VexGuestMIPS32State, guest_f28); break;
         case 29:
            ret = offsetof(VexGuestMIPS32State, guest_f29); break;
         case 30:
            ret = offsetof(VexGuestMIPS32State, guest_f30); break;
         case 31:
            ret = offsetof(VexGuestMIPS32State, guest_f31); break;
         default:
            vassert(0);
            break;
      }
   else
      switch (fregNo) {
         case 0:
            ret = offsetof(VexGuestMIPS64State, guest_f0); break;
         case 1:
            ret = offsetof(VexGuestMIPS64State, guest_f1); break;
         case 2:
            ret = offsetof(VexGuestMIPS64State, guest_f2); break;
         case 3:
            ret = offsetof(VexGuestMIPS64State, guest_f3); break;
         case 4:
            ret = offsetof(VexGuestMIPS64State, guest_f4); break;
         case 5:
            ret = offsetof(VexGuestMIPS64State, guest_f5); break;
         case 6:
            ret = offsetof(VexGuestMIPS64State, guest_f6); break;
         case 7:
            ret = offsetof(VexGuestMIPS64State, guest_f7); break;
         case 8:
            ret = offsetof(VexGuestMIPS64State, guest_f8); break;
         case 9:
            ret = offsetof(VexGuestMIPS64State, guest_f9); break;
         case 10:
            ret = offsetof(VexGuestMIPS64State, guest_f10); break;
         case 11:
            ret = offsetof(VexGuestMIPS64State, guest_f11); break;
         case 12:
            ret = offsetof(VexGuestMIPS64State, guest_f12); break;
         case 13:
            ret = offsetof(VexGuestMIPS64State, guest_f13); break;
         case 14:
            ret = offsetof(VexGuestMIPS64State, guest_f14); break;
         case 15:
            ret = offsetof(VexGuestMIPS64State, guest_f15); break;
         case 16:
            ret = offsetof(VexGuestMIPS64State, guest_f16); break;
         case 17:
            ret = offsetof(VexGuestMIPS64State, guest_f17); break;
         case 18:
            ret = offsetof(VexGuestMIPS64State, guest_f18); break;
         case 19:
            ret = offsetof(VexGuestMIPS64State, guest_f19); break;
         case 20:
            ret = offsetof(VexGuestMIPS64State, guest_f20); break;
         case 21:
            ret = offsetof(VexGuestMIPS64State, guest_f21); break;
         case 22:
            ret = offsetof(VexGuestMIPS64State, guest_f22); break;
         case 23:
            ret = offsetof(VexGuestMIPS64State, guest_f23); break;
         case 24:
            ret = offsetof(VexGuestMIPS64State, guest_f24); break;
         case 25:
            ret = offsetof(VexGuestMIPS64State, guest_f25); break;
         case 26:
            ret = offsetof(VexGuestMIPS64State, guest_f26); break;
         case 27:
            ret = offsetof(VexGuestMIPS64State, guest_f27); break;
         case 28:
            ret = offsetof(VexGuestMIPS64State, guest_f28); break;
         case 29:
            ret = offsetof(VexGuestMIPS64State, guest_f29); break;
         case 30:
            ret = offsetof(VexGuestMIPS64State, guest_f30); break;
         case 31:
            ret = offsetof(VexGuestMIPS64State, guest_f31); break;
         default:
            vassert(0);
            break;
      }
   return ret;
}

/* ---------------- MIPS32 DSP ASE(r2) accumulators ---------------- */

static UInt accumulatorGuestRegOffset(UInt acNo)
{
   vassert(!mode64);
   vassert(acNo <= 3);
   UInt ret;
   switch (acNo) {
      case 0:
         ret = offsetof(VexGuestMIPS32State, guest_ac0); break;
      case 1:
         ret = offsetof(VexGuestMIPS32State, guest_ac1); break;
      case 2:
         ret = offsetof(VexGuestMIPS32State, guest_ac2); break;
      case 3:
         ret = offsetof(VexGuestMIPS32State, guest_ac3); break;
      default:
         vassert(0);
    break;
   }
   return ret;
}

/* ---------------- MIPS32 MSA registers ---------------- */

static UInt msaGuestRegOffset(UInt msaRegNo) {
   vassert(msaRegNo <= 31);
   UInt ret;

   if (mode64) {
      switch (msaRegNo) {
         case 0:
            ret = offsetof(VexGuestMIPS64State, guest_w0);
            break;

         case 1:
            ret = offsetof(VexGuestMIPS64State, guest_w1);
            break;

         case 2:
            ret = offsetof(VexGuestMIPS64State, guest_w2);
            break;

         case 3:
            ret = offsetof(VexGuestMIPS64State, guest_w3);
            break;

         case 4:
            ret = offsetof(VexGuestMIPS64State, guest_w4);
            break;

         case 5:
            ret = offsetof(VexGuestMIPS64State, guest_w5);
            break;

         case 6:
            ret = offsetof(VexGuestMIPS64State, guest_w6);
            break;

         case 7:
            ret = offsetof(VexGuestMIPS64State, guest_w7);
            break;

         case 8:
            ret = offsetof(VexGuestMIPS64State, guest_w8);
            break;

         case 9:
            ret = offsetof(VexGuestMIPS64State, guest_w9);
            break;

         case 10:
            ret = offsetof(VexGuestMIPS64State, guest_w10);
            break;

         case 11:
            ret = offsetof(VexGuestMIPS64State, guest_w11);
            break;

         case 12:
            ret = offsetof(VexGuestMIPS64State, guest_w12);
            break;

         case 13:
            ret = offsetof(VexGuestMIPS64State, guest_w13);
            break;

         case 14:
            ret = offsetof(VexGuestMIPS64State, guest_w14);
            break;

         case 15:
            ret = offsetof(VexGuestMIPS64State, guest_w15);
            break;

         case 16:
            ret = offsetof(VexGuestMIPS64State, guest_w16);
            break;

         case 17:
            ret = offsetof(VexGuestMIPS64State, guest_w17);
            break;

         case 18:
            ret = offsetof(VexGuestMIPS64State, guest_w18);
            break;

         case 19:
            ret = offsetof(VexGuestMIPS64State, guest_w19);
            break;

         case 20:
            ret = offsetof(VexGuestMIPS64State, guest_w20);
            break;

         case 21:
            ret = offsetof(VexGuestMIPS64State, guest_w21);
            break;

         case 22:
            ret = offsetof(VexGuestMIPS64State, guest_w22);
            break;

         case 23:
            ret = offsetof(VexGuestMIPS64State, guest_w23);
            break;

         case 24:
            ret = offsetof(VexGuestMIPS64State, guest_w24);
            break;

         case 25:
            ret = offsetof(VexGuestMIPS64State, guest_w25);
            break;

         case 26:
            ret = offsetof(VexGuestMIPS64State, guest_w26);
            break;

         case 27:
            ret = offsetof(VexGuestMIPS64State, guest_w27);
            break;

         case 28:
            ret = offsetof(VexGuestMIPS64State, guest_w28);
            break;

         case 29:
            ret = offsetof(VexGuestMIPS64State, guest_w29);
            break;

         case 30:
            ret = offsetof(VexGuestMIPS64State, guest_w30);
            break;

         case 31:
            ret = offsetof(VexGuestMIPS64State, guest_w31);
            break;

         default:
            vassert(0);
            break;
      }
   } else {
      switch (msaRegNo) {
         case 0:
            ret = offsetof(VexGuestMIPS32State, guest_w0);
            break;

         case 1:
            ret = offsetof(VexGuestMIPS32State, guest_w1);
            break;

         case 2:
            ret = offsetof(VexGuestMIPS32State, guest_w2);
            break;

         case 3:
            ret = offsetof(VexGuestMIPS32State, guest_w3);
            break;

         case 4:
            ret = offsetof(VexGuestMIPS32State, guest_w4);
            break;

         case 5:
            ret = offsetof(VexGuestMIPS32State, guest_w5);
            break;

         case 6:
            ret = offsetof(VexGuestMIPS32State, guest_w6);
            break;

         case 7:
            ret = offsetof(VexGuestMIPS32State, guest_w7);
            break;

         case 8:
            ret = offsetof(VexGuestMIPS32State, guest_w8);
            break;

         case 9:
            ret = offsetof(VexGuestMIPS32State, guest_w9);
            break;

         case 10:
            ret = offsetof(VexGuestMIPS32State, guest_w10);
            break;

         case 11:
            ret = offsetof(VexGuestMIPS32State, guest_w11);
            break;

         case 12:
            ret = offsetof(VexGuestMIPS32State, guest_w12);
            break;

         case 13:
            ret = offsetof(VexGuestMIPS32State, guest_w13);
            break;

         case 14:
            ret = offsetof(VexGuestMIPS32State, guest_w14);
            break;

         case 15:
            ret = offsetof(VexGuestMIPS32State, guest_w15);
            break;

         case 16:
            ret = offsetof(VexGuestMIPS32State, guest_w16);
            break;

         case 17:
            ret = offsetof(VexGuestMIPS32State, guest_w17);
            break;

         case 18:
            ret = offsetof(VexGuestMIPS32State, guest_w18);
            break;

         case 19:
            ret = offsetof(VexGuestMIPS32State, guest_w19);
            break;

         case 20:
            ret = offsetof(VexGuestMIPS32State, guest_w20);
            break;

         case 21:
            ret = offsetof(VexGuestMIPS32State, guest_w21);
            break;

         case 22:
            ret = offsetof(VexGuestMIPS32State, guest_w22);
            break;

         case 23:
            ret = offsetof(VexGuestMIPS32State, guest_w23);
            break;

         case 24:
            ret = offsetof(VexGuestMIPS32State, guest_w24);
            break;

         case 25:
            ret = offsetof(VexGuestMIPS32State, guest_w25);
            break;

         case 26:
            ret = offsetof(VexGuestMIPS32State, guest_w26);
            break;

         case 27:
            ret = offsetof(VexGuestMIPS32State, guest_w27);
            break;

         case 28:
            ret = offsetof(VexGuestMIPS32State, guest_w28);
            break;

         case 29:
            ret = offsetof(VexGuestMIPS32State, guest_w29);
            break;

         case 30:
            ret = offsetof(VexGuestMIPS32State, guest_w30);
            break;

         case 31:
            ret = offsetof(VexGuestMIPS32State, guest_w31);
            break;

         default:
            vassert(0);
            break;
      }
   }

   return ret;
}


/* Do a endian load of a 32-bit word, regardless of the endianness of the
   underlying host. */
static inline UInt getUInt(const UChar * p)
{
   UInt w = 0;
#if defined (_MIPSEL)
   w = (w << 8) | p[3];
   w = (w << 8) | p[2];
   w = (w << 8) | p[1];
   w = (w << 8) | p[0];
#elif defined (_MIPSEB)
   w = (w << 8) | p[0];
   w = (w << 8) | p[1];
   w = (w << 8) | p[2];
   w = (w << 8) | p[3];
#endif
   return w;
}

#define BITS2(_b1,_b0) \
   (((_b1) << 1) | (_b0))

#define BITS3(_b2,_b1,_b0)                      \
  (((_b2) << 2) | ((_b1) << 1) | (_b0))

#define BITS4(_b3,_b2,_b1,_b0) \
   (((_b3) << 3) | ((_b2) << 2) | ((_b1) << 1) | (_b0))

#define BITS5(_b4,_b3,_b2,_b1,_b0)  \
   (((_b4) << 4) | BITS4((_b3),(_b2),(_b1),(_b0)))

#define BITS6(_b5,_b4,_b3,_b2,_b1,_b0)  \
   ((BITS2((_b5),(_b4)) << 4) \
    | BITS4((_b3),(_b2),(_b1),(_b0)))

#define BITS8(_b7,_b6,_b5,_b4,_b3,_b2,_b1,_b0)  \
   ((BITS4((_b7),(_b6),(_b5),(_b4)) << 4) \
    | BITS4((_b3),(_b2),(_b1),(_b0)))

#define LOAD_STORE_PATTERN \
   t1 = newTemp(mode64 ? Ity_I64 : Ity_I32); \
      if(!mode64) \
         assign(t1, binop(Iop_Add32, getIReg(rs), \
                                     mkU32(extend_s_16to32(imm)))); \
      else \
         assign(t1, binop(Iop_Add64, getIReg(rs), \
                                     mkU64(extend_s_16to64(imm)))); \

#define LOAD_STORE_PATTERN_MSA(imm) \
   t1 = newTemp(mode64 ? Ity_I64 : Ity_I32); \
      if (!mode64) \
         assign(t1, binop(Iop_Add32, getIReg(ws),  \
                                     mkU32(extend_s_10to32(imm)))); \
      else \
         assign(t1, binop(Iop_Add64, getIReg(ws), \
                                     mkU64(extend_s_10to64(imm)))); \

#define LOADX_STORE_PATTERN \
   t1 = newTemp(mode64 ? Ity_I64 : Ity_I32); \
      if(!mode64) \
         assign(t1, binop(Iop_Add32, getIReg(regRs), getIReg(regRt))); \
      else \
         assign(t1, binop(Iop_Add64, getIReg(regRs), getIReg(regRt)));

#define LWX_SWX_PATTERN64 \
   t2 = newTemp(Ity_I64); \
   assign(t2, binop(Iop_And64, mkexpr(t1), mkU64(0xFFFFFFFFFFFFFFFCULL))); \
   t4 = newTemp(Ity_I32); \
   assign(t4, mkNarrowTo32( ty, binop(Iop_And64, \
                                      mkexpr(t1), mkU64(0x3))));

#define LWX_SWX_PATTERN64_1 \
   t2 = newTemp(Ity_I64); \
   assign(t2, binop(Iop_And64, mkexpr(t1), mkU64(0xFFFFFFFFFFFFFFF8ULL))); \
   t4 = newTemp(Ity_I64); \
   assign(t4, binop(Iop_And64, mkexpr(t1), mkU64(0x7)));

#define LWX_SWX_PATTERN \
   t2 = newTemp(Ity_I32); \
   assign(t2, binop(Iop_And32, mkexpr(t1), mkU32(0xFFFFFFFC))); \
   t4 = newTemp(Ity_I32); \
   assign(t4, binop(Iop_And32, mkexpr(t1), mkU32(0x00000003)))

#define SXXV_PATTERN(op) \
   putIReg(rd, binop(op, \
         getIReg(rt), \
            unop(Iop_32to8, \
               binop(Iop_And32, \
                  getIReg(rs), \
                  mkU32(0x0000001F) \
               ) \
            ) \
         ) \
      )

#define SXXV_PATTERN64(op) \
   putIReg(rd, mkWidenFrom32(ty, binop(op, \
           mkNarrowTo32(ty, getIReg(rt)), \
             unop(Iop_32to8, \
                binop(Iop_And32, \
                   mkNarrowTo32(ty, getIReg(rs)), \
                   mkU32(0x0000001F) \
                ) \
             ) \
          ), True \
       ))

#define SXX_PATTERN(op) \
   putIReg(rd, binop(op, getIReg(rt), mkU8(sa)));

#define ALU_PATTERN(op) \
   putIReg(rd, binop(op, getIReg(rs), getIReg(rt)));

#define ALUI_PATTERN(op) \
   putIReg(rt, binop(op, getIReg(rs), mkU32(imm)));

#define ALUI_PATTERN64(op) \
   putIReg(rt, binop(op, getIReg(rs), mkU64(imm)));

#define ALU_PATTERN64(op) \
   putIReg(rd, mkWidenFrom32(ty, binop(op, \
                             mkNarrowTo32(ty, getIReg(rs)), \
                             mkNarrowTo32(ty, getIReg(rt))), True));

#define FP_CONDITIONAL_CODE \
   t3 = newTemp(Ity_I32);   \
   assign(t3, binop(Iop_And32, \
                 IRExpr_ITE( binop(Iop_CmpEQ32, mkU32(cc), mkU32(0)), \
                             binop(Iop_Shr32, getFCSR(), mkU8(23)), \
                             binop(Iop_Shr32, getFCSR(), mkU8(24+cc))), \
                 mkU32(0x1)));

#define ILLEGAL_INSTRUCTON \
   putPC(mkU32(guest_PC_curr_instr + 4)); \
   dres.jk_StopHere = Ijk_SigILL; \
   dres.whatNext    = Dis_StopHere;

#define LLADDR_INVALID \
   (mode64 ? mkU64(0xFFFFFFFFFFFFFFFFULL) : mkU32(0xFFFFFFFF))

/*------------------------------------------------------------*/
/*---                  Field helpers                       ---*/
/*------------------------------------------------------------*/

static UInt get_opcode(UInt mipsins)
{
   return (0xFC000000 & mipsins) >> 26;
}

static UInt get_rs(UInt mipsins)
{
   return (0x03E00000 & mipsins) >> 21;
}

static UInt get_rt(UInt mipsins)
{
   return (0x001F0000 & mipsins) >> 16;
}

static UInt get_imm(UInt mipsins)
{
   return (0x0000FFFF & mipsins);
}

static UInt get_instr_index(UInt mipsins)
{
   return (0x03FFFFFF & mipsins);
}

static UInt get_rd(UInt mipsins)
{
   return (0x0000F800 & mipsins) >> 11;
}

static UInt get_sa(UInt mipsins)
{
   return (0x000007C0 & mipsins) >> 6;
}

static UInt get_function(UInt mipsins)
{
   return (0x0000003F & mipsins);
}

static UInt get_ft(UInt mipsins)
{
   return (0x001F0000 & mipsins) >> 16;
}

static UInt get_fs(UInt mipsins)
{
   return (0x0000F800 & mipsins) >> 11;
}

static UInt get_fd(UInt mipsins)
{
   return (0x000007C0 & mipsins) >> 6;
}

static UInt get_mov_cc(UInt mipsins)
{
   return (0x001C0000 & mipsins) >> 18;
}

static UInt get_bc1_cc(UInt mipsins)
{
   return (0x001C0000 & mipsins) >> 18;
}

static UInt get_fpc_cc(UInt mipsins)
{
   return (0x00000700 & mipsins) >> 8;
}

static UInt get_tf(UInt mipsins)
{
   return (0x00010000 & mipsins) >> 16;
}

static UInt get_nd(UInt mipsins)
{
   return (0x00020000 & mipsins) >> 17;
}

static UInt get_fmt(UInt mipsins)
{
   return (0x03E00000 & mipsins) >> 21;
}

static UInt get_FC(UInt mipsins)
{
   return (0x000000F0 & mipsins) >> 4;
}

static UInt get_cond(UInt mipsins)
{
   return (0x0000000F & mipsins);
}

/* for break & syscall */
static UInt get_code(UInt mipsins)
{
   return (0xFFC0 & mipsins) >> 6;
}

static UInt get_lsb(UInt mipsins)
{
   return (0x7C0 & mipsins) >> 6;
}

static UInt get_msb(UInt mipsins)
{
   return (0x0000F800 & mipsins) >> 11;
}

static UInt get_rot(UInt mipsins)
{
   return (0x00200000 & mipsins) >> 21;
}

static UInt get_rotv(UInt mipsins)
{
   return (0x00000040 & mipsins) >> 6;
}

static UInt get_sel(UInt mipsins)
{
   return (0x00000007 & mipsins);
}

/* Get acc number for all MIPS32 DSP ASE(r2) instructions that use them,
   except for MFHI and MFLO. */
static UInt get_acNo(UInt mipsins)
{
   return (0x00001800 & mipsins) >> 11;
}

/* Get accumulator number for MIPS32 DSP ASEr2 MFHI and MFLO instructions. */
static UInt get_acNo_mfhilo(UInt mipsins)
{
   return (0x00600000 & mipsins) >> 21;
}

/* Get mask field (helper function for wrdsp instruction). */
static UInt get_wrdspMask(UInt mipsins)
{
   return (0x001ff800 & mipsins) >> 11;
}

/* Get mask field (helper function for rddsp instruction). */
static UInt get_rddspMask(UInt mipsins)
{
   return (0x03ff0000 & mipsins) >> 16;
}

/* Get shift field (helper function for DSP ASE instructions). */
static UInt get_shift(UInt mipsins)
{
   return (0x03f00000 & mipsins) >> 20;
}

/* Get immediate field for DSP ASE instructions. */
static UInt get_dspImm(UInt mipsins)
{
   return (0x03ff0000 & mipsins) >> 16;
}

static Bool branch_or_jump(const UChar * addr)
{
   UInt fmt;
   UInt cins = getUInt(addr);

   UInt opcode = get_opcode(cins);
   UInt rt = get_rt(cins);
   UInt function = get_function(cins);

   /* bgtz, blez, bne, beq, jal */
   if (opcode == 0x07 || opcode == 0x06 || opcode == 0x05 || opcode == 0x04
       || opcode == 0x03 || opcode == 0x02) {
      return True;
   }

   /* bgez */
   if (opcode == 0x01 && rt == 0x01) {
      return True;
   }

   /* bgezal */
   if (opcode == 0x01 && rt == 0x11) {
      return True;
   }

   /* bltzal */
   if (opcode == 0x01 && rt == 0x10) {
      return True;
   }

   /* bltz */
   if (opcode == 0x01 && rt == 0x00) {
      return True;
   }

   /* jalr */
   if (opcode == 0x00 && function == 0x09) {
      return True;
   }

   /* jr */
   if (opcode == 0x00 && function == 0x08) {
      return True;
   }

   if (opcode == 0x11) {
      /* bc1f & bc1t */
      fmt = get_fmt(cins);
      if (fmt == 0x08) {
         return True;
      }

      /* MSA branches */
      /* bnz.df, bz.df */
      if (fmt >= 0x18) {
         return True;
      }
      /* bnz.v */
      if (fmt == 0x0f) {
         return True;
      }
      /* bz.v */
      if (fmt == 0x0b) {
         return True;
      }

      /* R6 branches */
      /* bc1eqz */
      if (fmt == 0x09) {
         return True;
      }

      /* bc1nez */
      if (fmt == 0x0D) {
         return True;
      }
   }

   /* bposge32 */
   if (opcode == 0x01 && rt == 0x1c) {
      return True;
   }

   /* Cavium Specific instructions. */
   if (opcode == 0x32 || opcode == 0x3A || opcode == 0x36 || opcode == 0x3E) {
       /* BBIT0, BBIT1, BBIT032, BBIT132 */
      return True;
   }

   return False;
}

static Bool is_Branch_or_Jump_and_Link(const UChar * addr)
{
   UInt cins = getUInt(addr);

   UInt opcode = get_opcode(cins);
   UInt rt = get_rt(cins);
   UInt function = get_function(cins);

   /* jal */
   if (opcode == 0x02) {
      return True;
   }

   /* bgezal or bal(r6) */
   if (opcode == 0x01 && rt == 0x11) {
      return True;
   }

   /* bltzal */
   if (opcode == 0x01 && rt == 0x10) {
      return True;
   }

   /* jalr */
   if (opcode == 0x00 && function == 0x09) {
      return True;
   }

   return False;
}

static Bool branch_or_link_likely(const UChar * addr)
{
   UInt cins = getUInt(addr);
   UInt opcode = get_opcode(cins);
   UInt rt = get_rt(cins);

   /* bgtzl, blezl, bnel, beql */
   if (opcode == 0x17 || opcode == 0x16 || opcode == 0x15 || opcode == 0x14)
      return True;

   /* bgezl */
   if (opcode == 0x01 && rt == 0x03)
      return True;

   /* bgezall */
   if (opcode == 0x01 && rt == 0x13)
      return True;

   /* bltzall */
   if (opcode == 0x01 && rt == 0x12)
      return True;

   /* bltzl */
   if (opcode == 0x01 && rt == 0x02)
      return True;

   return False;
}

/*------------------------------------------------------------*/
/*--- Helper bits and pieces for creating IR fragments.    ---*/
/*------------------------------------------------------------*/

static IRExpr *mkU8(UInt i)
{
   vassert(i < 256);
   return IRExpr_Const(IRConst_U8((UChar) i));
}

/* Create an expression node for a 16-bit integer constant. */
static IRExpr *mkU16(UInt i)
{
   return IRExpr_Const(IRConst_U16(i));
}

/* Create an expression node for a 32-bit integer constant. */
static IRExpr *mkU32(UInt i)
{
   return IRExpr_Const(IRConst_U32(i));
}

/* Create an expression node for a 64-bit integer constant. */
static IRExpr *mkU64(ULong i)
{
   return IRExpr_Const(IRConst_U64(i));
}

static IRExpr *mkexpr(IRTemp tmp)
{
   return IRExpr_RdTmp(tmp);
}

static IRExpr *unop(IROp op, IRExpr * a)
{
   return IRExpr_Unop(op, a);
}

static IRExpr *binop(IROp op, IRExpr * a1, IRExpr * a2)
{
   return IRExpr_Binop(op, a1, a2);
}

static IRExpr *triop(IROp op, IRExpr * a1, IRExpr * a2, IRExpr * a3)
{
   return IRExpr_Triop(op, a1, a2, a3);
}

static IRExpr *qop ( IROp op, IRExpr * a1, IRExpr * a2, IRExpr * a3,
                     IRExpr * a4 )
{
   return IRExpr_Qop(op, a1, a2, a3, a4);
}

static IRExpr *load(IRType ty, IRExpr * addr)
{
   IRExpr *load1 = NULL;
#if defined (_MIPSEL)
   load1 = IRExpr_Load(Iend_LE, ty, addr);
#elif defined (_MIPSEB)
   load1 = IRExpr_Load(Iend_BE, ty, addr);
#endif
   return load1;
}

/* Add a statement to the list held by "irsb". */
static void stmt(IRStmt * st)
{
   addStmtToIRSB(irsb, st);
}

static void assign(IRTemp dst, IRExpr * e)
{
   stmt(IRStmt_WrTmp(dst, e));
}

static void store(IRExpr * addr, IRExpr * data)
{
#if defined (_MIPSEL)
   stmt(IRStmt_Store(Iend_LE, addr, data));
#elif defined (_MIPSEB)
   stmt(IRStmt_Store(Iend_BE, addr, data));
#endif
}

/* Generate a new temporary of the given type. */
static IRTemp newTemp(IRType ty)
{
   vassert(isPlausibleIRType(ty));
   return newIRTemp(irsb->tyenv, ty);
}

/* Generate an expression for SRC rotated right by ROT. */
static IRExpr *genROR32(IRExpr * src, Int rot)
{
   vassert(rot >= 0 && rot < 32);
   if (rot == 0)
      return src;
   return binop(Iop_Or32, binop(Iop_Shl32, src, mkU8(32 - rot)),
                          binop(Iop_Shr32, src, mkU8(rot)));
}

static IRExpr *genRORV32(IRExpr * src, IRExpr * rs)
{
   IRTemp t0 = newTemp(Ity_I8);
   IRTemp t1 = newTemp(Ity_I8);

   assign(t0, unop(Iop_32to8, binop(Iop_And32, rs, mkU32(0x0000001F))));
   assign(t1, binop(Iop_Sub8, mkU8(32), mkexpr(t0)));
   return binop(Iop_Or32, binop(Iop_Shl32, src, mkexpr(t1)),
                          binop(Iop_Shr32, src, mkexpr(t0)));
}


static UShort extend_s_9to16(UInt x)
{
   return (UShort) ((((Int) x) << 23) >> 23);
}

static UShort extend_s_10to16(UInt x)
{
   return (UShort) ((((Int) x) << 22) >> 22);
}

static UInt extend_s_10to32(UInt x)
{
   return (UInt)((((Int) x) << 22) >> 22);
}

static ULong extend_s_10to64(UInt x)
{
   return (ULong)((((Long) x) << 54) >> 54);
}

static UInt extend_s_16to32(UInt x)
{
   return (UInt) ((((Int) x) << 16) >> 16);
}

static UInt extend_s_18to32(UInt x)
{
   return (UInt) ((((Int) x) << 14) >> 14);
}

static UInt extend_s_19to32(UInt x)
{
   return (UInt) ((((Int) x) << 13) >> 13);
}

static UInt extend_s_23to32(UInt x)
{
   return (UInt) ((((Int) x) << 9) >> 9);
}

static UInt extend_s_26to32(UInt x)
{
   return (UInt) ((((Int) x) << 6) >> 6);
}

static ULong extend_s_16to64 ( UInt x )
{
   return (ULong) ((((Long) x) << 48) >> 48);
}

static ULong extend_s_18to64 ( UInt x )
{
   return (ULong) ((((Long) x) << 46) >> 46);
}

static ULong extend_s_19to64(UInt x)
{
   return (ULong) ((((Long) x) << 45) >> 45);
}

static ULong extend_s_23to64(UInt x)
{
   return (ULong) ((((Long) x) << 41) >> 41);
}

static ULong extend_s_26to64(UInt x)
{
   return (ULong) ((((Long) x) << 38) >> 38);
}

static ULong extend_s_32to64 ( UInt x )
{
   return (ULong) ((((Long) x) << 32) >> 32);
}

static void jmp_lit32 ( /*MOD*/ DisResult* dres, IRJumpKind kind, Addr32 d32 )
{
   vassert(dres->whatNext    == Dis_Continue);
   vassert(dres->len         == 0);
   vassert(dres->continueAt  == 0);
   vassert(dres->jk_StopHere == Ijk_INVALID);
   dres->whatNext    = Dis_StopHere;
   dres->jk_StopHere = kind;
   stmt( IRStmt_Put( OFFB_PC, mkU32(d32) ) );
}

static void jmp_lit64 ( /*MOD*/ DisResult* dres, IRJumpKind kind, Addr64 d64 )
{
   vassert(dres->whatNext    == Dis_Continue);
   vassert(dres->len         == 0);
   vassert(dres->continueAt  == 0);
   vassert(dres->jk_StopHere == Ijk_INVALID);
   dres->whatNext    = Dis_StopHere;
   dres->jk_StopHere = kind;
   stmt(IRStmt_Put(OFFB_PC, mkU64(d64)));
}

/* Get value from accumulator (helper function for MIPS32 DSP ASE instructions).
   This function should be called before any other operation if widening
   multiplications are used. */
static IRExpr *getAcc(UInt acNo)
{
   vassert(!mode64);
   vassert(acNo <= 3);
   return IRExpr_Get(accumulatorGuestRegOffset(acNo), Ity_I64);
}

/* Get value from DSPControl register (helper function for MIPS32 DSP ASE
   instructions). */
static IRExpr *getDSPControl(void)
{
   vassert(!mode64);
   return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_DSPControl), Ity_I32);
}

/* Put value to DSPControl register. Expression e is written to DSPControl as
   is. If only certain bits of DSPControl need to be changed, it should be done
   before calling putDSPControl(). It could be done by reading DSPControl and
   ORing it with appropriate mask. */
static void putDSPControl(IRExpr * e)
{
   vassert(!mode64);
   stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_DSPControl), e));
}

/* Fetch a byte from the guest insn stream. */
static UChar getIByte(Int delta)
{
   return guest_code[delta];
}

static IRExpr *getIReg(UInt iregNo)
{
   if (0 == iregNo) {
      return mode64 ? mkU64(0x0) : mkU32(0x0);
   } else {
      IRType ty = mode64 ? Ity_I64 : Ity_I32;
      vassert(iregNo < 32);
      return IRExpr_Get(integerGuestRegOffset(iregNo), ty);
   }
}


static IRExpr *getWReg(UInt wregNo) {
   vassert(wregNo <= 31);
   return IRExpr_Get(msaGuestRegOffset(wregNo), Ity_V128);
}

static IRExpr *getHI(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_HI), Ity_I64);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_HI), Ity_I32);
}

static IRExpr *getLO(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_LO), Ity_I64);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_LO), Ity_I32);
}

static IRExpr *getFCSR(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_FCSR), Ity_I32);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_FCSR), Ity_I32);
}

static IRExpr *getLLaddr(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_LLaddr), Ity_I64);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_LLaddr), Ity_I32);
}

static IRExpr *getLLdata(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_LLdata), Ity_I64);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_LLdata), Ity_I32);
}

static IRExpr *getMSACSR(void) {
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_MSACSR), Ity_I32);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_MSACSR), Ity_I32);
}

/* Get byte from register reg, byte pos from 0 to 3 (or 7 for MIPS64) . */
static IRExpr *getByteFromReg(UInt reg, UInt byte_pos)
{
  UInt pos = byte_pos * 8;
  if (mode64)
      return unop(Iop_64to8, binop(Iop_And64,
                                   binop(Iop_Shr64, getIReg(reg), mkU8(pos)),
                                   mkU64(0xFF)));
   else
      return unop(Iop_32to8, binop(Iop_And32,
                                   binop(Iop_Shr32, getIReg(reg), mkU8(pos)),
                                   mkU32(0xFF)));
}

static void putFCSR(IRExpr * e)
{
   if (mode64)
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_FCSR), e));
   else
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_FCSR), e));
}

static void putLLaddr(IRExpr * e)
{
   if (mode64)
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_LLaddr), e));
   else
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_LLaddr), e));
}

static void putLLdata(IRExpr * e)
{
   if (mode64)
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_LLdata), e));
   else
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_LLdata), e));
}

static void putMSACSR(IRExpr * e) {
   if (mode64)
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_MSACSR), e));
   else
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_MSACSR), e));
}

/* fs   - fpu source register number.
   inst - fpu instruction that needs to be executed.
   sz32 - size of source register.
   opN  - number of operads:
          1 - unary operation.
          2 - binary operation. */
static void calculateFCSR(UInt fs, UInt ft, UInt inst, Bool sz32, UInt opN)
{
   IRDirty *d;
   IRTemp fcsr = newTemp(Ity_I32);
   /* IRExpr_GSPTR() => Need to pass pointer to guest state to helper. */
   if (fp_mode64)
      d = unsafeIRDirty_1_N(fcsr, 0,
                            "mips_dirtyhelper_calculate_FCSR_fp64",
                            &mips_dirtyhelper_calculate_FCSR_fp64,
                            mkIRExprVec_4(IRExpr_GSPTR(),
                                          mkU32(fs),
                                          mkU32(ft),
                                          mkU32(inst)));
   else
      d = unsafeIRDirty_1_N(fcsr, 0,
                            "mips_dirtyhelper_calculate_FCSR_fp32",
                            &mips_dirtyhelper_calculate_FCSR_fp32,
                            mkIRExprVec_4(IRExpr_GSPTR(),
                                          mkU32(fs),
                                          mkU32(ft),
                                          mkU32(inst)));

   if (opN == 1) {  /* Unary operation. */
      /* Declare we're reading guest state. */
      if (sz32 || fp_mode64)
         d->nFxState = 2;
      else
         d->nFxState = 3;
      vex_bzero(&d->fxState, sizeof(d->fxState));

      d->fxState[0].fx     = Ifx_Read;  /* read */
      if (mode64)
         d->fxState[0].offset = offsetof(VexGuestMIPS64State, guest_FCSR);
      else
         d->fxState[0].offset = offsetof(VexGuestMIPS32State, guest_FCSR);
      d->fxState[0].size   = sizeof(UInt);
      d->fxState[1].fx     = Ifx_Read;  /* read */
      d->fxState[1].offset = floatGuestRegOffset(fs);
      d->fxState[1].size   = sizeof(ULong);

      if (!(sz32 || fp_mode64)) {
         d->fxState[2].fx     = Ifx_Read;  /* read */
         d->fxState[2].offset = floatGuestRegOffset(fs+1);
         d->fxState[2].size   = sizeof(ULong);
      }
   } else if (opN == 2) {  /* Binary operation. */
      /* Declare we're reading guest state. */
      if (sz32 || fp_mode64)
         d->nFxState = 3;
      else
         d->nFxState = 5;
      vex_bzero(&d->fxState, sizeof(d->fxState));

      d->fxState[0].fx     = Ifx_Read;  /* read */
      if (mode64)
         d->fxState[0].offset = offsetof(VexGuestMIPS64State, guest_FCSR);
      else
         d->fxState[0].offset = offsetof(VexGuestMIPS32State, guest_FCSR);
      d->fxState[0].size   = sizeof(UInt);
      d->fxState[1].fx     = Ifx_Read;  /* read */
      d->fxState[1].offset = floatGuestRegOffset(fs);
      d->fxState[1].size   = sizeof(ULong);
      d->fxState[2].fx     = Ifx_Read;  /* read */
      d->fxState[2].offset = floatGuestRegOffset(ft);
      d->fxState[2].size   = sizeof(ULong);

      if (!(sz32 || fp_mode64)) {
         d->fxState[3].fx     = Ifx_Read;  /* read */
         d->fxState[3].offset = floatGuestRegOffset(fs+1);
         d->fxState[3].size   = sizeof(ULong);
         d->fxState[4].fx     = Ifx_Read;  /* read */
         d->fxState[4].offset = floatGuestRegOffset(ft+1);
         d->fxState[4].size   = sizeof(ULong);
      }
   }

   stmt(IRStmt_Dirty(d));

   putFCSR(mkexpr(fcsr));
}

/* ws, wt - source MSA register numbers.
   inst   - MSA fp instruction that needs to be executed.
   opN    - number of operads:
             1 - unary operation.
             2 - binary operation. */
static void calculateMSACSR(UInt ws, UInt wt, UInt inst, UInt opN) {
   IRDirty *d;
   IRTemp msacsr = newTemp(Ity_I32);
   /* IRExpr_BBPTR() => Need to pass pointer to guest state to helper. */
   d = unsafeIRDirty_1_N(msacsr, 0,
                         "mips_dirtyhelper_calculate_MSACSR",
                         &mips_dirtyhelper_calculate_MSACSR,
                         mkIRExprVec_4(IRExpr_GSPTR(),
                                       mkU32(ws),
                                       mkU32(wt),
                                       mkU32(inst)));

   if (opN == 1) {  /* Unary operation. */
      /* Declare we're reading guest state. */
      d->nFxState = 2;
      vex_bzero(&d->fxState, sizeof(d->fxState));
      d->fxState[0].fx     = Ifx_Read;  /* read */

      if (mode64)
         d->fxState[0].offset = offsetof(VexGuestMIPS64State, guest_MSACSR);
      else
         d->fxState[0].offset = offsetof(VexGuestMIPS32State, guest_MSACSR);

      d->fxState[0].size   = sizeof(UInt);
      d->fxState[1].fx     = Ifx_Read;  /* read */
      d->fxState[1].offset = msaGuestRegOffset(ws);
      d->fxState[1].size   = sizeof(ULong);
   } else if (opN == 2) {  /* Binary operation. */
      /* Declare we're reading guest state. */
      d->nFxState = 3;
      vex_bzero(&d->fxState, sizeof(d->fxState));
      d->fxState[0].fx     = Ifx_Read;  /* read */

      if (mode64)
         d->fxState[0].offset = offsetof(VexGuestMIPS64State, guest_MSACSR);
      else
         d->fxState[0].offset = offsetof(VexGuestMIPS32State, guest_MSACSR);

      d->fxState[0].size   = sizeof(UInt);
      d->fxState[1].fx     = Ifx_Read;  /* read */
      d->fxState[1].offset = msaGuestRegOffset(ws);
      d->fxState[1].size   = sizeof(ULong);
      d->fxState[2].fx     = Ifx_Read;  /* read */
      d->fxState[2].offset = msaGuestRegOffset(wt);
      d->fxState[2].size   = sizeof(ULong);
   }

   stmt(IRStmt_Dirty(d));
   putMSACSR(mkexpr(msacsr));
}

static IRExpr *getULR(void)
{
   if (mode64)
      return IRExpr_Get(offsetof(VexGuestMIPS64State, guest_ULR), Ity_I64);
   else
      return IRExpr_Get(offsetof(VexGuestMIPS32State, guest_ULR), Ity_I32);
}

static void putIReg(UInt archreg, IRExpr * e)
{
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   vassert(archreg < 32);
   vassert(typeOfIRExpr(irsb->tyenv, e) == ty);
   if (archreg != 0)
      stmt(IRStmt_Put(integerGuestRegOffset(archreg), e));
}

static void putWReg(UInt wregNo, IRExpr * e) {
   vassert(wregNo <= 31);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_V128);
   stmt(IRStmt_Put(msaGuestRegOffset(wregNo), e));
   stmt(IRStmt_Put(floatGuestRegOffset(wregNo),
                   unop(Iop_ReinterpI64asF64, unop(Iop_V128to64, e))));
}

static IRExpr *mkNarrowTo32(IRType ty, IRExpr * src)
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to32, src) : src;
}

static void putLO(IRExpr * e)
{
   if (mode64) {
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_LO), e));
   } else {
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_LO), e));
   /* Add value to lower 32 bits of ac0 to maintain compatibility between
      regular MIPS32 instruction set and MIPS DSP ASE. Keep higher 32bits
      unchanged. */
      IRTemp t_lo = newTemp(Ity_I32);
      IRTemp t_hi = newTemp(Ity_I32);
      assign(t_lo, e);
      assign(t_hi, unop(Iop_64HIto32, getAcc(0)));
      stmt(IRStmt_Put(accumulatorGuestRegOffset(0),
           binop(Iop_32HLto64, mkexpr(t_hi), mkexpr(t_lo))));
   }
}

static void putHI(IRExpr * e)
{
   if (mode64) {
      stmt(IRStmt_Put(offsetof(VexGuestMIPS64State, guest_HI), e));
   } else {
      stmt(IRStmt_Put(offsetof(VexGuestMIPS32State, guest_HI), e));
   /* Add value to higher 32 bits of ac0 to maintain compatibility between
      regular MIPS32 instruction set and MIPS DSP ASE. Keep lower 32bits
      unchanged. */
      IRTemp t_lo = newTemp(Ity_I32);
      IRTemp t_hi = newTemp(Ity_I32);
      assign(t_hi, e);
      assign(t_lo, unop(Iop_64to32, getAcc(0)));
      stmt(IRStmt_Put(accumulatorGuestRegOffset(0),
           binop(Iop_32HLto64, mkexpr(t_hi), mkexpr(t_lo))));
   }
}

/* Put value to accumulator(helper function for MIPS32 DSP ASE instructions). */
static void putAcc(UInt acNo, IRExpr * e)
{
   vassert(!mode64);
   vassert(acNo <= 3);
   vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_I64);
   stmt(IRStmt_Put(accumulatorGuestRegOffset(acNo), e));
/* If acNo = 0, split value to HI and LO regs in order to maintain compatibility
   between MIPS32 and MIPS DSP ASE insn sets. */
   if (0 == acNo) {
     putLO(unop(Iop_64to32, e));
     putHI(unop(Iop_64HIto32, e));
   }
}

static IRExpr *mkNarrowTo8 ( IRType ty, IRExpr * src )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to8, src) : unop(Iop_32to8, src);
}

static IRExpr *mkNarrowTo16 ( IRType ty, IRExpr * src )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? unop(Iop_64to16, src) : unop(Iop_32to16, src);
}

static void putPC(IRExpr * e)
{
   stmt(IRStmt_Put(OFFB_PC, e));
}

static IRExpr *mkWidenFrom32(IRType ty, IRExpr * src, Bool sined)
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   if (ty == Ity_I32)
      return src;
   return (sined) ? unop(Iop_32Sto64, src) : unop(Iop_32Uto64, src);
}

/* Narrow 8/16/32 bit int expr to 8/16/32.  Clearly only some
   of these combinations make sense. */
static IRExpr *narrowTo(IRType dst_ty, IRExpr * e)
{
   IRType src_ty = typeOfIRExpr(irsb->tyenv, e);
   if (src_ty == dst_ty)
      return e;
   if (src_ty == Ity_I32 && dst_ty == Ity_I16)
      return unop(Iop_32to16, e);
   if (src_ty == Ity_I32 && dst_ty == Ity_I8)
      return unop(Iop_32to8, e);
   if (src_ty == Ity_I64 && dst_ty == Ity_I8) {
      vassert(mode64);
      return unop(Iop_64to8, e);
   }
   if (src_ty == Ity_I64 && dst_ty == Ity_I16) {
      vassert(mode64);
      return unop(Iop_64to16, e);
   }
   vpanic("narrowTo(mips)");
   return 0;
}

static IRExpr *getLoFromF64(IRType ty, IRExpr * src)
{
   vassert(ty == Ity_F32 || ty == Ity_F64);
   if (ty == Ity_F64) {
      IRTemp t0, t1;
      t0 = newTemp(Ity_I64);
      t1 = newTemp(Ity_I32);
      assign(t0, unop(Iop_ReinterpF64asI64, src));
      assign(t1, unop(Iop_64to32, mkexpr(t0)));
      return unop(Iop_ReinterpI32asF32, mkexpr(t1));
   } else
      return src;
}

static inline IRExpr *getHiFromF64(IRExpr * src)
{
   vassert(typeOfIRExpr(irsb->tyenv, src) == Ity_F64);
   return unop(Iop_ReinterpI32asF32, unop(Iop_64HIto32,
                                          unop(Iop_ReinterpF64asI64, src)));
}

static IRExpr *mkWidenFromF32(IRType ty, IRExpr * src)
{
   vassert(ty == Ity_F32 || ty == Ity_F64);
   if (ty == Ity_F64) {
      IRTemp t0 = newTemp(Ity_I32);
      IRTemp t1 = newTemp(Ity_I64);
      assign(t0, unop(Iop_ReinterpF32asI32, src));
      assign(t1, binop(Iop_32HLto64, mkU32(0x0), mkexpr(t0)));
      return unop(Iop_ReinterpI64asF64, mkexpr(t1));
   } else
      return src;
}

/* Convenience function to move to next instruction on condition. */
static void mips_next_insn_if(IRExpr *condition) {
   vassert(typeOfIRExpr(irsb->tyenv, condition) == Ity_I1);

   stmt(IRStmt_Exit(condition, Ijk_Boring,
        mode64 ? IRConst_U64(guest_PC_curr_instr + 4) :
                 IRConst_U32(guest_PC_curr_instr + 4),
        OFFB_PC));
}

static IRExpr *dis_branch_likely(IRExpr * guard, UInt imm)
{
   ULong branch_offset;
   IRTemp t0;

   /* PC = PC + (SignExtend(signed_immed_24) << 2)
      An 18-bit signed offset (the 16-bit offset field shifted left 2 bits)
      is added to the address of the instruction following
      the branch (not the branch itself), in the branch delay slot, to form
      a PC-relative effective target address. */
   if (mode64)
      branch_offset = extend_s_18to64(imm << 2);
   else
      branch_offset = extend_s_18to32(imm << 2);

   t0 = newTemp(Ity_I1);
   assign(t0, guard);

   if (mode64)
      stmt(IRStmt_Exit(mkexpr(t0), Ijk_Boring,
                       IRConst_U64(guest_PC_curr_instr + 8), OFFB_PC));
   else
      stmt(IRStmt_Exit(mkexpr(t0), Ijk_Boring,
                       IRConst_U32(guest_PC_curr_instr + 8), OFFB_PC));

   irsb->jumpkind = Ijk_Boring;

   if (mode64)
      return mkU64(guest_PC_curr_instr + 4 + branch_offset);
   else
      return mkU32(guest_PC_curr_instr + 4 + branch_offset);
}

static void dis_branch(Bool link, IRExpr * guard, UInt imm, IRStmt ** set)
{
   ULong branch_offset;
   IRTemp t0;

   if (link) {  /* LR (GPR31) = addr of the 2nd instr after branch instr */
      if (mode64)
         putIReg(31, mkU64(guest_PC_curr_instr + 8));
      else
         putIReg(31, mkU32(guest_PC_curr_instr + 8));
   }

   /* PC = PC + (SignExtend(signed_immed_24) << 2)
      An 18-bit signed offset (the 16-bit offset field shifted left 2 bits)
      is added to the address of the instruction following
      the branch (not the branch itself), in the branch delay slot, to form
      a PC-relative effective target address. */

   if (mode64)
      branch_offset = extend_s_18to64(imm << 2);
   else
      branch_offset = extend_s_18to32(imm << 2);

   t0 = newTemp(Ity_I1);
   assign(t0, guard);
   if (mode64)
      *set = IRStmt_Exit(mkexpr(t0), link ? Ijk_Call : Ijk_Boring,
                         IRConst_U64(guest_PC_curr_instr + 4 + branch_offset),
                         OFFB_PC);
   else
      *set = IRStmt_Exit(mkexpr(t0), link ? Ijk_Call : Ijk_Boring,
                         IRConst_U32(guest_PC_curr_instr + 4 +
                                     (UInt) branch_offset), OFFB_PC);
}

static void dis_branch_compact(Bool link, IRExpr * guard, UInt imm,
                               DisResult *dres)
{
   ULong branch_offset;
   IRTemp t0;

   if (link) {  /* LR (GPR31) = addr of the instr after branch instr */
      if (mode64)
         putIReg(31, mkU64(guest_PC_curr_instr + 4));
      else
         putIReg(31, mkU32(guest_PC_curr_instr + 4));
      dres->jk_StopHere = Ijk_Call;
   } else {
      dres->jk_StopHere = Ijk_Boring;
   }

   dres->whatNext = Dis_StopHere;

   /* PC = PC + (SignExtend(signed_immed_24) << 2)
      An 18-bit signed offset (the 16-bit offset field shifted left 2 bits)
      is added to the address of the instruction following
      the branch (not the branch itself), in the branch delay slot, to form
      a PC-relative effective target address. */

   if (mode64)
      branch_offset = extend_s_18to64(imm << 2);
   else
      branch_offset = extend_s_18to32(imm << 2);

   t0 = newTemp(Ity_I1);
   assign(t0, guard);

   if (mode64) {
      stmt(IRStmt_Exit(mkexpr(t0), link ? Ijk_Call : Ijk_Boring,
                         IRConst_U64(guest_PC_curr_instr + 4 + branch_offset),
                         OFFB_PC));
      putPC(mkU64(guest_PC_curr_instr + 4));
   } else {
      stmt(IRStmt_Exit(mkexpr(t0), link ? Ijk_Call : Ijk_Boring,
                         IRConst_U32(guest_PC_curr_instr + 4 +
                                     (UInt) branch_offset), OFFB_PC));
      putPC(mkU32(guest_PC_curr_instr + 4));
   }
}

static IRExpr *getFReg(UInt fregNo)
{
   vassert(fregNo < 32);
   IRType ty = fp_mode64 ? Ity_F64 : Ity_F32;
   return IRExpr_Get(floatGuestRegOffset(fregNo), ty);
}

static IRExpr *getDReg(UInt dregNo)
{
   vassert(dregNo < 32);
   if (fp_mode64) {
      return IRExpr_Get(floatGuestRegOffset(dregNo), Ity_F64);
   } else {
      /* Read a floating point register pair and combine their contents into a
         64-bit value */
      IRTemp t0 = newTemp(Ity_F32);
      IRTemp t1 = newTemp(Ity_F32);
      IRTemp t2 = newTemp(Ity_F64);
      IRTemp t3 = newTemp(Ity_I32);
      IRTemp t4 = newTemp(Ity_I32);
      IRTemp t5 = newTemp(Ity_I64);

      assign(t0, getFReg(dregNo & (~1)));
      assign(t1, getFReg(dregNo | 1));

      assign(t3, unop(Iop_ReinterpF32asI32, mkexpr(t0)));
      assign(t4, unop(Iop_ReinterpF32asI32, mkexpr(t1)));
      assign(t5, binop(Iop_32HLto64, mkexpr(t4), mkexpr(t3)));
      assign(t2, unop(Iop_ReinterpI64asF64, mkexpr(t5)));

      return mkexpr(t2);
   }
}

static void putFReg(UInt dregNo, IRExpr * e)
{
   vassert(dregNo < 32);
   IRType ty = fp_mode64 ? Ity_F64 : Ity_F32;
   vassert(typeOfIRExpr(irsb->tyenv, e) == ty);

   if (fp_mode64_fre) {
      IRTemp t0 = newTemp(Ity_F32);
      assign(t0, getLoFromF64(ty, e));
#if defined (_MIPSEL)
      stmt(IRStmt_Put(floatGuestRegOffset(dregNo), mkexpr(t0)));
      if (dregNo & 1)
         stmt(IRStmt_Put(floatGuestRegOffset(dregNo) - 4, mkexpr(t0)));
#else
      stmt(IRStmt_Put(floatGuestRegOffset(dregNo) + 4, mkexpr(t0)));
      if (dregNo & 1)
         stmt(IRStmt_Put(floatGuestRegOffset(dregNo & (~1)), mkexpr(t0)));
#endif
   } else {
      stmt(IRStmt_Put(floatGuestRegOffset(dregNo), e));
   }

   if (has_msa && fp_mode64) {
      stmt(IRStmt_Put(msaGuestRegOffset(dregNo),
           binop(Iop_64HLtoV128,
                    unop(Iop_ReinterpF64asI64, e),
                    unop(Iop_ReinterpF64asI64, e))));
   }
}

static void putDReg(UInt dregNo, IRExpr * e)
{
   if (fp_mode64) {
      vassert(dregNo < 32);
      IRType ty = Ity_F64;
      vassert(typeOfIRExpr(irsb->tyenv, e) == ty);
      stmt(IRStmt_Put(floatGuestRegOffset(dregNo), e));
      if (fp_mode64_fre) {
         IRTemp t0 = newTemp(Ity_F32);
         if (dregNo & 1) {
            assign(t0, getLoFromF64(ty, e));
#if defined (_MIPSEL)
            stmt(IRStmt_Put(floatGuestRegOffset(dregNo) - 4, mkexpr(t0)));
#else
            stmt(IRStmt_Put(floatGuestRegOffset(dregNo & (~1)), mkexpr(t0)));
#endif
         } else {
            assign(t0, getHiFromF64(e));
#if defined (_MIPSEL)
            stmt(IRStmt_Put(floatGuestRegOffset(dregNo | 1), mkexpr(t0)));
#else
            stmt(IRStmt_Put(floatGuestRegOffset(dregNo | 1) + 4, mkexpr(t0)));
#endif
         }
      }
      if (has_msa)
         stmt(IRStmt_Put(msaGuestRegOffset(dregNo),
              binop(Iop_64HLtoV128,
                    unop(Iop_ReinterpF64asI64, e),
                    unop(Iop_ReinterpF64asI64, e))));
   } else {
      vassert(dregNo < 32);
      vassert(typeOfIRExpr(irsb->tyenv, e) == Ity_F64);
      IRTemp t1 = newTemp(Ity_F64);
      IRTemp t4 = newTemp(Ity_I32);
      IRTemp t5 = newTemp(Ity_I32);
      IRTemp t6 = newTemp(Ity_I64);
      assign(t1, e);
      assign(t6, unop(Iop_ReinterpF64asI64, mkexpr(t1)));
      assign(t4, unop(Iop_64HIto32, mkexpr(t6)));  /* hi */
      assign(t5, unop(Iop_64to32, mkexpr(t6)));    /* lo */
      putFReg(dregNo & (~1), unop(Iop_ReinterpI32asF32, mkexpr(t5)));
      putFReg(dregNo | 1, unop(Iop_ReinterpI32asF32, mkexpr(t4)));
   }
}

static void setFPUCondCode(IRExpr * e, UInt cc)
{
   if (cc == 0) {
      putFCSR(binop(Iop_And32, getFCSR(), mkU32(0xFF7FFFFF)));
      putFCSR(binop(Iop_Or32, getFCSR(), binop(Iop_Shl32, e, mkU8(23))));
   } else {
      putFCSR(binop(Iop_And32, getFCSR(), unop(Iop_Not32,
                               binop(Iop_Shl32, mkU32(0x01000000), mkU8(cc)))));
      putFCSR(binop(Iop_Or32, getFCSR(), binop(Iop_Shl32, e, mkU8(24 + cc))));
   }
}

static IRExpr* get_IR_roundingmode ( void )
{
/*
   rounding mode | MIPS | IR
   ------------------------
   to nearest    | 00  | 00
   to zero       | 01  | 11
   to +infinity  | 10  | 10
   to -infinity  | 11  | 01
*/
   IRTemp rm_MIPS = newTemp(Ity_I32);
   /* Last two bits in FCSR are rounding mode. */

   if (mode64)
      assign(rm_MIPS, binop(Iop_And32, IRExpr_Get(offsetof(VexGuestMIPS64State,
                                       guest_FCSR), Ity_I32), mkU32(3)));
   else
      assign(rm_MIPS, binop(Iop_And32, IRExpr_Get(offsetof(VexGuestMIPS32State,
                                       guest_FCSR), Ity_I32), mkU32(3)));

   /* rm_IR = XOR( rm_MIPS32, (rm_MIPS32 << 1) & 2) */

   return binop(Iop_Xor32, mkexpr(rm_MIPS), binop(Iop_And32,
                binop(Iop_Shl32, mkexpr(rm_MIPS), mkU8(1)), mkU32(2)));
}

static IRExpr* get_IR_roundingmode_MSA ( void ) {
   /*
      rounding mode | MIPS | IR
      ------------------------
      to nearest    | 00  | 00
      to zero       | 01  | 11
      to +infinity  | 10  | 10
      to -infinity  | 11  | 01
   */
   IRTemp rm_MIPS = newTemp(Ity_I32);
   /* Last two bits in MSACSR are rounding mode. */

   if (mode64)
      assign(rm_MIPS, binop(Iop_And32, IRExpr_Get(offsetof(VexGuestMIPS64State,
                            guest_MSACSR), Ity_I32), mkU32(3)));
   else
      assign(rm_MIPS, binop(Iop_And32, IRExpr_Get(offsetof(VexGuestMIPS32State,
                            guest_MSACSR), Ity_I32), mkU32(3)));

   /* rm_IR = XOR( rm_MIPS32, (rm_MIPS32 << 1) & 2) */
   return binop(Iop_Xor32, mkexpr(rm_MIPS), binop(Iop_And32,
                binop(Iop_Shl32, mkexpr(rm_MIPS), mkU8(1)), mkU32(2)));
}

/* sz, ULong -> IRExpr */
static IRExpr *mkSzImm ( IRType ty, ULong imm64 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return ty == Ity_I64 ? mkU64(imm64) : mkU32((UInt) imm64);
}

static IRConst *mkSzConst ( IRType ty, ULong imm64 )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return (ty == Ity_I64 ? IRConst_U64(imm64) : IRConst_U32((UInt) imm64));
}

/* Make sure we get valid 32 and 64bit addresses */
static Addr64 mkSzAddr ( IRType ty, Addr64 addr )
{
   vassert(ty == Ity_I32 || ty == Ity_I64);
   return (ty == Ity_I64 ? (Addr64) addr :
                           (Addr64) extend_s_32to64(toUInt(addr)));
}

/* Shift and Rotate instructions for MIPS64 */
static Bool dis_instr_shrt ( UInt theInstr )
{
   UInt opc2 = get_function(theInstr);
   UChar regRs = get_rs(theInstr);
   UChar regRt = get_rt(theInstr);
   UChar regRd = get_rd(theInstr);
   UChar uImmsa = get_sa(theInstr);
   Long sImmsa = extend_s_16to64(uImmsa);
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IRTemp tmp = newTemp(ty);
   IRTemp tmpOr = newTemp(ty);
   IRTemp tmpRt = newTemp(ty);
   IRTemp tmpRs = newTemp(ty);
   IRTemp tmpRd = newTemp(ty);

   assign(tmpRs, getIReg(regRs));
   assign(tmpRt, getIReg(regRt));

   switch (opc2) {
      case 0x3A:
         if ((regRs & 0x01) == 0) {
            /* Doubleword Shift Right Logical - DSRL; MIPS64 */
            DIP("dsrl r%u, r%u, %lld", regRd, regRt, sImmsa);
            assign(tmpRd, binop(Iop_Shr64, mkexpr(tmpRt), mkU8(uImmsa)));
            putIReg(regRd, mkexpr(tmpRd));
         } else if ((regRs & 0x01) == 1) {
            /* Doubleword Rotate Right - DROTR; MIPS64r2 */
            vassert(mode64);
            DIP("drotr r%u, r%u, %lld", regRd, regRt, sImmsa);
            IRTemp tmpL = newTemp(ty);
            IRTemp tmpR = newTemp(ty);
            assign(tmpR, binop(Iop_Shr64, mkexpr(tmpRt), mkU8(uImmsa)));
            assign(tmp, binop(Iop_Shl64, mkexpr(tmpRt), mkU8(63 - uImmsa)));
            assign(tmpL, binop(Iop_Shl64, mkexpr(tmp), mkU8(1)));
            assign(tmpRd, binop(Iop_Or64, mkexpr(tmpL), mkexpr(tmpR)));
            putIReg(regRd, mkexpr(tmpRd));
         } else
            return False;
         break;

      case 0x3E:
         if ((regRs & 0x01) == 0) {
            /* Doubleword Shift Right Logical Plus 32 - DSRL32; MIPS64 */
            DIP("dsrl32 r%u, r%u, %lld", regRd, regRt, sImmsa + 32);
            assign(tmpRd, binop(Iop_Shr64, mkexpr(tmpRt), mkU8(uImmsa + 32)));
            putIReg(regRd, mkexpr(tmpRd));
         } else if ((regRs & 0x01) == 1) {
            /* Doubleword Rotate Right Plus 32 - DROTR32; MIPS64r2 */
            DIP("drotr32 r%u, r%u, %lld", regRd, regRt, sImmsa);
            vassert(mode64);
            IRTemp tmpL = newTemp(ty);
            IRTemp tmpR = newTemp(ty);
            /* (tmpRt >> sa) | (tmpRt << (64 - sa)) */
            assign(tmpR, binop(Iop_Shr64, mkexpr(tmpRt), mkU8(uImmsa + 32)));
            assign(tmp, binop(Iop_Shl64, mkexpr(tmpRt),
                              mkU8(63 - (uImmsa + 32))));
            assign(tmpL, binop(Iop_Shl64, mkexpr(tmp), mkU8(1)));
            assign(tmpRd, binop(Iop_Or64, mkexpr(tmpL), mkexpr(tmpR)));
            putIReg(regRd, mkexpr(tmpRd));
         } else
            return False;
         break;

      case 0x16:
         if ((uImmsa & 0x01) == 0) {
            /* Doubleword Shift Right Logical Variable - DSRLV; MIPS64 */
            DIP("dsrlv r%u, r%u, r%u", regRd, regRt, regRs);
            IRTemp tmpRs8 = newTemp(Ity_I8);
            /* s = tmpRs[5..0] */
            assign(tmp, binop(Iop_And64, mkexpr(tmpRs), mkU64(63)));
            assign(tmpRs8, mkNarrowTo8(ty, mkexpr(tmp)));
            assign(tmpRd, binop(Iop_Shr64, mkexpr(tmpRt), mkexpr(tmpRs8)));
            putIReg(regRd, mkexpr(tmpRd));
         } else if ((uImmsa & 0x01) == 1) {
            /* Doubleword Rotate Right Variable - DROTRV; MIPS64r2 */
            DIP("drotrv r%u, r%u, r%u", regRd, regRt, regRs);
            IRTemp tmpL = newTemp(ty);
            IRTemp tmpR = newTemp(ty);
            IRTemp tmpRs8 = newTemp(Ity_I8);
            IRTemp tmpLs8 = newTemp(Ity_I8);
            IRTemp tmp64 = newTemp(ty);
            /* s = tmpRs[5...0]
               m = 64 - s
               (tmpRt << s) | (tmpRt >> m) */

            assign(tmp64, binop(Iop_And64, mkexpr(tmpRs), mkSzImm(ty, 63)));
            assign(tmp, binop(Iop_Sub64, mkU64(63), mkexpr(tmp64)));

            assign(tmpLs8, mkNarrowTo8(ty, mkexpr(tmp)));
            assign(tmpRs8, mkNarrowTo8(ty, mkexpr(tmp64)));

            assign(tmpR, binop(Iop_Shr64, mkexpr(tmpRt), mkexpr(tmpRs8)));
            assign(tmpL, binop(Iop_Shl64, mkexpr(tmpRt), mkexpr(tmpLs8)));
            assign(tmpRd, binop(Iop_Shl64, mkexpr(tmpL), mkU8(1)));
            assign(tmpOr, binop(Iop_Or64, mkexpr(tmpRd), mkexpr(tmpR)));

            putIReg(regRd, mkexpr(tmpOr));
         } else
            return False;
         break;

      case 0x38:  /* Doubleword Shift Left Logical - DSLL; MIPS64 */
         DIP("dsll r%u, r%u, %lld", regRd, regRt, sImmsa);
         vassert(mode64);
         assign(tmpRd, binop(Iop_Shl64, mkexpr(tmpRt), mkU8(uImmsa)));
         putIReg(regRd, mkexpr(tmpRd));
         break;

      case 0x3C:  /* Doubleword Shift Left Logical Plus 32 - DSLL32; MIPS64 */
         DIP("dsll32 r%u, r%u, %lld", regRd, regRt, sImmsa);
         assign(tmpRd, binop(Iop_Shl64, mkexpr(tmpRt), mkU8(uImmsa + 32)));
         putIReg(regRd, mkexpr(tmpRd));
         break;

      case 0x14: {  /* Doubleword Shift Left Logical Variable - DSLLV; MIPS64 */
         DIP("dsllv r%u, r%u, r%u", regRd, regRt, regRs);
         IRTemp tmpRs8 = newTemp(Ity_I8);

         assign(tmp, binop(Iop_And64, mkexpr(tmpRs), mkSzImm(ty, 63)));
         assign(tmpRs8, mkNarrowTo8(ty, mkexpr(tmp)));
         assign(tmpRd, binop(Iop_Shl64, mkexpr(tmpRt), mkexpr(tmpRs8)));
         putIReg(regRd, mkexpr(tmpRd));
         break;
      }

      case 0x3B:  /* Doubleword Shift Right Arithmetic - DSRA; MIPS64 */
         DIP("dsra r%u, r%u, %lld", regRd, regRt, sImmsa);
         assign(tmpRd, binop(Iop_Sar64, mkexpr(tmpRt), mkU8(uImmsa)));
         putIReg(regRd, mkexpr(tmpRd));
         break;

      case 0x3F:  /* Doubleword Shift Right Arithmetic Plus 32 - DSRA32;
                     MIPS64 */
         DIP("dsra32 r%u, r%u, %lld", regRd, regRt, sImmsa);
         assign(tmpRd, binop(Iop_Sar64, mkexpr(tmpRt), mkU8(uImmsa + 32)));
         putIReg(regRd, mkexpr(tmpRd));
         break;

      case 0x17: {  /* Doubleword Shift Right Arithmetic Variable - DSRAV;
                       MIPS64 */
         DIP("dsrav r%u, r%u, r%u", regRd, regRt, regRs);
         IRTemp tmpRs8 = newTemp(Ity_I8);
         assign(tmp, binop(Iop_And64, mkexpr(tmpRs), mkSzImm(ty, 63)));
         assign(tmpRs8, mkNarrowTo8(ty, mkexpr(tmp)));
         assign(tmpRd, binop(Iop_Sar64, mkexpr(tmpRt), mkexpr(tmpRs8)));
         putIReg(regRd, mkexpr(tmpRd));
         break;

      }

      default:
         return False;

   }
   return True;
}

static IROp mkSzOp ( IRType ty, IROp op8 )
{
   Int adj;
   vassert(ty == Ity_I8 || ty == Ity_I16 || ty == Ity_I32 || ty == Ity_I64);
   vassert(op8 == Iop_Add8 || op8 == Iop_Sub8 || op8 == Iop_Mul8
           || op8 == Iop_Or8 || op8 == Iop_And8 || op8 == Iop_Xor8
           || op8 == Iop_Shl8 || op8 == Iop_Shr8 || op8 == Iop_Sar8
           || op8 == Iop_CmpEQ8 || op8 == Iop_CmpNE8 || op8 == Iop_Not8);
   adj = ty == Ity_I8 ? 0 : (ty == Ity_I16 ? 1 : (ty == Ity_I32 ? 2 : 3));
   return adj + op8;
}

/*********************************************************/
/*---             Floating Point Compare              ---*/
/*********************************************************/
/* Function that returns a string that represent mips cond
   mnemonic for the input code. */
static const HChar* showCondCode(UInt code) {
   const HChar* ret;
   switch (code) {
      case 0: ret = "f"; break;
      case 1: ret = "un"; break;
      case 2: ret = "eq"; break;
      case 3: ret = "ueq"; break;
      case 4: ret = "olt"; break;
      case 5: ret = "ult"; break;
      case 6: ret = "ole"; break;
      case 7: ret = "ule"; break;
      case 8: ret = "sf"; break;
      case 9: ret = "ngle"; break;
      case 10: ret = "seq"; break;
      case 11: ret = "ngl"; break;
      case 12: ret = "lt"; break;
      case 13: ret = "nge"; break;
      case 14: ret = "le"; break;
      case 15: ret = "ngt"; break;
      default: vpanic("showCondCode"); break;
   }
   return ret;
}

static Bool dis_instr_CCondFmt ( UInt cins )
{
   IRTemp t0, t1, t2, t3, tmp5, tmp6;
   IRTemp ccIR = newTemp(Ity_I32);
   IRTemp ccMIPS = newTemp(Ity_I32);
   UInt FC = get_FC(cins);
   UInt fmt = get_fmt(cins);
   UInt fs = get_fs(cins);
   UInt ft = get_ft(cins);
   UInt cond = get_cond(cins);

   if (FC == 0x3) {  /* C.cond.fmt */
      UInt fpc_cc = get_fpc_cc(cins);
      switch (fmt) {
         case 0x10: {  /* C.cond.S */
            DIP("c.%s.s %u, f%u, f%u", showCondCode(cond), fpc_cc, fs, ft);
            if (fp_mode64) {
               t0 = newTemp(Ity_I32);
               t1 = newTemp(Ity_I32);
               t2 = newTemp(Ity_I32);
               t3 = newTemp(Ity_I32);

               tmp5 = newTemp(Ity_F64);
               tmp6 = newTemp(Ity_F64);

               assign(tmp5, unop(Iop_F32toF64, getLoFromF64(Ity_F64,
                                 getFReg(fs))));
               assign(tmp6, unop(Iop_F32toF64, getLoFromF64(Ity_F64,
                                 getFReg(ft))));

               assign(ccIR, binop(Iop_CmpF64, mkexpr(tmp5), mkexpr(tmp6)));
               putHI(mkWidenFrom32(mode64 ? Ity_I64: Ity_I32,
                                   mkexpr(ccIR), True));
               /* Map compare result from IR to MIPS
                  FP cmp result | MIPS | IR
                  --------------------------
                  UN            | 0x1 | 0x45
                  EQ            | 0x2 | 0x40
                  GT            | 0x4 | 0x00
                  LT            | 0x8 | 0x01
                */

               /* ccMIPS = Shl(1, (~(ccIR>>5) & 2) | ((ccIR ^ (ccIR>>6)) & 1) */
               assign(ccMIPS, binop(Iop_Shl32, mkU32(1), unop(Iop_32to8,
                              binop(Iop_Or32, binop(Iop_And32, unop(Iop_Not32,
                              binop(Iop_Shr32, mkexpr(ccIR),mkU8(5))),mkU32(2)),
                              binop(Iop_And32, binop(Iop_Xor32, mkexpr(ccIR),
                              binop(Iop_Shr32, mkexpr(ccIR), mkU8(6))),
                              mkU32(1))))));
               putLO(mkWidenFrom32(mode64 ? Ity_I64: Ity_I32,
                                   mkexpr(ccMIPS), True));

               /* UN */
               assign(t0, binop(Iop_And32, mkexpr(ccMIPS), mkU32(0x1)));
               /* EQ */
               assign(t1, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                                                 mkU8(0x1)), mkU32(0x1)));
               /* NGT */
               assign(t2, binop(Iop_And32, unop(Iop_Not32, binop(Iop_Shr32,
                                 mkexpr(ccMIPS), mkU8(0x2))),mkU32(0x1)));
               /* LT */
               assign(t3, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                                                 mkU8(0x3)), mkU32(0x1)));
               switch (cond) {
                  case 0x0:
                     setFPUCondCode(mkU32(0), fpc_cc);
                     break;
                  case 0x1:
                     setFPUCondCode(mkexpr(t0), fpc_cc);
                     break;
                  case 0x2:
                     setFPUCondCode(mkexpr(t1), fpc_cc);
                     break;
                  case 0x3:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0x4:
                     setFPUCondCode(mkexpr(t3), fpc_cc);
                     break;
                  case 0x5:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                          fpc_cc);
                     break;
                  case 0x6:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0x7:
                     setFPUCondCode(mkexpr(t2), fpc_cc);
                     break;
                  case 0x8:
                     setFPUCondCode(mkU32(0), fpc_cc);
                     break;
                  case 0x9:
                     setFPUCondCode(mkexpr(t0), fpc_cc);
                     break;
                  case 0xA:
                     setFPUCondCode(mkexpr(t1), fpc_cc);
                     break;
                  case 0xB:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0xC:
                     setFPUCondCode(mkexpr(t3), fpc_cc);
                     break;
                  case 0xD:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                          fpc_cc);
                     break;
                  case 0xE:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0xF:
                     setFPUCondCode(mkexpr(t2), fpc_cc);
                     break;

                  default:
                     return False;
               }

            } else {
               t0 = newTemp(Ity_I32);
               t1 = newTemp(Ity_I32);
               t2 = newTemp(Ity_I32);
               t3 = newTemp(Ity_I32);

               assign(ccIR, binop(Iop_CmpF64, unop(Iop_F32toF64, getFReg(fs)),
                                  unop(Iop_F32toF64, getFReg(ft))));
               /* Map compare result from IR to MIPS
                  FP cmp result | MIPS | IR
                  --------------------------
                  UN            | 0x1 | 0x45
                  EQ            | 0x2 | 0x40
                  GT            | 0x4 | 0x00
                  LT            | 0x8 | 0x01
                */

               /* ccMIPS = Shl(1, (~(ccIR>>5) & 2) | ((ccIR ^ (ccIR>>6)) & 1) */
               assign(ccMIPS, binop(Iop_Shl32, mkU32(1), unop(Iop_32to8,
                              binop(Iop_Or32, binop(Iop_And32, unop(Iop_Not32,
                              binop(Iop_Shr32, mkexpr(ccIR), mkU8(5))),
                                    mkU32(2)), binop(Iop_And32,
                              binop(Iop_Xor32, mkexpr(ccIR),
                              binop(Iop_Shr32, mkexpr(ccIR), mkU8(6))),
                              mkU32(1))))));
               /* UN */
               assign(t0, binop(Iop_And32, mkexpr(ccMIPS), mkU32(0x1)));
               /* EQ */
               assign(t1, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                      mkU8(0x1)), mkU32(0x1)));
               /* NGT */
               assign(t2, binop(Iop_And32, unop(Iop_Not32, binop(Iop_Shr32,
                      mkexpr(ccMIPS), mkU8(0x2))), mkU32(0x1)));
               /* LT */
               assign(t3, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                      mkU8(0x3)), mkU32(0x1)));

               switch (cond) {
                  case 0x0:
                     setFPUCondCode(mkU32(0), fpc_cc);
                     break;
                  case 0x1:
                     setFPUCondCode(mkexpr(t0), fpc_cc);
                     break;
                  case 0x2:
                     setFPUCondCode(mkexpr(t1), fpc_cc);
                     break;
                  case 0x3:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0x4:
                     setFPUCondCode(mkexpr(t3), fpc_cc);
                     break;
                  case 0x5:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                          fpc_cc);
                     break;
                  case 0x6:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0x7:
                     setFPUCondCode(mkexpr(t2), fpc_cc);
                     break;
                  case 0x8:
                     setFPUCondCode(mkU32(0), fpc_cc);
                     break;
                  case 0x9:
                     setFPUCondCode(mkexpr(t0), fpc_cc);
                     break;
                  case 0xA:
                     setFPUCondCode(mkexpr(t1), fpc_cc);
                     break;
                  case 0xB:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0xC:
                     setFPUCondCode(mkexpr(t3), fpc_cc);
                     break;
                  case 0xD:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                          fpc_cc);
                     break;
                  case 0xE:
                     setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                          fpc_cc);
                     break;
                  case 0xF:
                     setFPUCondCode(mkexpr(t2), fpc_cc);
                     break;

                  default:
                     return False;
               }
            }
         }
            break;

         case 0x11: {  /* C.cond.D */
            DIP("c.%s.d %u, f%u, f%u", showCondCode(cond), fpc_cc, fs, ft);
            t0 = newTemp(Ity_I32);
            t1 = newTemp(Ity_I32);
            t2 = newTemp(Ity_I32);
            t3 = newTemp(Ity_I32);
            assign(ccIR, binop(Iop_CmpF64, getDReg(fs), getDReg(ft)));
            /* Map compare result from IR to MIPS
               FP cmp result | MIPS | IR
               --------------------------
               UN            | 0x1 | 0x45
               EQ            | 0x2 | 0x40
               GT            | 0x4 | 0x00
               LT            | 0x8 | 0x01
             */

            /* ccMIPS = Shl(1, (~(ccIR>>5) & 2) | ((ccIR ^ (ccIR>>6)) & 1) */
            assign(ccMIPS, binop(Iop_Shl32, mkU32(1), unop(Iop_32to8,
                           binop(Iop_Or32, binop(Iop_And32, unop(Iop_Not32,
                           binop(Iop_Shr32, mkexpr(ccIR), mkU8(5))), mkU32(2)),
                           binop(Iop_And32, binop(Iop_Xor32, mkexpr(ccIR),
                           binop(Iop_Shr32, mkexpr(ccIR), mkU8(6))),
                           mkU32(1))))));

            /* UN */
            assign(t0, binop(Iop_And32, mkexpr(ccMIPS), mkU32(0x1)));
            /* EQ */
            assign(t1, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                   mkU8(0x1)), mkU32(0x1)));
            /* NGT */
            assign(t2, binop(Iop_And32, unop(Iop_Not32, binop(Iop_Shr32,
                   mkexpr(ccMIPS), mkU8(0x2))), mkU32(0x1)));
            /* LT */
            assign(t3, binop(Iop_And32, binop(Iop_Shr32, mkexpr(ccMIPS),
                   mkU8(0x3)), mkU32(0x1)));

            switch (cond) {
               case 0x0:
                  setFPUCondCode(mkU32(0), fpc_cc);
                  break;
               case 0x1:
                  setFPUCondCode(mkexpr(t0), fpc_cc);
                  break;
               case 0x2:
                  setFPUCondCode(mkexpr(t1), fpc_cc);
                  break;
               case 0x3:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                       fpc_cc);
                  break;
               case 0x4:
                  setFPUCondCode(mkexpr(t3), fpc_cc);
                  break;
               case 0x5:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                       fpc_cc);
                  break;
               case 0x6:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                       fpc_cc);
                  break;
               case 0x7:
                  setFPUCondCode(mkexpr(t2), fpc_cc);
                  break;
               case 0x8:
                  setFPUCondCode(mkU32(0), fpc_cc);
                  break;
               case 0x9:
                  setFPUCondCode(mkexpr(t0), fpc_cc);
                  break;
               case 0xA:
                  setFPUCondCode(mkexpr(t1), fpc_cc);
                  break;
               case 0xB:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t1)),
                                       fpc_cc);
                  break;
               case 0xC:
                  setFPUCondCode(mkexpr(t3), fpc_cc);
                  break;
               case 0xD:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t0), mkexpr(t3)),
                                       fpc_cc);
                  break;
               case 0xE:
                  setFPUCondCode(binop(Iop_Or32, mkexpr(t3), mkexpr(t1)),
                                       fpc_cc);
                  break;
               case 0xF:
                  setFPUCondCode(mkexpr(t2), fpc_cc);
                  break;
               default:
                  return False;
            }
         }
         break;

         default:
            return False;
      }
   } else {
      return False;
   }

   return True;
}

/*********************************************************/
/*---        Branch Instructions for mips64           ---*/
/*********************************************************/
static Bool dis_instr_branch ( UInt theInstr, DisResult * dres,
                               Bool(*resteerOkFn) (void *, Addr),
                               void *callback_opaque, IRStmt ** set )
{
   UInt jmpKind = 0;
   UChar opc1 = get_opcode(theInstr);
   UChar regRs = get_rs(theInstr);
   UChar regRt = get_rt(theInstr);
   UInt offset = get_imm(theInstr);
   Long sOffset = extend_s_16to64(offset);
   IRType ty = mode64 ? Ity_I64 : Ity_I32;
   IROp opSlt = mode64 ? Iop_CmpLT64S : Iop_CmpLT32S;

   IRTemp tmp = newTemp(ty);
   IRTemp tmpRs = newTemp(ty);
   IRTemp tmpRt = newTemp(ty);
   IRTemp tmpLt = newTemp(ty);
   IRTemp tmpReg0 = newTemp(ty);

   UChar regLnk = 31;   /* reg 31 is link reg in MIPS */
   Addr64 addrTgt = 0;
   Addr64 cia = guest_PC_curr_instr;

   IRExpr *eConst0 = mkSzImm(ty, (UInt) 0);
   IRExpr *eNia = mkSzImm(ty, cia + 8);
   IRExpr *eCond = NULL;

   assign(tmpRs, getIReg(regRs));
   assign(tmpRt, getIReg(regRt));
   assign(tmpReg0, getIReg(0));

   eCond = binop(mkSzOp(ty, Iop_CmpNE8), mkexpr(tmpReg0), mkexpr(tmpReg0));

   switch (opc1) {
      case 0x01:
         switch (regRt) {
            case 0x00: {  /* BLTZ rs, offset */
               addrTgt = mkSzAddr(ty, cia + 4 + (sOffset << 2));
               IRTemp tmpLtRes = newTemp(Ity_I1);

               assign(tmp, eConst0);
               assign(tmpLtRes, binop(opSlt, mkexpr(tmpRs), mkexpr(tmp)));
               assign(tmpLt, mode64 ? unop(Iop_1Uto64, mkexpr(tmpLtRes)) :
                      unop(Iop_1Uto32, mkexpr(tmpLtRes)));

               eCond = binop(mkSzOp(ty, Iop_CmpNE8), mkexpr(tmpLt),
                             mkexpr(tmpReg0));

               jmpKind = Ijk_Boring;
               break;
            }

            case 0x01: {  /* BGEZ rs, offset */
               IRTemp tmpLtRes = newTemp(Ity_I1);
               addrTgt = mkSzAddr(ty, cia + 4 + (sOffset << 2));

               assign(tmp, eConst0);
               assign(tmpLtRes, binop(opSlt, mkexpr(tmpRs), mkexpr(tmp)));
               assign(tmpLt, mode64 ? unop(Iop_1Uto64, mkexpr(tmpLtRes)) :
                                      unop(Iop_1Uto32, mkexpr(tmpLtRes)));
               eCond = binop(mkSzOp(ty, Iop_CmpEQ8), mkexpr(tmpLt),
                                    mkexpr(tmpReg0));

               jmpKind = Ijk_Boring;
               break;
            }

            case 0x11: {  /* BGEZAL rs, offset */
               addrTgt = mkSzAddr(ty, cia + 4 + (sOffset << 2));
               putIReg(regLnk, eNia);
               IRTemp tmpLtRes = newTemp(Ity_I1);

               assign(tmpLtRes, binop(opSlt, mkexpr(tmpRs), eConst0));
               assign(tmpLt, mode64 ? unop(Iop_1Uto64, mkexpr(tmpLtRes)) :
                                      unop(Iop_1Uto32, mkexpr(tmpLtRes)));

               eCond = binop(mkSzOp(ty, Iop_CmpEQ8), mkexpr(tmpLt),
                                    mkexpr(tmpReg0));

               jmpKind = Ijk_Call;
               break;
            }

            case 0x10: {  /* BLTZAL rs, offset */
               IRTemp tmpLtRes = newTemp(Ity_I1);
               IRTemp tmpRes = newTemp(ty);

               addrTgt = mkSzAddr(ty, cia + 4 + (sOffset << 2));
               putIReg(regLnk, eNia);

               assign(tmp, eConst0);
               assign(tmpLtRes, binop(opSlt, mkexpr(tmpRs), mkexpr(tmp)));
               assign(tmpRes, mode64 ? unop(Iop_1Uto64,
                      mkexpr(tmpLtRes)) : unop(Iop_1Uto32, mkexpr(tmpLtRes)));
               eCond = binop(mkSzOp(ty, Iop_CmpNE8), mkexpr(tmpRes),
                                                     mkexpr(tmpReg0));

               jmpKind = Ijk_Call;
               break;
            }

         }
         break;
      default:
         return False;
      }
   *set = IRStmt_Exit(eCond, jmpKind, mkSzConst(ty, addrTgt), OFFB_PC);
   return True;
}

/*********************************************************/
/*---         Cavium Specific Instructions            ---*/
/*********************************************************/

/* Convenience function to yield to thread scheduler */
static void jump_back(IRExpr *condition)
{
   stmt( IRStmt_Exit(condition,
                     Ijk_Yield,
                     IRConst_U64( guest_PC_curr_instr ),
                     OFFB_PC) );
}

/* Based on s390_irgen_load_and_add32. */
static void mips_load_store32(IRTemp op1addr, IRTemp new_val,
                              IRTemp expd, UChar rd, Bool putIntoRd)
{
   IRCAS *cas;
   IRTemp old_mem = newTemp(Ity_I32);
   IRType ty      = mode64 ? Ity_I64 : Ity_I32;

   cas = mkIRCAS(IRTemp_INVALID, old_mem,
#if defined (_MIPSEL)
                 Iend_LE, mkexpr(op1addr),
#else /* _MIPSEB */
                 Iend_BE, mkexpr(op1addr),
#endif
                 NULL, mkexpr(expd), /* expected value */
                 NULL, mkexpr(new_val)  /* new value */);
   stmt(IRStmt_CAS(cas));

   /* If old_mem contains the expected value, then the CAS succeeded.
      Otherwise, it did not */
   jump_back(binop(Iop_CmpNE32, mkexpr(old_mem), mkexpr(expd)));
   if (putIntoRd)
      putIReg(rd, mkWidenFrom32(ty, mkexpr(old_mem), True));
}

/* Based on s390_irgen_load_and_add64. */
static void mips_load_store64(IRTemp op1addr, IRTemp new_val,
                              IRTemp expd, UChar rd, Bool putIntoRd)
{
   IRCAS *cas;
   IRTemp old_mem = newTemp(Ity_I64);
   vassert(mode64);
   cas = mkIRCAS(IRTemp_INVALID, old_mem,
#if defined (_MIPSEL)
                 Iend_LE, mkexpr(op1addr),
#else /* _MIPSEB */
                 Iend_BE, mkexpr(op1addr),
#endif
                 NULL, mkexpr(expd), /* expected value */
                 NULL, mkexpr(new_val)  /* new value */);
   stmt(IRStmt_CAS(cas));

   /* If old_mem contains the expected value, then the CAS succeeded.
      Otherwise, it did not */
   jump_back(binop(Iop_CmpNE64, mkexpr(old_mem), mkexpr(expd)));
   if (putIntoRd)
      putIReg(rd, mkexpr(old_mem));
}

static Bool dis_instr_CVM ( UInt theInstr )
{
   UChar  opc2     = get_function(theInstr);
   UChar  opc1     = get_opcode(theInstr);
   UChar  regRs    = get_rs(theInstr);
   UChar  regRt    = get_rt(theInstr);
   UChar  regRd    = get_rd(theInstr);
   /* MIPS trap instructions extract code from theInstr[15:6].
      Cavium OCTEON instructions SNEI, SEQI extract immediate operands
      from the same bit field [15:6]. */
   UInt   imm      = get_code(theInstr);
   UChar  lenM1    = get_msb(theInstr);
   UChar  p        = get_lsb(theInstr);
   IRType ty       = mode64? Ity_I64 : Ity_I32;
   IRTemp tmp      = newTemp(ty);
   IRTemp tmpRs    = newTemp(ty);
   IRTemp tmpRt    = newTemp(ty);
   IRTemp t1       = newTemp(ty);
   UInt size;
   assign(tmpRs, getIReg(regRs));

   switch(opc1) {
      case 0x1C: {
         switch(opc2) { 
            case 0x03: {  /* DMUL rd, rs, rt */
               DIP("dmul r%u, r%u, r%u", regRd, regRs, regRt);
               IRTemp t0 = newTemp(Ity_I128);
               assign(t0, binop(Iop_MullU64, getIReg(regRs), getIReg(regRt)));
               putIReg(regRd, unop(Iop_128to64, mkexpr(t0)));
               break;
            }

            case 0x18: {  /* Store Atomic Add Word - SAA; Cavium OCTEON */
               DIP("saa r%u, (r%u)", regRt, regRs);
               IRTemp addr = newTemp(Ity_I64);
               IRTemp new_val = newTemp(Ity_I32);
               IRTemp old = newTemp(Ity_I32);
               assign(addr, getIReg(regRs));
               assign(old, load(Ity_I32, mkexpr(addr)));
               assign(new_val, binop(Iop_Add32,
                                     mkexpr(old),
                                     mkNarrowTo32(ty, getIReg(regRt))));
               mips_load_store32(addr, new_val, old, 0, False);
               break;
            }

            /* Store Atomic Add Doubleword - SAAD; Cavium OCTEON */
            case 0x19: {
               DIP( "saad r%u, (r%u)", regRt, regRs);
               IRTemp addr = newTemp(Ity_I64);
               IRTemp new_val = newTemp(Ity_I64);
               IRTemp old = newTemp(Ity_I64);
               assign(addr, getIReg(regRs));
               assign(old, load(Ity_I64, mkexpr(addr)));
               assign(new_val, binop(Iop_Add64,
                                     mkexpr(old),
                                     getIReg(regRt)));
               mips_load_store64(addr, new_val, old, 0, False);
               break;
            }

            /* LAI, LAID, LAD, LADD, LAS, LASD,
               LAC, LACD, LAA, LAAD, LAW, LAWD */
            case 0x1f: {
               UInt opc3 = get_sa(theInstr);
               IRTemp addr = newTemp(Ity_I64);
               switch (opc3) {
                  /* Load Atomic Increment Word - LAI; Cavium OCTEON2 */
                  case 0x02: {
                     DIP("lai r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     assign(new_val, binop(Iop_Add32,
                                           mkexpr(old),
                                           mkU32(1)));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Increment Doubleword - LAID; Cavium OCTEON2 */
                  case 0x03: {
                     DIP("laid r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     assign(new_val, binop(Iop_Add64,
                                           mkexpr(old),
                                           mkU64(1)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Decrement Word - LAD; Cavium OCTEON2 */
                  case 0x06: {
                     DIP("lad r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     assign(new_val, binop(Iop_Sub32,
                                           mkexpr(old),
                                           mkU32(1)));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Decrement Doubleword - LADD; Cavium OCTEON2 */
                  case 0x07: {
                     DIP("ladd r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     assign(new_val, binop(Iop_Sub64,
                                           mkexpr(old),
                                           mkU64(1)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Set Word - LAS; Cavium OCTEON2 */
                  case 0x0a: {
                     DIP("las r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(new_val, mkU32(0xffffffff));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Set Doubleword - LASD; Cavium OCTEON2 */
                  case 0x0b: {
                     DIP("lasd r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(new_val, mkU64(0xffffffffffffffffULL));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Clear Word - LAC; Cavium OCTEON2 */
                  case 0x0e: {
                     DIP("lac r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(new_val, mkU32(0));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Clear Doubleword - LACD; Cavium OCTEON2 */
                  case 0x0f: {
                     DIP("lacd r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(new_val, mkU64(0));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Add Word - LAA; Cavium OCTEON2 */
                  case 0x12: {
                     DIP("laa r%u,(r%u),r%u\n", regRd, regRs, regRt);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     assign(new_val, binop(Iop_Add32,
                                           mkexpr(old),
                                           mkNarrowTo32(ty, getIReg(regRt))));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Add Doubleword - LAAD; Cavium OCTEON2 */
                  case 0x13: {
                     DIP("laad r%u,(r%u),r%u\n", regRd, regRs, regRt);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     assign(new_val, binop(Iop_Add64,
                                           load(Ity_I64, mkexpr(addr)),
                                           getIReg(regRt)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Swap Word - LAW; Cavium OCTEON2 */
                  case 0x16: {
                     DIP("law r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I32);
                     IRTemp old = newTemp(Ity_I32);
                     assign(addr, getIReg(regRs));
                     assign(new_val, mkNarrowTo32(ty, getIReg(regRt)));
                     assign(old, load(Ity_I32, mkexpr(addr)));
                     mips_load_store32(addr, new_val, old, regRd, True);
                     break;
                  }
                  /* Load Atomic Swap Doubleword - LAWD; Cavium OCTEON2 */
                  case 0x17: {
                     DIP("lawd r%u,(r%u)\n", regRd, regRs);
                     IRTemp new_val = newTemp(Ity_I64);
                     IRTemp old = newTemp(Ity_I64);
                     assign(addr, getIReg(regRs));
                     assign(new_val, getIReg(regRt));
                     assign(old, load(Ity_I64, mkexpr(addr)));
                     mips_load_store64(addr, new_val, old, regRd, True);
                     break;
                  }
                  default:
                     vex_printf("Unknown laxx instruction, opc3=0x%x\n", opc3);
                     vex_printf("Instruction=0x%08x\n", theInstr);
                     return False;
               }
               break;
            }

            /* Unsigned Byte Add - BADDU rd, rs, rt; Cavium OCTEON */
            case 0x28: {
               DIP("BADDU r%u, r%u, r%u", regRs, regRt, regRd);
               IRTemp t0 = newTemp(Ity_I8);
 
               assign(t0, binop(Iop_Add8,
                                mkNarrowTo8(ty, getIReg(regRs)),
                                mkNarrowTo8(ty, getIReg(regRt))));
 
               if (mode64)
                  putIReg(regRd, binop(mkSzOp(ty, Iop_And8),
                                       unop(Iop_8Uto64, mkexpr(t0)),
                                       mkSzImm(ty, 0xFF)));
               else
                  putIReg(regRd, binop(mkSzOp(ty, Iop_And8),
                                       unop(Iop_8Uto32, mkexpr(t0)),
                                       mkSzImm(ty, 0xFF)));
               break;
            }
 
            case 0x2c: {  /* Count Ones in a Word - POP; Cavium OCTEON */
               int i, shift[5];
               IRTemp mask[5];
               IRTemp old = newTemp(ty);
               IRTemp nyu = IRTemp_INVALID;
               assign(old, getIReg(regRs));
               DIP("pop r%u, r%u", regRd, regRs);
 
               for (i = 0; i < 5; i++) {
                  mask[i] = newTemp(ty);
                  shift[i] = 1 << i;
               }
               if(mode64) {
                  assign(mask[0], mkU64(0x0000000055555555));
                  assign(mask[1], mkU64(0x0000000033333333));
                  assign(mask[2], mkU64(0x000000000F0F0F0F));
                  assign(mask[3], mkU64(0x0000000000FF00FF));
                  assign(mask[4], mkU64(0x000000000000FFFF));
  
                  for (i = 0; i < 5; i++) {
                     nyu = newTemp(ty);
                     assign(nyu,
                            binop(Iop_Add64,
                                  binop(Iop_And64,
                                        mkexpr(old), mkexpr(mask[i])),
                                  binop(Iop_And64,
                                        binop(Iop_Shr64,
                                              mkexpr(old), mkU8(shift[i])),
                                        mkexpr(mask[i]))));
                     old = nyu;
                  }
               } else {
                  assign(mask[0], mkU32(0x55555555));
                  assign(mask[1], mkU32(0x33333333));
                  assign(mask[2], mkU32(0x0F0F0F0F));
                  assign(mask[3], mkU32(0x00FF00FF));
                  assign(mask[4], mkU32(0x0000FFFF));
                  assign(old, getIReg(regRs));
 
                  for (i = 0; i < 5; i++) {
                     nyu = newTemp(ty);
                     assign(nyu,
                            binop(Iop_Add32,
                                  binop(Iop_And32,
                                        mkexpr(old), mkexpr(mask[i])),
                                  binop(Iop_And32,
                                        binop(Iop_Shr32,
                                              mkexpr(old), mkU8(shift[i])),
                                        mkexpr(mask[i]))));
                     old = nyu;
                  }
               }
               putIReg(regRd, mkexpr(nyu));
               break;
            }
 
            /* Count Ones in a Doubleword - DPOP; Cavium OCTEON */
            case 0x2d: {
               int i, shift[6];
               IRTemp mask[6];
               IRTemp old = newTemp(ty);
               IRTemp nyu = IRTemp_INVALID;
               DIP("dpop r%u, r%u", regRd, regRs);
 
               for (i = 0; i < 6; i++) {
                  mask[i] = newTemp(ty);
                  shift[i] = 1 << i;
               }
               vassert(mode64); /*Caution! Only for Mode 64*/
               assign(mask[0], mkU64(0x5555555555555555ULL));
               assign(mask[1], mkU64(0x3333333333333333ULL));
               assign(mask[2], mkU64(0x0F0F0F0F0F0F0F0FULL));
               assign(mask[3], mkU64(0x00FF00FF00FF00FFULL));
               assign(mask[4], mkU64(0x0000FFFF0000FFFFULL));
               assign(mask[5], mkU64(0x00000000FFFFFFFFULL));
               assign(old, getIReg(regRs));
               for (i = 0; i < 6; i++) {
                  nyu = newTemp(Ity_I64);
                  assign(nyu,
                         binop(Iop_Add64,
                               binop(Iop_And64,
                                     mkexpr(old), mkexpr(mask[i])),
                               binop(Iop_And64,
                                     binop(Iop_Shr64,
                                           mkexpr(old), mkU8(shift[i])),
                                     mkexpr(mask[i]))));
                  old = nyu;
               }
               putIReg(regRd, mkexpr(nyu));
               break;
            }

            case 0x32:  /* 5. CINS rd, rs, p, lenm1 */
               DIP("cins r%u, r%u, %u, %u\n", regRt, regRs, p, lenM1); 
               assign ( tmp  , binop(Iop_Shl64, mkexpr(tmpRs),
                                     mkU8(64-( lenM1+1 ))));
               assign ( tmpRt, binop(Iop_Shr64, mkexpr( tmp ),
                                     mkU8(64-(p+lenM1+1))));
               putIReg( regRt, mkexpr(tmpRt));
               break;

            case 0x33:  /* 6. CINS32 rd, rs, p+32, lenm1 */
               DIP("cins32 r%u, r%u, %d, %d\n", regRt, regRs, p+32, lenM1);
               assign ( tmp  , binop(Iop_Shl64, mkexpr(tmpRs),
                                     mkU8(64-( lenM1+1 ))));
               assign ( tmpRt, binop(Iop_Shr64, mkexpr( tmp ),
                                     mkU8(32-(p+lenM1+1))));
               putIReg( regRt, mkexpr(tmpRt));
               break;

            case 0x3A:  /* 3. EXTS rt, rs, p len */
               DIP("exts r%u, r%u, %d, %d\n", regRt, regRs, p, lenM1); 
               size = lenM1 + 1;  /* lenm1+1 */
               UChar lsAmt = 64 - (p + size);  /* p+lenm1+1 */
               UChar rsAmt = 64 - size;  /* lenm1+1 */
               tmp = newTemp(Ity_I64);
               assign(tmp, binop(Iop_Shl64, mkexpr(tmpRs), mkU8(lsAmt)));
               putIReg(regRt, binop(Iop_Sar64, mkexpr(tmp), mkU8(rsAmt)));
               break;

            case 0x3B:  /* 4. EXTS32 rt, rs, p len */
               DIP("exts32 r%u, r%u, %d, %d\n", regRt, regRs, p, lenM1); 
               assign ( tmp  , binop(Iop_Shl64, mkexpr(tmpRs),
                                     mkU8(32-(p+lenM1+1))));
               assign ( tmpRt, binop(Iop_Sar64, mkexpr(tmp),
                                     mkU8(64-(lenM1+1))) );
               putIReg( regRt, mkexpr(tmpRt));
               break;

            case 0x2B:  /* 20. SNE rd, rs, rt */
               DIP("sne r%u, r%u, r%u", regRd,regRs, regRt);
               if (mode64)
                  putIReg(regRd, unop(Iop_1Uto64, binop(Iop_CmpNE64,
                                                        getIReg(regRs),
                                                        getIReg(regRt))));
               else
                  putIReg(regRd,unop(Iop_1Uto32, binop(Iop_CmpNE32,
                                                       getIReg(regRs),
                                                       getIReg(regRt))));
               break;

            case 0x2A:  /* Set Equals - SEQ; Cavium OCTEON */
               DIP("seq r%u, r%u, %d", regRd, regRs, regRt);
               if (mode64)
                  putIReg(regRd, unop(Iop_1Uto64,
                                      binop(Iop_CmpEQ64, getIReg(regRs),
                                            getIReg(regRt))));
               else
                  putIReg(regRd, unop(Iop_1Uto32,
                                      binop(Iop_CmpEQ32, getIReg(regRs),
                                            getIReg(regRt))));
               break;

            case 0x2E:  /* Set Equals Immediate - SEQI; Cavium OCTEON */
               DIP("seqi r%u, r%u, %u", regRt, regRs, imm);
               if (mode64)
                  putIReg(regRt, unop(Iop_1Uto64,
                                      binop(Iop_CmpEQ64, getIReg(regRs),
                                            mkU64(extend_s_10to64(imm)))));
               else
                  putIReg(regRt, unop(Iop_1Uto32,
                                      binop(Iop_CmpEQ32, getIReg(regRs),
                                            mkU32(extend_s_10to32(imm)))));
               break;

            case 0x2F:  /* Set Not Equals Immediate - SNEI; Cavium OCTEON */
               DIP("snei r%u, r%u, %u", regRt, regRs, imm);
               if (mode64)
                  putIReg(regRt, unop(Iop_1Uto64,
                                   binop(Iop_CmpNE64,
                                         getIReg(regRs),
                                         mkU64(extend_s_10to64(imm)))));
               else
                  putIReg(regRt, unop(Iop_1Uto32,
                                   binop(Iop_CmpNE32,
                                         getIReg(regRs),
                                         mkU32(extend_s_10to32(imm)))));
               break;

            default:
               return False;
         }
         break;
      } /* opc1 0x1C ends here*/
      case 0x1F: {
         switch(opc2) {
            case 0x0A: {  // lx - Load indexed instructions
               switch (get_sa(theInstr)) {
                  case 0x00: {  // LWX rd, index(base)
                     DIP("lwx r%u, r%u(r%u)", regRd, regRt, regRs);
                     LOADX_STORE_PATTERN;
                     putIReg(regRd, mkWidenFrom32(ty, load(Ity_I32, mkexpr(t1)),
                                                  True));
                     break;
                  }
                  case 0x04:  // LHX rd, index(base)
                     DIP("lhx r%u, r%u(r%u)", regRd, regRt, regRs);
                     LOADX_STORE_PATTERN;
                     if (mode64)
                        putIReg(regRd, unop(Iop_16Sto64, load(Ity_I16,
                                                              mkexpr(t1))));
                     else
                        putIReg(regRd, unop(Iop_16Sto32, load(Ity_I16,
                                                              mkexpr(t1))));
                     break;
                  case 0x08: {  // LDX rd, index(base)
                     DIP("ldx r%u, r%u(r%u)", regRd, regRt, regRs);
                     vassert(mode64); /* Currently Implemented only for n64 */
                     LOADX_STORE_PATTERN;
                     putIReg(regRd, load(Ity_I64, mkexpr(t1)));
                     break;
                  }
                  case 0x06: {  // LBUX rd, index(base)
                     DIP("lbux r%u, r%u(r%u)", regRd, regRt, regRs);
                     LOADX_STORE_PATTERN;
                     if (mode64)
                        putIReg(regRd, unop(Iop_8Uto64, load(Ity_I8,
                                                             mkexpr(t1))));
                     else
                        putIReg(regRd, unop(Iop_8Uto32, load(Ity_I8,
                                                             mkexpr(t1))));
                     break;
                  }
                  case 0x10: {  // LWUX rd, index(base) (Cavium OCTEON)
                     DIP("lwux r%u, r%u(r%u)", regRd, regRt, regRs);
                     LOADX_STORE_PATTERN; /* same for both 32 and 64 modes*/
                     putIReg(regRd, mkWidenFrom32(ty, load(Ity_I32, mkexpr(t1)),
                                                  False));
                     break;
                  }
                  case 0x14: {  // LHUX rd, index(base) (Cavium OCTEON)
                     DIP("lhux r%u, r%u(r%u)", regRd, regRt, regRs);
                     LOADX_STORE_PATTERN;
                     if (mode64)
                        putIReg(regRd,
                                unop(Iop_16Uto64, load(Ity_I16, mkexpr(t1))));
                     else
                        putIReg(regRd,
                                unop(Iop_16Uto32, load(Ity_I16, mkexpr(t1))));
                     break;
                  }
                  case 0x16: {  // LBX rd, index(base) (Cavium OCTEON)
                     DIP("lbx r%u, r%u(r%u)", regRd, regRs, regRt);
                     LOADX_STORE_PATTERN;
                     if (mode64)
                        putIReg(regRd,
                                unop(Iop_8Sto64, load(Ity_I8, mkexpr(t1))));
                     else
                        putIReg(regRd,
                                unop(Iop_8Sto32, load(Ity_I8, mkexpr(t1))));
                     break;
                  }
                  default:
                     vex_printf("\nUnhandled LX instruction opc3 = %x\n",
                                get_sa(theInstr));
                     return False;
               }
               break;
            }
         } /* opc1 = 0x1F & opc2 = 0xA (LX) ends here*/
         break;
      } /* opc1 = 0x1F ends here*/
      default:
         return False; 
   } /* main opc1 switch ends here */
   return True;
}

/*------------------------------------------------------------*/
/*---       Disassemble a single DSP ASE instruction       ---*/
/*------------------------------------------------------------*/

static UInt disDSPInstr_MIPS_WRK ( UInt cins )
{
   IRTemp t0, t1 = 0, t2, t3, t4, t5, t6, t7, t8, t9, t10, t11, t12, t13, t14,
          t15, t16, t17;
   UInt opcode, rs, rt, rd, sa, function, ac, ac_mfhilo, rddsp_mask,
        wrdsp_mask, dsp_imm, shift;

   opcode = get_opcode(cins);
   rs = get_rs(cins);
   rt = get_rt(cins);
   rd = get_rd(cins);
   sa = get_sa(cins);
   function = get_function(cins);
   ac = get_acNo(cins);
   ac_mfhilo = get_acNo_mfhilo(cins);
   rddsp_mask = get_rddspMask(cins);
   wrdsp_mask = get_wrdspMask(cins);
   dsp_imm = get_dspImm(cins);
   shift = get_shift(cins);

   switch (opcode) {
      case 0x00: {  /* Special */
         switch (function) {
            case 0x10: {  /* MFHI */
               DIP("mfhi ac%u r%u", ac_mfhilo, rd);
               putIReg(rd, unop(Iop_64HIto32, getAcc(ac_mfhilo)));
               break;
            }

            case 0x11: {  /* MTHI */
               DIP("mthi ac%u r%u", ac, rs);
               t1 = newTemp(Ity_I32);
               assign(t1, unop(Iop_64to32, getAcc(ac)));
               putAcc(ac, binop(Iop_32HLto64, getIReg(rs), mkexpr(t1)));
               break;
            }

            case 0x12: {  /* MFLO */
               DIP("mflo ac%u r%u", ac_mfhilo, rd);
               putIReg(rd, unop(Iop_64to32, getAcc(ac_mfhilo)));
               break;
            }

            case 0x13: {  /* MTLO */
               DIP("mtlo ac%u r%u", ac, rs);
               t1 = newTemp(Ity_I32);
               assign(t1, unop(Iop_64HIto32, getAcc(ac)));
               putAcc(ac, binop(Iop_32HLto64, mkexpr(t1), getIReg(rs)));
               break;
            }

            case 0x18: {  /* MULT */
               DIP("mult ac%u r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               assign(t1, binop(Iop_MullS32, mkNarrowTo32(Ity_I32, getIReg(rs)),
                                mkNarrowTo32(Ity_I32, getIReg(rt))));
               putAcc(ac, mkexpr(t1));
               break;
            }

            case 0x19: {  /* MULTU */
               DIP("multu ac%u r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               assign(t1, binop(Iop_MullU32, mkNarrowTo32(Ity_I32, getIReg(rs)),
                                             mkNarrowTo32(Ity_I32,
                                                          getIReg(rt))));
               putAcc(ac, mkexpr(t1));
            break;
            }
         }
         break;
      }
      case 0x1C: {  /* Special2 */
         switch (function) {
            case 0x00: {  /* MADD */
               DIP("madd ac%u, r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               t2 = newTemp(Ity_I64);
               t3 = newTemp(Ity_I64);

               assign(t1, getAcc(ac));
               assign(t2, binop(Iop_MullS32, getIReg(rs), getIReg(rt)));
               assign(t3, binop(Iop_Add64, mkexpr(t1), mkexpr(t2)));

               putAcc(ac, mkexpr(t3));
               break;
            }
            case 0x01: {  /* MADDU */
               DIP("maddu ac%u r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               t2 = newTemp(Ity_I64);
               t3 = newTemp(Ity_I64);

               assign(t1, getAcc(ac));
               assign(t2, binop(Iop_MullU32, getIReg(rs), getIReg(rt)));
               assign(t3, binop(Iop_Add64, mkexpr(t2), mkexpr(t1)));

               putAcc(ac, mkexpr(t3));
               break;
            }
            case 0x04: {  /* MSUB */
               DIP("msub ac%u r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               t2 = newTemp(Ity_I64);
               t3 = newTemp(Ity_I64);

               assign(t1, getAcc(ac));
               assign(t2, binop(Iop_MullS32, getIReg(rs), getIReg(rt)));
               assign(t3, binop(Iop_Sub64, mkexpr(t1), mkexpr(t2)));

               putAcc(ac, mkexpr(t3));
               break;
            }
            case 0x05: {  /* MSUBU */
               DIP("msubu ac%u r%u, r%u", ac, rs, rt);
               t1 = newTemp(Ity_I64);
               t2 = newTemp(Ity_I64);
               t3 = newTemp(Ity_I64);

               assign(t1, getAcc(ac));
               assign(t2, binop(Iop_MullU32, getIReg(rs), getIReg(rt)));
               assign(t3, binop(Iop_Sub64, mkexpr(t1), mkexpr(t2)));

               putAcc(ac, mkexpr(t3));
               break;
            }
         }
         break;
      }
      case 0x1F: {  /* Special3 */
         switch (function) {
            case 0x12: {  /* ABSQ_S.PH */
               switch (sa) {
                  case 0x1: {  /* ABSQ_S.QB */
                     DIP("absq_s.qb r%u, r%u", rd, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I8);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I8);
                     t8 = newTemp(Ity_I8);
                     t9 = newTemp(Ity_I1);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I8);
                     t12 = newTemp(Ity_I8);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I1);
                     t15 = newTemp(Ity_I8);
                     t16 = newTemp(Ity_I32);
                     t17 = newTemp(Ity_I32);

                     /* Absolute value of the rightmost byte (bits 7-0). */
                     /* t0 - rightmost byte. */
                     assign(t0, unop(Iop_16to8, unop(Iop_32to16, getIReg(rt))));
                     /* t1 holds 1 if t0 is equal to 0x80, or 0 otherwise. */
                     assign(t1, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32, mkexpr(t0)),
                                      mkU32(0x00000080)));
                     /* t2 holds 1 if value in t0 is negative, 0 otherwise. */
                     assign(t2, unop(Iop_32to1,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 getIReg(rt),
                                                 mkU32(0x00000080)),
                                           mkU8(0x7))));
                     /* t3 holds abs(t0). */
                     assign(t3, IRExpr_ITE(mkexpr(t1),
                                           mkU8(0x7F),
                                           IRExpr_ITE(mkexpr(t2),
                                                      binop(Iop_Add8,
                                                            unop(Iop_Not8,
                                                                 mkexpr(t0)),
                                                            mkU8(0x1)),
                                                      mkexpr(t0))));

                     /* Absolute value of bits 15-8. */
                     /* t4 - input byte. */
                     assign(t4,
                            unop(Iop_16HIto8, unop(Iop_32to16, getIReg(rt))));
                     /* t5 holds 1 if t4 is equal to 0x80, or 0 otherwise. */
                     assign(t5, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32, mkexpr(t4)),
                                      mkU32(0x00000080)));
                     /* t6 holds 1 if value in t4 is negative, 0 otherwise. */
                     assign(t6, unop(Iop_32to1,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 getIReg(rt),
                                                 mkU32(0x00008000)),
                                           mkU8(15))));
                     /* t3 holds abs(t4). */
                     assign(t7, IRExpr_ITE(mkexpr(t5),
                                           mkU8(0x7F),
                                           IRExpr_ITE(mkexpr(t6),
                                                      binop(Iop_Add8,
                                                            unop(Iop_Not8,
                                                                 mkexpr(t4)),
                                                            mkU8(0x1)),
                                                      mkexpr(t4))));

                     /* Absolute value of bits 23-15. */
                     /* t8 - input byte. */
                     assign(t8,
                            unop(Iop_16to8, unop(Iop_32HIto16, getIReg(rt))));
                     /* t9 holds 1 if t8 is equal to 0x80, or 0 otherwise. */
                     assign(t9, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32, mkexpr(t8)),
                                      mkU32(0x00000080)));
                     /* t6 holds 1 if value in t8 is negative, 0 otherwise. */
                     assign(t10, unop(Iop_32to1,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0x00800000)),
                                            mkU8(23))));
                     /* t3 holds abs(t8). */
                     assign(t11, IRExpr_ITE(mkexpr(t9),
                                            mkU8(0x7F),
                                            IRExpr_ITE(mkexpr(t10),
                                                       binop(Iop_Add8,
                                                             unop(Iop_Not8,
                                                                  mkexpr(t8)),
                                                             mkU8(0x1)),
                                                       mkexpr(t8))));

                     /* Absolute value of bits 31-24. */
                     /* t12 - input byte. */
                     assign(t12,
                            unop(Iop_16HIto8, unop(Iop_32HIto16, getIReg(rt))));
                     /* t13 holds 1 if t12 is equal to 0x80, or 0 otherwise. */
                     assign(t13, binop(Iop_CmpEQ32,
                                       unop(Iop_8Uto32, mkexpr(t12)),
                                       mkU32(0x00000080)));
                     /* t14 holds 1 if value in t12 is negative, 0 otherwise. */
                     assign(t14, unop(Iop_32to1,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0x80000000)),
                                            mkU8(31))));
                     /* t15 holds abs(t12). */
                     assign(t15, IRExpr_ITE(mkexpr(t13),
                                            mkU8(0x7F),
                                            IRExpr_ITE(mkexpr(t14),
                                                       binop(Iop_Add8,
                                                             unop(Iop_Not8,
                                                                  mkexpr(t12)),
                                                             mkU8(0x1)),
                                                       mkexpr(t12))));

                     /* t16 holds !0 if any of input bytes is 0x80 or 0
                        otherwise. */
                     assign(t16,
                            binop(Iop_Or32,
                                  binop(Iop_Or32,
                                        binop(Iop_Or32,
                                              unop(Iop_1Sto32, mkexpr(t13)),
                                              unop(Iop_1Sto32, mkexpr(t9))),
                                        unop(Iop_1Sto32, mkexpr(t5))),
                                  unop(Iop_1Sto32, mkexpr(t1))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    mkexpr(t16),
                                                    mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000))));

                     /* t17 = t15|t11|t7|t3 */
                     assign(t17,
                            binop(Iop_16HLto32,
                                  binop(Iop_8HLto16, mkexpr(t15), mkexpr(t11)),
                                  binop(Iop_8HLto16, mkexpr(t7), mkexpr(t3))));

                     putIReg(rd, mkexpr(t17));
                     break;
                  }
                  case 0x2: {  /* REPL.QB */
                     DIP("repl.qb r%u, %u", rd, dsp_imm);
                     vassert(!mode64);

                     putIReg(rd, mkU32((dsp_imm << 24) | (dsp_imm << 16) |
                                       (dsp_imm << 8) | (dsp_imm)));
                     break;
                  }
                  case 0x3: {  /* REPLV.QB */
                     DIP("replv.qb r%u, r%u", rd, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I8);

                     assign(t0, unop(Iop_32to8,
                                binop(Iop_And32, getIReg(rt), mkU32(0xff))));
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16, mkexpr(t0), mkexpr(t0)),
                                   binop(Iop_8HLto16, mkexpr(t0), mkexpr(t0))));
                     break;
                  }
                  case 0x4: {  /* PRECEQU.PH.QBL */
                     DIP("precequ.ph.qbl r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0xff000000)),
                                             mkU8(1)),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x00ff0000)),
                                             mkU8(9))));
                     break;
                  }
                  case 0x5: {  /* PRECEQU.PH.QBR */
                     DIP("precequ.ph.qbr r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shl32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x0000ff00)),
                                             mkU8(15)),
                                       binop(Iop_Shl32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x000000ff)),
                                             mkU8(7))));
                     break;
                  }
                  case 0x6: {  /* PRECEQU.PH.QBLA */
                     DIP("precequ.ph.qbla r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0xff000000)),
                                             mkU8(1)),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x0000ff00)),
                                             mkU8(1))));
                     break;
                  }
                  case 0x7: {  /* PRECEQU.PH.QBRA */
                     DIP("precequ.ph.qbra r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shl32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x00ff0000)),
                                             mkU8(7)),
                                       binop(Iop_Shl32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x000000ff)),
                                             mkU8(7))));
                     break;
                  }
                  case 0x9: {  /* ABSQ_S.PH */
                     DIP("absq_s.ph r%u, r%u", rd, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I16);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I16);
                     t4 = newTemp(Ity_I16);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I16);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     /* t0 holds lower 16 bits of value in rt. */
                     assign(t0, unop(Iop_32to16, getIReg(rt)));
                     /* t1 holds 1 if t0 is equal to 0x8000. */
                     assign(t1, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32, mkexpr(t0)),
                                      mkU32(0x00008000)));
                     /* t2 holds 1 if value in t0 is negative, 0 otherwise. */
                     assign(t2, unop(Iop_32to1,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 getIReg(rt),
                                                 mkU32(0x00008000)),
                                           mkU8(15))));
                     /* t3 holds abs(t0). */
                     assign(t3, IRExpr_ITE(mkexpr(t1),
                                           mkU16(0x7FFF),
                                           IRExpr_ITE(mkexpr(t2),
                                                      binop(Iop_Add16,
                                                            unop(Iop_Not16,
                                                                 mkexpr(t0)),
                                                            mkU16(0x1)),
                                                      mkexpr(t0))));

                     /* t4 holds lower 16 bits of value in rt. */
                     assign(t4, unop(Iop_32HIto16, getIReg(rt)));
                     /* t5 holds 1 if t4 is equal to 0x8000. */
                     assign(t5, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32, mkexpr(t4)),
                                      mkU32(0x00008000)));
                     /* t6 holds 1 if value in t4 is negative, 0 otherwise. */
                     assign(t6, unop(Iop_32to1,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 getIReg(rt),
                                                 mkU32(0x80000000)),
                                           mkU8(31))));
                     /* t7 holds abs(t4). */
                     assign(t7, IRExpr_ITE(mkexpr(t5),
                                           mkU16(0x7FFF),
                                           IRExpr_ITE(mkexpr(t6),
                                                      binop(Iop_Add16,
                                                            unop(Iop_Not16,
                                                                 mkexpr(t4)),
                                                            mkU16(0x1)),
                                                      mkexpr(t4))));
                     /* If any of the two input halfwords is equal 0x8000,
                        set bit 20 in DSPControl register. */
                     assign(t8, binop(Iop_Or32,
                                      unop(Iop_1Sto32, mkexpr(t5)),
                                      unop(Iop_1Sto32, mkexpr(t1))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    mkexpr(t8),
                                                    mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000))));

                     /* t9 = t7|t3 */
                     assign(t9, binop(Iop_16HLto32, mkexpr(t7), mkexpr(t3)));

                     putIReg(rd, mkexpr(t9));
                     break;
                  }
                  case 0xA: {  /* REPL.PH */
                     DIP("repl.ph r%u, %u", rd, dsp_imm);
                     vassert(!mode64);
                     UShort immediate = extend_s_10to16(dsp_imm);

                     putIReg(rd, mkU32(immediate << 16 | immediate));
                     break;
                  }
                  case 0xB: {  /* REPLV.PH */
                     DIP("replv.ph r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, getIReg(rt)),
                                       unop(Iop_32to16, getIReg(rt))));
                     break;
                  }
                  case 0xC: {  /* PRECEQ.W.PHL */
                     DIP("preceq.w.phl r%u, r%u", rd, rt);
                     vassert(!mode64);
                     putIReg(rd, binop(Iop_And32,
                                       getIReg(rt),
                                       mkU32(0xffff0000)));
                     break;
                  }
                  case 0xD: {  /* PRECEQ.W.PHR */
                     DIP("preceq.w.phr r%u, r%u", rd, rt);
                     vassert(!mode64);
                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, getIReg(rt)),
                                       mkU16(0x0)));
                     break;
                  }
                  case 0x11: {  /* ABSQ_S.W */
                     DIP("absq_s.w r%u, r%u", rd, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I1);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);

                     assign(t0,
                            binop(Iop_CmpEQ32, getIReg(rt), mkU32(0x80000000)));

                     putDSPControl(IRExpr_ITE(mkexpr(t0),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     assign(t1, binop(Iop_CmpLT32S, getIReg(rt), mkU32(0x0)));

                     assign(t2, IRExpr_ITE(mkexpr(t0),
                                           mkU32(0x7FFFFFFF),
                                           IRExpr_ITE(mkexpr(t1),
                                                      binop(Iop_Add32,
                                                            unop(Iop_Not32,
                                                                 getIReg(rt)),
                                                            mkU32(0x1)),
                                                      getIReg(rt))));
                     putIReg(rd, mkexpr(t2));
                     break;
                  }
                  case 0x1B: {  /* BITREV */
                     DIP("bitrev r%u, r%u", rd, rt);
                     vassert(!mode64);
                     /* 32bit reversal as seen on Bit Twiddling Hacks site
                        http://graphics.stanford.edu/~seander/bithacks.html
                        section ReverseParallel */
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_Or32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0xaaaaaaaa)),
                                            mkU8(0x1)),
                                      binop(Iop_Shl32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0x55555555)),
                                            mkU8(0x1))));
                     assign(t2, binop(Iop_Or32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t1),
                                                  mkU32(0xcccccccc)),
                                            mkU8(0x2)),
                                      binop(Iop_Shl32,
                                            binop(Iop_And32,
                                                  mkexpr(t1),
                                                  mkU32(0x33333333)),
                                            mkU8(0x2))));
                     assign(t3, binop(Iop_Or32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0xf0f0f0f0)),
                                            mkU8(0x4)),
                                      binop(Iop_Shl32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0x0f0f0f0f)),
                                            mkU8(0x4))));
                     assign(t4, binop(Iop_Or32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t3),
                                                  mkU32(0xff00ff00)),
                                            mkU8(0x8)),
                                      binop(Iop_Shl32,
                                            binop(Iop_And32,
                                                  mkexpr(t3),
                                                  mkU32(0x00ff00ff)),
                                            mkU8(0x8))));
                     assign(t5, binop(Iop_Or32,
                                      binop(Iop_Shr32,
                                            mkexpr(t4),
                                            mkU8(0x10)),
                                      binop(Iop_Shl32,
                                            mkexpr(t4),
                                            mkU8(0x10))));
                     putIReg(rd, binop(Iop_Shr32,
                                       mkexpr(t5),
                                       mkU8(16)));
                     break;
                  }
                  case 0x1C: {  /* PRECEU.PH.QBL */
                     DIP("preceu.ph.qbl r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0xff000000)),
                                             mkU8(8)),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x00ff0000)),
                                             mkU8(16))));
                     break;
                  }
                  case 0x1E: {  /* PRECEU.PH.QBLA */
                     DIP("preceu.ph.qbla r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0xff000000)),
                                             mkU8(8)),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x0000ff00)),
                                             mkU8(8))));
                     break;
                  }
                  case 0x1D: {  /* PRECEU.PH.QBR */
                     DIP("preceu.ph.qbr r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Shl32,
                                             binop(Iop_And32,
                                                   getIReg(rt),
                                                   mkU32(0x0000ff00)),
                                             mkU8(8)),
                                       binop(Iop_And32,
                                             getIReg(rt),
                                             mkU32(0x000000ff))));
                     break;
                  }
                  case 0x1F: {  /* PRECEU.PH.QBRA */
                     DIP("preceu.ph.qbra r%u, r%u", rd, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_And32,
                                             getIReg(rt),
                                             mkU32(0x00ff0000)),
                                       binop(Iop_And32,
                                             getIReg(rt),
                                             mkU32(0x000000ff))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of ABSQ_S.PH */
            }
            case 0x38: {  /* EXTR.W */
               switch(sa) {
                  case 0x0: {  /* EXTR.W */
                     DIP("extr.w r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     if (0 == rs) {
                        assign(t1, mkexpr(t0));
                     } else {
                        assign(t1, binop(Iop_Sar64, mkexpr(t0), mkU8(rs)));
                     }
                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));
                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least sifgnificant bit of the shifted value
                        from acc. */
                     if (0 == rs) {
                        assign(t8, mkU64(0x0ULL));
                     } else {
                        assign(t8, binop(Iop_And64,
                                         binop(Iop_Shr64,
                                               mkexpr(t0),
                                               mkU8(rs-1)),
                                         mkU64(0x1ULL)));
                     }
                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     if (0 == rs) {
                        putIReg(rt, unop(Iop_64to32, mkexpr(t0)));
                     } else {
                        putIReg(rt, unop(Iop_64to32, mkexpr(t1)));
                     }
                     break;
                  }
                  case 0x1: {  /* EXTRV.W */
                     DIP("extrv.w r%u, ac%u, r%u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);
                     t15 = newTemp(Ity_I8);

                     assign(t15, unop(Iop_32to8,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0x1f))));
                     assign(t0, getAcc(ac));
                     assign(t1, binop(Iop_Sar64, mkexpr(t0), mkexpr(t15)));
                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        unop(Iop_8Uto32,
                                                             mkexpr(t15)),
                                                        mkU32(0)),
                                                  unop(Iop_64to32, mkexpr(t0)),
                                                  unop(Iop_64to32, mkexpr(t1))));

                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));
                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least sifgnificant bit of the shifted value
                        from acc. */
                     assign(t8,
                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                             unop(Iop_8Uto32,
                                                  mkexpr(t15)),
                                             mkU32(0)),
                                       mkU64(0x0ULL),
                                       binop(Iop_And64,
                                             binop(Iop_Shr64,
                                                   mkexpr(t0),
                                                   unop(Iop_32to8,
                                                        binop(Iop_Sub32,
                                                              unop(Iop_8Uto32,
                                                                   mkexpr(t15)),
                                                                   mkU32(1)))),
                                             mkU64(0x1ULL))));

                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     break;
                  }
                  case 0x2: {  /* EXTP */
                     DIP("extp r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     /* Extract pos field of DSPControl register. */
                     assign(t1, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));

                     /* Check if (pos - size) >= 0 [size <= pos]
                        if (pos < size)
                           put 1 to EFI field of DSPControl register
                        else
                           extract bits from acc and put 0 to EFI field of
                           DSPCtrl */
                     assign(t2, binop(Iop_CmpLT32U, mkexpr(t1), mkU32(rs)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    binop(Iop_And32,
                                                          getDSPControl(),
                                                          mkU32(0xffffbfff)),
                                                    mkU32(0x4000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xffffbfff))));

                     /* If pos <= 31, shift right the value from the acc
                        (pos-size) times and take (size+1) bits from the least
                        significant positions. Otherwise, shift left the value
                        (63-pos) times, take (size+1) bits from the most
                        significant positions and shift right (31-size) times.*/
                     assign(t3, binop(Iop_CmpLE32U, mkexpr(t1), mkU32(31)));

                     assign(t4,
                           IRExpr_ITE(mkexpr(t3),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkexpr(t1), mkU32(rs))),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkU32(63), mkexpr(t1)))));

                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Shr64,
                                                 mkexpr(t0), mkexpr(t4)),
                                           binop(Iop_Shl64,
                                                 mkexpr(t0), mkexpr(t4))));

                     /* t6 holds a mask for bit extraction */
                     assign(t6,
                            IRExpr_ITE(mkexpr(t3),
                                       unop(Iop_Not64,
                                            binop(Iop_Shl64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  mkU8(rs+1))),
                                       unop(Iop_Not64,
                                            binop(Iop_Shr64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  mkU8(rs+1)))));

                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           unop(Iop_64to32,
                                                binop(Iop_And64,
                                                      mkexpr(t5),
                                                      mkexpr(t6))),
                                           binop(Iop_Shr32,
                                                 unop(Iop_64HIto32,
                                                      binop(Iop_And64,
                                                            mkexpr(t5),
                                                            mkexpr(t6))),
                                                 mkU8(31-rs))));

                     putIReg(rt, mkexpr(t7));
                     break;
                  }
                  case 0x3: {  /* EXTPV */
                     DIP("extpv r%u, ac%u, r%u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t8, binop(Iop_And32, getIReg(rs), mkU32(0x1f)));
                     assign(t0, getAcc(ac));
                     /* Extract pos field of DSPControl register. */
                     assign(t1, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));

                     /* Check if (pos - size) >= 0 [size <= pos]
                        if (pos < size)
                           put 1 to EFI field of DSPControl register
                        else
                           extract bits from acc and put 0 to EFI field of
                           DSPCtrl */
                     assign(t2, binop(Iop_CmpLT32U, mkexpr(t1), mkexpr(t8)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    binop(Iop_And32,
                                                          getDSPControl(),
                                                          mkU32(0xffffbfff)),
                                                    mkU32(0x4000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xffffbfff))));

                     /* If pos <= 31, shift right the value from the acc
                        (pos-size) times and take (size+1) bits from the least
                        significant positions. Otherwise, shift left the value
                        (63-pos) times, take (size+1) bits from the most
                        significant positions and shift right (31-size)
                        times. */
                     assign(t3, binop(Iop_CmpLE32U, mkexpr(t1), mkU32(31)));

                     assign(t4,
                           IRExpr_ITE(mkexpr(t3),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkexpr(t1), mkexpr(t8))),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkU32(63), mkexpr(t1)))));

                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Shr64,
                                                 mkexpr(t0), mkexpr(t4)),
                                           binop(Iop_Shl64,
                                                 mkexpr(t0), mkexpr(t4))));

                     /* t6 holds a mask for bit extraction. */
                     assign(t6,
                            IRExpr_ITE(mkexpr(t3),
                                       unop(Iop_Not64,
                                            binop(Iop_Shl64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  unop(Iop_32to8,
                                                       binop(Iop_Add32,
                                                             mkexpr(t8),
                                                             mkU32(1))))),
                                       unop(Iop_Not64,
                                            binop(Iop_Shr64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  unop(Iop_32to8,
                                                       binop(Iop_Add32,
                                                             mkexpr(t8),
                                                             mkU32(1)))))));

                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           unop(Iop_64to32,
                                                binop(Iop_And64,
                                                      mkexpr(t5),
                                                      mkexpr(t6))),
                                           binop(Iop_Shr32,
                                                 unop(Iop_64HIto32,
                                                      binop(Iop_And64,
                                                            mkexpr(t5),
                                                            mkexpr(t6))),
                                                 unop(Iop_32to8,
                                                      binop(Iop_Sub32,
                                                            mkU32(31),
                                                            mkexpr(t8))))));

                     putIReg(rt, mkexpr(t7));
                     break;
                  }
                  case 0x4: {  /* EXTR_R.W */
                     DIP("extr_r.w r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);
                     t15 = newTemp(Ity_I64);
                     t16 = newTemp(Ity_I1);

                     assign(t0, getAcc(ac));
                     assign(t16, binop(Iop_CmpEQ32,
                                       mkU32(rs),
                                       mkU32(0)));
                     assign(t1, IRExpr_ITE(mkexpr(t16),
                                           mkexpr(t0),
                                           binop(Iop_Sar64,
                                                 mkexpr(t0),
                                                 mkU8(rs))));
                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least significant bit of the shifted value
                        from acc. */
                     assign(t15, binop(Iop_Shr64,
                                       mkexpr(t0),
                                       unop(Iop_32to8,
                                            binop(Iop_Sub32,
                                                  binop(Iop_And32,
                                                        mkU32(rs),
                                                        mkU32(0x1f)),
                                                  mkU32(1)))));

                     assign(t8,
                            IRExpr_ITE(mkexpr(t16),
                                       mkU64(0x0ULL),
                                       binop(Iop_And64,
                                             mkexpr(t15),
                                             mkU64(0x0000000000000001ULL))));
                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));
                     putIReg(rt, unop(Iop_64to32, mkexpr(t9)));

                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     break;
                  }
                  case 0x5: {  /* EXTRV_R.W */
                     DIP("extrv_r.w r%u, ac%u, r%u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);
                     t15 = newTemp(Ity_I8);

                     assign(t15, unop(Iop_32to8,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0x1f))));
                     assign(t0, getAcc(ac));
                     assign(t1, binop(Iop_Sar64, mkexpr(t0), mkexpr(t15)));

                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));
                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least sifgnificant bit of the shifted value
                        from acc. */
                     assign(t8,
                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                             unop(Iop_8Uto32,
                                                  mkexpr(t15)),
                                             mkU32(0)),
                                       mkU64(0x0ULL),
                                       binop(Iop_And64,
                                             binop(Iop_Shr64,
                                                   mkexpr(t0),
                                                   unop(Iop_32to8,
                                                        binop(Iop_Sub32,
                                                              unop(Iop_8Uto32,
                                                                   mkexpr(t15)),
                                                                   mkU32(1)))),
                                             mkU64(0x1ULL))));

                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));
                     /* Put rounded value in destination register. */
                     putIReg(rt, unop(Iop_64to32, mkexpr(t9)));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     break;
                  }
                  case 0x6: {  /* EXTR_RS.W */
                     DIP("extr_rs.w r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);
                     t16 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     if (0 == rs) {
                        assign(t1, mkexpr(t0));
                     } else {
                        assign(t1, binop(Iop_Sar64, mkexpr(t0), mkU8(rs)));
                     }

                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));
                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least sifgnificant bit of the shifted value
                        from acc. */
                     if (0 == rs) {
                        assign(t8, mkU64(0x0ULL));
                     } else {
                        assign(t8, binop(Iop_And64,
                                         binop(Iop_Shr64,
                                               mkexpr(t0),
                                               mkU8(rs-1)),
                                         mkU64(0x1ULL)));
                     }

                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     assign(t16, binop(Iop_And32,
                                       unop(Iop_64HIto32,
                                            mkexpr(t9)),
                                       mkU32(0x80000000)));
                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t14),
                                                  mkU32(0)),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             mkexpr(t16),
                                                             mkU32(0)),
                                                       mkU32(0x7fffffff),
                                                       mkU32(0x80000000)),
                                            unop(Iop_64to32, mkexpr(t9))));
                     break;
                  }
                  case 0x7: {  /* EXTRV_RS.W */
                     DIP("extrv_rs.w r%u, ac%u, r%u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I32);
                     t15 = newTemp(Ity_I32);
                     t16 = newTemp(Ity_I32);
                     t17 = newTemp(Ity_I1);

                     assign(t15, binop(Iop_And32,
                                       getIReg(rs),
                                       mkU32(0x1f)));
                     assign(t17, binop(Iop_CmpEQ32,
                                       mkexpr(t15),
                                       mkU32(0)));
                     assign(t0, getAcc(ac));
                     assign(t1, IRExpr_ITE(mkexpr(t17),
                                           mkexpr(t0),
                                           binop(Iop_Sar64,
                                                 mkexpr(t0),
                                                 unop(Iop_32to8,
                                                      mkexpr(t15)))));

                     /* Check if bits 63..31 of the result in t1 aren't 0. */
                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));
                     /* Check if bits 63..31 of the result in t1 aren't
                        0x1ffffffff. */
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t1)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));
                     /* If bits 63..31 aren't 0 nor 0x1ffffffff, set DSP
                        control register. */
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t3)),
                                            unop(Iop_1Sto32, mkexpr(t4))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t5)),
                                            unop(Iop_1Sto32, mkexpr(t6)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t7),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     /* If the last discarded bit is 1, there would be carry
                        when rounding, otherwise there wouldn't. We use that
                        fact and just add the value of the last discarded bit
                        to the least sifgnificant bit of the shifted value
                        from acc. */
                     assign(t8,
                            IRExpr_ITE(mkexpr(t17),
                                       mkU64(0x0ULL),
                                       binop(Iop_And64,
                                             binop(Iop_Shr64,
                                                   mkexpr(t0),
                                                   unop(Iop_32to8,
                                                        binop(Iop_Sub32,
                                                              mkexpr(t15),
                                                              mkU32(1)))),
                                             mkU64(0x1ULL))));

                     assign(t9, binop(Iop_Add64, mkexpr(t1), mkexpr(t8)));

                     /* Repeat previous steps for the rounded value. */
                     assign(t10, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0)));
                     assign(t11, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0)));

                     assign(t12, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32,
                                           mkexpr(t9)),
                                      mkU32(0xffffffff)));
                     assign(t13, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32,
                                                 mkexpr(t9)),
                                            mkU32(0x80000000)),
                                      mkU32(0x80000000)));

                     assign(t14, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t10)),
                                            unop(Iop_1Sto32, mkexpr(t11))),
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32, mkexpr(t12)),
                                            unop(Iop_1Sto32, mkexpr(t13)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t14),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));

                     assign(t16, binop(Iop_And32,
                                       unop(Iop_64HIto32,
                                            mkexpr(t9)),
                                       mkU32(0x80000000)));
                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t14),
                                                  mkU32(0)),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             mkexpr(t16),
                                                             mkU32(0)),
                                                       mkU32(0x7fffffff),
                                                       mkU32(0x80000000)),
                                            unop(Iop_64to32, mkexpr(t9))));
                     break;
                  }
                  case 0xA: {  /* EXTPDP */
                     DIP("extpdp r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     /* Extract pos field of DSPControl register. */
                     assign(t1, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));

                     /* Check if (pos - size) >= 0 [size <= pos]
                        if (pos < size)
                           put 1 to EFI field of DSPControl register
                        else
                           extract bits from acc and put 0 to EFI field of
                           DSPCtrl */
                     assign(t2, binop(Iop_CmpLT32U, mkexpr(t1), mkU32(rs)));

                     assign(t8, binop(Iop_Or32,
                                      binop(Iop_And32,
                                            getDSPControl(),
                                            mkU32(0xffffbfc0)),
                                      binop(Iop_And32,
                                            binop(Iop_Sub32,
                                                  binop(Iop_And32,
                                                        getDSPControl(),
                                                        mkU32(0x3f)),
                                                  mkU32(rs+1)),
                                            mkU32(0x3f))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                     binop(Iop_And32,
                                                           getDSPControl(),
                                                           mkU32(0xffffbfff)),
                                                     mkU32(0x4000)),
                                              mkexpr(t8)));

                     /* If pos <= 31, shift right the value from the acc
                        (pos-size) times and take (size+1) bits from the least
                        significant positions. Otherwise, shift left the value
                        (63-pos) times, take (size+1) bits from the most
                        significant positions and shift right (31-size) times.
                     */
                     assign(t3, binop(Iop_CmpLE32U, mkexpr(t1), mkU32(31)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t3),
                                       unop(Iop_32to8,
                                            binop(Iop_Sub32,
                                                  mkexpr(t1), mkU32(rs))),
                                       unop(Iop_32to8,
                                            binop(Iop_Sub32,
                                                  mkU32(63), mkexpr(t1)))));

                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Shr64,
                                                 mkexpr(t0), mkexpr(t4)),
                                           binop(Iop_Shl64,
                                                 mkexpr(t0), mkexpr(t4))));

                     /* t6 holds a mask for bit extraction. */
                     assign(t6,
                            IRExpr_ITE(mkexpr(t3),
                                       unop(Iop_Not64,
                                            binop(Iop_Shl64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  mkU8(rs+1))),
                                       unop(Iop_Not64,
                                            binop(Iop_Shr64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  mkU8(rs+1)))));

                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           unop(Iop_64to32,
                                                binop(Iop_And64,
                                                      mkexpr(t5),
                                                      mkexpr(t6))),
                                           binop(Iop_Shr32,
                                                 unop(Iop_64HIto32,
                                                      binop(Iop_And64,
                                                            mkexpr(t5),
                                                            mkexpr(t6))),
                                                 mkU8(31-rs))));

                     putIReg(rt, mkexpr(t7));
                     break;
                  }
                  case 0xB: {  /* EXTPDPV */
                     DIP("extpdpv r%u, ac%u, r%u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     assign(t8, binop(Iop_And32, getIReg(rs), mkU32(0x1f)));
                     assign(t0, getAcc(ac));
                     /* Extract pos field of DSPControl register. */
                     assign(t1, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));

                     /* Check if (pos - size) >= 0 [size <= pos]
                        if (pos < size)
                           put 1 to EFI field of DSPControl register
                        else
                           extract bits from acc and put 0 to EFI field of
                           DSPCtrl */
                     assign(t2, binop(Iop_CmpLT32U, mkexpr(t1), mkexpr(t8)));

                     assign(t9, binop(Iop_Or32,
                                      binop(Iop_And32,
                                            getDSPControl(),
                                            mkU32(0xffffbfc0)),
                                      binop(Iop_And32,
                                            binop(Iop_Sub32,
                                                  binop(Iop_And32,
                                                        getDSPControl(),
                                                        mkU32(0x3f)),
                                                  binop(Iop_Add32,
                                                        mkexpr(t8),
                                                        mkU32(0x1))),
                                            mkU32(0x3f))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    binop(Iop_And32,
                                                          getDSPControl(),
                                                          mkU32(0xffffbfff)),
                                                    mkU32(0x4000)),
                                              mkexpr(t9)));

                     /* If pos <= 31, shift right the value from the acc
                        (pos-size) times and take (size+1) bits from the least
                        significant positions. Otherwise, shift left the value
                        (63-pos) times, take (size+1) bits from the most
                        significant positions and shift right (31-size) times.
                     */
                     assign(t3, binop(Iop_CmpLE32U, mkexpr(t1), mkU32(31)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t3),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkexpr(t1), mkexpr(t8))),
                                      unop(Iop_32to8,
                                           binop(Iop_Sub32,
                                                 mkU32(63), mkexpr(t1)))));

                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Shr64,
                                                 mkexpr(t0), mkexpr(t4)),
                                           binop(Iop_Shl64,
                                                 mkexpr(t0), mkexpr(t4))));

                     /* t6 holds a mask for bit extraction. */
                     assign(t6,
                            IRExpr_ITE(mkexpr(t3),
                                       unop(Iop_Not64,
                                            binop(Iop_Shl64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  unop(Iop_32to8,
                                                       binop(Iop_Add32,
                                                             mkexpr(t8),
                                                             mkU32(1))))),
                                       unop(Iop_Not64,
                                            binop(Iop_Shr64,
                                                  mkU64(0xffffffffffffffffULL),
                                                  unop(Iop_32to8,
                                                       binop(Iop_Add32,
                                                             mkexpr(t8),
                                                             mkU32(1)))))));

                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           unop(Iop_64to32,
                                                binop(Iop_And64,
                                                      mkexpr(t5),
                                                      mkexpr(t6))),
                                           binop(Iop_Shr32,
                                                 unop(Iop_64HIto32,
                                                      binop(Iop_And64,
                                                            mkexpr(t5),
                                                            mkexpr(t6))),
                                                 unop(Iop_32to8,
                                                      binop(Iop_Sub32,
                                                            mkU32(31),
                                                            mkexpr(t8))))));

                     putIReg(rt, mkexpr(t7));
                     break;
                  }
                  case 0xE: {  /* EXTR_S.H */
                     DIP("extr_s.h r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Sar64, mkexpr(t0), mkU8(rs)));

                     assign(t2, binop(Iop_Or32,
                                      getDSPControl(), mkU32(0x00800000)));

                     assign(t9, binop(Iop_And32,
                                      unop(Iop_64to32,
                                           mkexpr(t1)),
                                      mkU32(0x80000000)));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t9),
                                                    binop(Iop_And32,
                                                          unop(Iop_64HIto32,
                                                               mkexpr(t0)),
                                                          mkU32(0x80000000))),
                                              mkexpr(t2),
                                              getDSPControl()));

                     /* Check if t1 > 0x7fff ((t1 - 0x7fff) > 0)
                        1. subtract 0x7fff from t1
                        2. if the resulting number is positive (sign bit = 0)
                           and any of the other bits is 1, the value is > 0. */
                     assign(t3, binop(Iop_Sub64,
                                      mkexpr(t1),
                                      mkU64(0x0000000000007fffULL)));
                     assign(t4, binop(Iop_And32,
                                       binop(Iop_Or32,
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       binop(Iop_And32,
                                                             unop(Iop_64HIto32,
                                                                  mkexpr(t3)),
                                                             mkU32(0x7fffffff)))),
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       unop(Iop_64to32,
                                                            mkexpr(t3))))),
                                       unop(Iop_1Sto32,
                                            binop(Iop_CmpEQ32,
                                                  binop(Iop_And32,
                                                        unop(Iop_64HIto32,
                                                                  mkexpr(t3)),
                                                             mkU32(0x80000000)),
                                                  mkU32(0)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t4)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     /* Check if t1<0xffffffffffff8000 (0xffffffffffff8000-t1)>0
                        1. subtract t1 from 0xffffffffffff8000
                        2. if the resulting number is positive (sign bit = 0)
                            and any of the other bits is 1, the value is > 0 */
                     assign(t6, binop(Iop_Sub64,
                                       mkU64(0xffffffffffff8000ULL),
                                       mkexpr(t1)));
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       binop(Iop_And32,
                                                             unop(Iop_64HIto32,
                                                                  mkexpr(t6)),
                                                             mkU32(0x7fffffff)))),
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       unop(Iop_64to32,
                                                            mkexpr(t6))))),
                                      unop(Iop_1Sto32,
                                            binop(Iop_CmpEQ32,
                                                  binop(Iop_And32,
                                                        unop(Iop_64HIto32,
                                                                  mkexpr(t6)),
                                                             mkU32(0x80000000)),
                                                  mkU32(0)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t7)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t4)),
                                            mkU32(0x00007fff),
                                            IRExpr_ITE(binop(Iop_CmpNE32,
                                                             mkU32(0),
                                                             mkexpr(t7)),
                                                       mkU32(0xffff8000),
                                                       unop(Iop_64to32,
                                                            mkexpr(t1)))));
                     break;
                  }
                  case 0xF: {  /* EXTRV_S.H */
                     DIP("extrv_s.h r%u, ac%u, %u", rt, ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Sar64,
                                      mkexpr(t0),
                                      unop(Iop_32to8,
                                           binop(Iop_And32,
                                                 getIReg(rs),
                                                 mkU32(0x1f)))));

                     assign(t2, binop(Iop_Or32,
                                      getDSPControl(), mkU32(0x00800000)));

                     assign(t9, binop(Iop_And32,
                                      unop(Iop_64to32,
                                           mkexpr(t1)),
                                      mkU32(0x80000000)));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t9),
                                                    binop(Iop_And32,
                                                          unop(Iop_64HIto32,
                                                               mkexpr(t0)),
                                                          mkU32(0x80000000))),
                                              mkexpr(t2),
                                              getDSPControl()));

                     /* Check if t1 > 0x7fff ((t1 - 0x7fff) > 0)
                        1. subtract 0x7fff from t1
                        2. if the resulting number is positive (sign bit = 0)
                           and any of the other bits is 1, the value is > 0. */
                     assign(t3, binop(Iop_Sub64,
                                      mkexpr(t1),
                                      mkU64(0x0000000000007fffULL)));
                     assign(t4, binop(Iop_And32,
                                       binop(Iop_Or32,
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       binop(Iop_And32,
                                                             unop(Iop_64HIto32,
                                                                  mkexpr(t3)),
                                                             mkU32(0x7fffffff)))),
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       unop(Iop_64to32,
                                                            mkexpr(t3))))),
                                       unop(Iop_1Sto32,
                                            binop(Iop_CmpEQ32,
                                                  binop(Iop_And32,
                                                        unop(Iop_64HIto32,
                                                                  mkexpr(t3)),
                                                             mkU32(0x80000000)),
                                                  mkU32(0)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t4)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     /* Check if t1<0xffffffffffff8000 (0xffffffffffff8000-t1)>0
                        1. subtract t1 from 0xffffffffffff8000
                        2. if the resulting number is positive (sign bit = 0)
                            and any of the other bits is 1, the value is > 0 */
                     assign(t6, binop(Iop_Sub64,
                                       mkU64(0xffffffffffff8000ULL),
                                       mkexpr(t1)));
                     assign(t7, binop(Iop_And32,
                                      binop(Iop_Or32,
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       binop(Iop_And32,
                                                             unop(Iop_64HIto32,
                                                                  mkexpr(t6)),
                                                             mkU32(0x7fffffff)))),
                                            unop(Iop_1Sto32,
                                                 binop(Iop_CmpNE32,
                                                       mkU32(0),
                                                       unop(Iop_64to32,
                                                            mkexpr(t6))))),
                                      unop(Iop_1Sto32,
                                            binop(Iop_CmpEQ32,
                                                  binop(Iop_And32,
                                                        unop(Iop_64HIto32,
                                                                  mkexpr(t6)),
                                                             mkU32(0x80000000)),
                                                  mkU32(0)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t7)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00800000)),
                                              getDSPControl()));
                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkU32(0),
                                                    mkexpr(t4)),
                                            mkU32(0x00007fff),
                                            IRExpr_ITE(binop(Iop_CmpNE32,
                                                             mkU32(0),
                                                             mkexpr(t7)),
                                                       mkU32(0xffff8000),
                                                       unop(Iop_64to32,
                                                            mkexpr(t1)))));
                     break;
                  }
                  case 0x12: {  /* RDDSP*/
                     DIP("rddsp r%u, mask 0x%x", rd, rddsp_mask);
                     vassert(!mode64);

                     putIReg(rd, mkU32(0x0));

                     if ((rddsp_mask & 0x1) == 0x1) {
                        /* Read pos field (bits 5-0) of DSPControl register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0x0000003F))));
                     }

                     if ((rddsp_mask & 0x2) == 0x2) {
                        /* Read scount field (bits 12-7) of DSPControl
                           register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0x00001F80))));
                     }

                     if ((rddsp_mask & 0x4) == 0x4) {
                        /* Read C field (bit 13) of DSPControl register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0x00002000))));
                     }

                     if ((rddsp_mask & 0x8) == 0x8) {
                        /* Read outflag field (bit s 23-16) of DSPControl
                           register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0x00FF0000))));
                     }

                     if ((rddsp_mask & 0x10) == 0x10) {
                        /* Read ccond field (bits 31-24) of DSPControl
                           register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0xFF000000))));
                     }

                     if ((rddsp_mask & 0x20) == 0x20) {
                        /* Read EFI field (bit 14) of DSPControl register. */
                        putIReg(rd, binop(Iop_Or32,
                                          getIReg(rd),
                                          binop(Iop_And32,
                                                getDSPControl(),
                                                mkU32(0x00004000))));
                     }

                     if ((rddsp_mask & 0x3f) == 0x3f) {
                        /* Read all fields of DSPControl register. */
                        putIReg(rd, getDSPControl());
                     }
                     break;
                  }
                  case 0x13: {  /* WRDSP */
                     DIP("wrdsp r%u, mask 0x%x", rs, wrdsp_mask);
                     vassert(!mode64);

                     if ((wrdsp_mask & 0x3f) == 0x3f) {
                        /* If mips64 put all fields of rs, except bit 15 and bit
                           6, to DSPControl register, otherwise put all except
                           bits 15, 6 and bits 31..28. */
                        putDSPControl(mode64 ?
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0xffff7fbf)) :
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0x0fff7fbf)));
                     } else {
                        if ((wrdsp_mask & 0x1) == 0x1) {
                           /* Put bits 5-0 of rs to DSPControl register pos
                              field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0xFFFF7F40)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mkU32(0x0000003F))));
                        }

                        if ((wrdsp_mask & 0x2) == 0x2) {
                           /* Put bits 12-7 of rs to DSPControl scount field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0xFFFFE03F)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mkU32(0x00001F80))));
                        }

                        if ((wrdsp_mask & 0x4) == 0x4) {
                           /* Put bit 13 of rs to DSPControl register C
                              field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0xFFFF5FBF)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mkU32(0x00002000))));
                        }

                        if ((wrdsp_mask & 0x8) == 0x8) {
                           /* Put bits 23-16 of rs to DSPControl reg outflag
                              field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0xFF007FBF)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mkU32(0x00FF0000))));
                        }

                        if ((wrdsp_mask & 0x10) == 0x10) {
                           /* Put bits 31-24 of rs to DSPControl reg ccond
                              field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0x00FF7FBF)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mode64 ? mkU32(0xFF000000)
                                                            : mkU32(0x0F000000))
                                               )
                                        );
                        }

                        if ((wrdsp_mask & 0x20) == 0x20) {
                           /* Put bit 14 of rs to DSPControl register EFI
                              field. */
                           putDSPControl(binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     getDSPControl(),
                                                     mkU32(0xFFFF3FBF)),
                                               binop(Iop_And32,
                                                     getIReg(rs),
                                                     mkU32(0x00004000))));
                        }
                     }
                     break;
                  }
                  case 0x1A: {  /* SHILO */
                     DIP("shilo ac%u, %u", ac, shift);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));

                     putAcc(ac, mkexpr(t0));

                     if (0x20 == (shift & 0x3f)) {
                        putAcc(ac, binop(Iop_32HLto64,
                                         unop(Iop_64to32, mkexpr(t0)),
                                         mkU32(0x0)));
                     } else if (0x20 == (shift & 0x20)) {
                        assign(t1, binop(Iop_Shl64,
                                         mkexpr(t0),
                                         unop(Iop_32to8,
                                              binop(Iop_Add32,
                                                    unop(Iop_Not32,
                                                         mkU32(shift)),
                                                    mkU32(0x1)))));

                        putAcc(ac, mkexpr(t1));
                     } else {
                        assign(t1, binop(Iop_Shr64, mkexpr(t0), mkU8(shift)));

                        putAcc(ac, mkexpr(t1));
                     }
                     break;
                  }
                  case 0x1B: {  /* SHILOV */
                     DIP("shilov ac%u, r%u", ac, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));
                     assign(t1, binop(Iop_And32, getIReg(rs), mkU32(0x3f)));
                     assign(t2, binop(Iop_CmpEQ32, mkexpr(t1), mkU32(0x20)));
                     assign(t3, binop(Iop_Shl64,
                                      mkexpr(t0),
                                      unop(Iop_32to8,
                                           binop(Iop_Add32,
                                                 unop(Iop_Not32,
                                                      mkexpr(t1)),
                                                 mkU32(0x1)))));
                     assign(t4, binop(Iop_Shr64,
                                      mkexpr(t0),
                                      unop(Iop_32to8,
                                           mkexpr(t1))));

                     putAcc(ac,
                            IRExpr_ITE(mkexpr(t2),
                                       binop(Iop_32HLto64,
                                             unop(Iop_64to32, mkexpr(t0)),
                                             mkU32(0x0)),
                                       IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        binop(Iop_And32,
                                                              mkexpr(t1),
                                                              mkU32(0x20)),
                                                        mkU32(0x20)),
                                                  mkexpr(t3),
                                                  mkexpr(t4))));
                     break;
                  }
                  case 0x1F: {  /* MTHLIP */
                     DIP("mthlip r%u, ac%u", rs, ac);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);

                     assign(t0, getAcc(ac));
                     putAcc(ac, binop(Iop_32HLto64,
                                      unop(Iop_64to32, mkexpr(t0)),
                                      getIReg(rs)));
                     assign(t1, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpLE32U,
                                                    mkU32(32),
                                                    mkexpr(t1)),
                                              binop(Iop_Or32,
                                                    binop(Iop_Sub32,
                                                          mkexpr(t1),
                                                          mkU32(32)),
                                                   binop(Iop_And32,
                                                         getDSPControl(),
                                                         mkU32(0xffffffc0))),
                                              binop(Iop_Or32,
                                                    binop(Iop_Add32,
                                                          mkexpr(t1),
                                                          mkU32(32)),
                                                    binop(Iop_And32,
                                                          getDSPControl(),
                                                          mkU32(0xffffffc0)))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of EXTR.W */
            }
            case 0xA: {  /* LX */
               switch(sa) {
                  case 0x0: {  /* LWX */
                     DIP("lwx r%u, r%u(r%u)", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Add32, getIReg(rt), getIReg(rs)));

                     putIReg(rd, load(Ity_I32, mkexpr(t0)));
                     break;
                  }
                  case 0x4: {  /* LHX */
                     DIP("lhx r%u, r%u(r%u)", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Add32, getIReg(rt), getIReg(rs)));

                     putIReg(rd, unop(Iop_16Sto32, load(Ity_I16, mkexpr(t0))));
                     break;
                  }
                  case 0x6: {  /* LBUX */
                     DIP("lbux r%u, r%u(r%u)", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Add32, getIReg(rt), getIReg(rs)));

                     putIReg(rd, unop(Iop_8Uto32, load(Ity_I8, mkexpr(t0))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of LX */
            }
            case 0xC: {  /* INSV */
               switch(sa) {
                  case 0x0: {  /* INSV */
                     DIP("insv r%u, r%u", rt, rs);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     /* t0 <- pos field of DSPControl register. */
                     assign(t0, binop(Iop_And32, getDSPControl(), mkU32(0x3f)));
                     /* t1 <- scount field of DSPControl register. */
                     assign(t1, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            getDSPControl(),
                                            mkU32(0x1f80)),
                                      mkU8(7)));

                     assign(t2, unop(Iop_32to8,
                                     binop(Iop_Add32,
                                           mkexpr(t1),
                                           mkexpr(t0))));

                     /* 32-(pos+size) most significant bits of rt. */
                     assign(t6, binop(Iop_Shl32,
                                      binop(Iop_Shr32,
                                            getIReg(rt),
                                            mkexpr(t2)),
                                      mkexpr(t2)));

                     assign(t3, unop(Iop_32to8,
                                     binop(Iop_Sub32,
                                           mkU32(32),
                                           mkexpr(t0))));
                     /* Pos least significant bits of rt. */
                     assign(t7, binop(Iop_Shr32,
                                      binop(Iop_Shl32,
                                            getIReg(rt),
                                            mkexpr(t3)),
                                      mkexpr(t3)));

                     /* Size least significant bits of rs,
                        shifted to appropriate position. */
                     assign(t8, binop(Iop_Shl32,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            unop(Iop_Not32,
                                                 binop(Iop_Shl32,
                                                       mkU32(0xffffffff),
                                                       unop(Iop_32to8,
                                                            mkexpr(t1))))),
                                      unop(Iop_32to8,
                                           mkexpr(t0))));

                     putIReg(rt, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  mkexpr(t0),
                                                  mkU32(0)),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             mkexpr(t1),
                                                             mkU32(32)),
                                                       getIReg(rs),
                                                       binop(Iop_Or32,
                                                             mkexpr(t6),
                                                             mkexpr(t8))),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             unop(Iop_8Uto32,
                                                                  mkexpr(t2)),
                                                             mkU32(32)),
                                                       binop(Iop_Or32,
                                                             mkexpr(t7),
                                                             mkexpr(t8)),
                                                       binop(Iop_Or32,
                                                             binop(Iop_Or32,
                                                                   mkexpr(t6),
                                                                   mkexpr(t7)),
                                                             mkexpr(t8)))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* enf of INSV */
            }
            case 0x10: {  /* ADDU.QB */
               switch(sa) {
                  case 0x00: {  /* ADDU.QB */
                     DIP("addu.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I32);

                     /* Add rightmost bytes of rs and rt. */
                     assign(t0,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t1 will be 1 if there is overflow, 0 otherwise. */
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t2,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t3 will be 1 if there is overflow, 0 otherwise. */
                     assign(t3, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t2),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t4,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t5 will be 1 if there is overflow, 0 otherwise. */
                     assign(t5, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t4),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t6,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t7 will be 1 if there is overflow, 0 otherwise. */
                     assign(t7, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t6),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     assign(t8,
                            binop(Iop_Or32,
                                  binop(Iop_Or32,
                                        binop(Iop_Or32,
                                              unop(Iop_1Sto32, mkexpr(t7)),
                                              unop(Iop_1Sto32,  mkexpr(t5))),
                                        unop(Iop_1Sto32, mkexpr(t3))),
                                  unop(Iop_1Sto32, mkexpr(t1))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    mkexpr(t8),
                                                    mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       binop(Iop_8HLto16,
                                             unop(Iop_32to8, mkexpr(t6)),
                                             unop(Iop_32to8, mkexpr(t4))),
                                       binop(Iop_8HLto16,
                                             unop(Iop_32to8, mkexpr(t2)),
                                             unop(Iop_32to8, mkexpr(t0)))));
                     break;
                  }
                  case 0x1: {  /* SUBU.QB */
                     DIP("subu.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I32);

                     /* Subtract rightmost bytes of rs and rt. */
                     assign(t0,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t1 will be 1 if there is overflow, 0 otherwise. */
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Subtract bits 15-8 of rs and rt. */
                     assign(t2,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t3 will be 1 if there is overflow, 0 otherwise. */
                     assign(t3, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t2),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Subtract bits 15-8 of rs and rt. */
                     assign(t4,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t5 will be 1 if there is overflow, 0 otherwise. */
                     assign(t5, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t4),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     /* Subtract bits 15-8 of rs and rt. */
                     assign(t6,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t7 will be 1 if there is overflow, 0 otherwise. */
                     assign(t7, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t6),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));

                     assign(t8, binop(Iop_Or32,
                                      binop(Iop_Or32,
                                            binop(Iop_Or32,
                                                  unop(Iop_1Sto32, mkexpr(t7)),
                                                  unop(Iop_1Sto32, mkexpr(t5))),
                                            unop(Iop_1Sto32, mkexpr(t3))),
                                      unop(Iop_1Sto32, mkexpr(t1))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                     mkexpr(t8),
                                                     mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       binop(Iop_8HLto16,
                                             unop(Iop_32to8, mkexpr(t6)),
                                             unop(Iop_32to8, mkexpr(t4))),
                                       binop(Iop_8HLto16,
                                             unop(Iop_32to8, mkexpr(t2)),
                                             unop(Iop_32to8, mkexpr(t0)))));
                     break;
                  }
                  case 0x04: {  /* ADDU_S.QB */
                     DIP("addu_s.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I8);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I8);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I8);
                     t12 = newTemp(Ity_I32);

                     /* Add rightmost bytes of rs and rt. */
                     assign(t0,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t1 will be 1 if there is overflow, 0 otherwise. */
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     /* Saturate if necessary. */
                     assign(t2, IRExpr_ITE(mkexpr(t1),
                                           mkU8(0xff),
                                           unop(Iop_32to8, mkexpr(t0))));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t3,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* t4 will be 1 if there is overflow, 0 otherwise. */
                     assign(t4, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t3),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     /* Saturate if necessary. */
                     assign(t5, IRExpr_ITE(mkexpr(t4),
                                           mkU8(0xff),
                                           unop(Iop_32to8, mkexpr(t3))));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t6,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t7 will be 1 if there is overflow, 0 otherwise. */
                     assign(t7, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t6),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     /* Saturate if necessary. */
                     assign(t8, IRExpr_ITE(mkexpr(t7),
                                           mkU8(0xff),
                                           unop(Iop_32to8, mkexpr(t6))));

                     /* Add bits 15-8 of rs and rt. */
                     assign(t9,
                            binop(Iop_Add32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     /* t10 will be 1 if there is overflow, 0 otherwise. */
                     assign(t10, binop(Iop_CmpEQ32,
                                       binop(Iop_And32,
                                             mkexpr(t9),
                                             mkU32(0x00000100)),
                                       mkU32(0x00000100)));
                     /* Saturate if necessary. */
                     assign(t11, IRExpr_ITE(mkexpr(t10),
                                            mkU8(0xff),
                                            unop(Iop_32to8, mkexpr(t9))));

                     assign(t12,
                            binop(Iop_Or32,
                                  binop(Iop_Or32,
                                        binop(Iop_Or32,
                                              unop(Iop_1Sto32, mkexpr(t10)),
                                              unop(Iop_1Sto32, mkexpr(t7))),
                                        unop(Iop_1Sto32, mkexpr(t4))),
                                  unop(Iop_1Sto32, mkexpr(t1))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    mkexpr(t12),
                                                    mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000))));

                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16, mkexpr(t11), mkexpr(t8)),
                                   binop(Iop_8HLto16, mkexpr(t5), mkexpr(t2))));
                     break;
                  }
                  case 0x05: {  /* SUBU_S.QB */
                     DIP("subu_s.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     /* Use C function to easily calculate the result
                        and write it in the register more conveniently
                        Underflow is checked using step by step subtraction. */
                     assign(t1, binop(Iop_QSub8Ux4, getIReg(rs), getIReg(rt)));

                     /* Subtract each byte of rs and rt. */
                     assign(t6,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t7,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t8,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t9,
                            binop(Iop_Sub32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));

                     /* Put 1 to bit 20 in DSPControl if there is underflow
                        in either byte. */
                     assign(t2, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t6),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     assign(t3, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t7),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     assign(t4, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t8),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     assign(t5, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            mkexpr(t9),
                                            mkU32(0x00000100)),
                                      mkU32(0x00000100)));
                     putDSPControl(IRExpr_ITE(mkexpr(t5),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     putIReg(rd, mkexpr(t1));
                     break;
                  }
                  case 0x6: {  /* MULEU_S.PH.QBL */
                     DIP("muleu_s.ph.qbl r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);

                     assign(t0,
                            unop(Iop_64to32,
                                 binop(Iop_MullU32,
                                       unop(Iop_8Uto32,
                                            unop(Iop_16HIto8,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs)))),
                                       unop(Iop_16Uto32,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t1,
                            unop(Iop_64to32,
                                 binop(Iop_MullU32,
                                       unop(Iop_8Uto32,
                                            unop(Iop_16to8,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs)))),
                                       unop(Iop_16Uto32,
                                            unop(Iop_32to16, getIReg(rt))))));

                     assign(t2, binop(Iop_CmpNE32,
                                      mkU32(0x0),
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x03ff0000))));
                     assign(t3, binop(Iop_CmpNE32,
                                      mkU32(0x0),
                                      binop(Iop_And32,
                                            mkexpr(t1),
                                            mkU32(0x03ff0000))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x200000)),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x200000)),
                                                         getDSPControl())));
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   IRExpr_ITE(mkexpr(t2),
                                              mkU16(0xffff),
                                              unop(Iop_32to16, mkexpr(t0))),
                                   IRExpr_ITE(mkexpr(t3),
                                              mkU16(0xffff),
                                              unop(Iop_32to16, mkexpr(t1)))));
                     break;
                  }
                  case 0x7: {  /* MULEU_S.PH.QBR */
                     DIP("muleu_s.ph.qbr r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);

                     assign(t0, unop(Iop_64to32,
                                     binop(Iop_MullU32,
                                           unop(Iop_8Uto32,
                                                unop(Iop_16HIto8,
                                                     unop(Iop_32to16,
                                                          getIReg(rs)))),
                                           unop(Iop_16Uto32,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t1, unop(Iop_64to32,
                                     binop(Iop_MullU32,
                                           unop(Iop_8Uto32,
                                                unop(Iop_16to8,
                                                     unop(Iop_32to16,
                                                          getIReg(rs)))),
                                           unop(Iop_16Uto32,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));

                     assign(t2, binop(Iop_CmpNE32,
                                      mkU32(0x0),
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x03ff0000))));
                     assign(t3, binop(Iop_CmpNE32,
                                      mkU32(0x0),
                                      binop(Iop_And32,
                                            mkexpr(t1),
                                            mkU32(0x03ff0000))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x200000)),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x200000)),
                                                         getDSPControl())));
                     putIReg(rd, binop(Iop_16HLto32,
                                       IRExpr_ITE(mkexpr(t2),
                                                  mkU16(0xffff),
                                                  unop(Iop_32to16,
                                                       mkexpr(t0))),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU16(0xffff),
                                                  unop(Iop_32to16,
                                                       mkexpr(t1)))));
                     break;
                  }
                  case 0x08: {  /* ADDU.PH */
                     DIP("addu.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);

                     /* Add lower halves. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Detect overflow. */
                     assign(t1, binop(Iop_CmpLT32U,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, mkexpr(t0))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs)))));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Add higher halves. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Detect overflow. */
                     assign(t3, binop(Iop_CmpLT32U,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, mkexpr(t2))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16,
                                                getIReg(rs)))));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t2)),
                                       unop(Iop_32to16, mkexpr(t0))));
                     break;
                  }
                  case 0x9: {  /* SUBU.PH */
                     DIP("subu.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);

                     /* Substract lower halves. */
                     assign(t0, binop(Iop_Sub32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Detect underflow. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            mkexpr(t0),
                                            mkU32(0x00010000)),
                                      mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Subtract higher halves. */
                     assign(t2, binop(Iop_Sub32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Detect underflow. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            mkexpr(t2),
                                            mkU32(0x00010000)),
                                      mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t2)),
                                       unop(Iop_32to16, mkexpr(t0))));
                     break;
                  }
                  case 0xA: {  /* ADDQ.PH */
                     DIP("addq.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);

                     /* Add lower halves. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t6, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t0))),
                                      mkU32(0x1)));
                     /* Detect overflow. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t0),
                                                  mkU32(0x8000)),
                                            mkU8(15)),
                                      mkexpr(t6)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Add higher halves. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t7, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x1)));
                     /* Detect overflow. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0x00008000)),
                                            mkU8(15)),
                                      mkexpr(t7)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t2)),
                                       unop(Iop_32to16, mkexpr(t0))));
                     break;
                  }
                  case 0xB: {  /* SUBQ.PH */
                     DIP("subq.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);

                     /* Subtract lower halves. */
                     assign(t0, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t6, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t0))),
                                      mkU32(0x1)));
                     /* Compare the signs of input value and the result. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t0),
                                                  mkU32(0x8000)),
                                            mkU8(15)),
                                      mkexpr(t6)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Subtract higher halves. */
                     assign(t2, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t7, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x1)));
                     /* Compare the signs of input value and the result. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0x00008000)),
                                            mkU8(15)),
                                      mkexpr(t7)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t2)),
                                       unop(Iop_32to16, mkexpr(t0))));
                     break;
                  }
                  case 0xC: {  /* ADDU_S.PH */
                     DIP("addu_s.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);

                     /* Add lower halves. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Detect overflow. */
                     assign(t1, binop(Iop_CmpLT32U,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, mkexpr(t0))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs)))));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Add higher halves. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Detect overflow. */
                     assign(t3, binop(Iop_CmpLT32U,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, mkexpr(t2))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs)))));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, binop(Iop_16HLto32,
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU16(0xffff),
                                                  unop(Iop_32to16,
                                                       mkexpr(t2))),
                                       IRExpr_ITE(mkexpr(t1),
                                                  mkU16(0xffff),
                                                  unop(Iop_32to16,
                                                       mkexpr(t0)))));
                     break;
                  }
                  case 0xD: {  /* SUBU_S.PH */
                     DIP("subu_s.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);

                     /* Subtract lower halves. */
                     assign(t0, binop(Iop_Sub32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Detect underflow. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            mkexpr(t0), mkU32(0x00010000)),
                                      mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     /* Subtract higher halves. */
                     assign(t2, binop(Iop_Sub32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Detect underflow. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            mkexpr(t2), mkU32(0x00010000)),
                                      mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   IRExpr_ITE(mkexpr(t3),
                                              mkU16(0x0000),
                                              unop(Iop_32to16, mkexpr(t2))),
                                   IRExpr_ITE(mkexpr(t1),
                                              mkU16(0x0000),
                                              unop(Iop_32to16, mkexpr(t0)))));
                     break;
                  }
                  case 0xE: {  /* ADDQ_S.PH */
                     DIP("addq_s.ph r%u r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I16);
                     t5 = newTemp(Ity_I16);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);

                     /* Add lower halves. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t6, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t0))),
                                      mkU32(0x1)));
                     /* Detect overflow. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t0),
                                                  mkU32(0x8000)),
                                            mkU8(15)),
                                      mkexpr(t6)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     /* Saturate if needed. */
                     assign(t4, IRExpr_ITE(mkexpr(t1),
                                           IRExpr_ITE(binop(Iop_CmpEQ32,
                                                            mkexpr(t6),
                                                            mkU32(0x0)),
                                                      mkU16(0x7fff),
                                                      mkU16(0x8000)),
                                           unop(Iop_32to16, mkexpr(t0))));

                     /* Add higher halves. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t7, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x1)));
                     /* Detect overflow. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0x00008000)),
                                            mkU8(15)),
                                      mkexpr(t7)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     /* Saturate if needed. */
                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           IRExpr_ITE(binop(Iop_CmpEQ32,
                                                            mkexpr(t7),
                                                            mkU32(0x0)),
                                                      mkU16(0x7fff),
                                                      mkU16(0x8000)),
                                           unop(Iop_32to16, mkexpr(t2))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t5), mkexpr(t4)));
                     break;
                  }
                  case 0xF: {  /* SUBQ_S.PH */
                     DIP("subq_s.ph r%u r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I16);
                     t5 = newTemp(Ity_I16);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);

                     /* Subtract lower halves. */
                     assign(t0, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t6, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t0))),
                                      mkU32(0x1)));
                     /* Detect overflow or underflow. */
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t0),
                                                  mkU32(0x8000)),
                                            mkU8(15)),
                                      mkexpr(t6)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     /* Saturate if needed. */
                     assign(t4, IRExpr_ITE(mkexpr(t1),
                                           IRExpr_ITE(binop(Iop_CmpEQ32,
                                                            mkexpr(t6),
                                                            mkU32(0x0)),
                                                      mkU16(0x7fff),
                                                      mkU16(0x8000)),
                                           unop(Iop_32to16, mkexpr(t0))));

                     /* Subtract higher halves. */
                     assign(t2, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));

                     /* Bit 16 of the result. */
                     assign(t7, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x1)));
                     /* Detect overflow or underflow. */
                     assign(t3, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  mkexpr(t2),
                                                  mkU32(0x00008000)),
                                            mkU8(15)),
                                      mkexpr(t7)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     /* Saturate if needed. */
                     assign(t5, IRExpr_ITE(mkexpr(t3),
                                           IRExpr_ITE(binop(Iop_CmpEQ32,
                                                            mkexpr(t7),
                                                            mkU32(0x0)),
                                                      mkU16(0x7fff),
                                                      mkU16(0x8000)),
                                           unop(Iop_32to16, mkexpr(t2))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t5), mkexpr(t4)));
                     break;
                  }
                  case 0x10: {  /* ADDSC */
                     DIP("addsc r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);

                     /* The carry bit result out of the addition operation is
                        written to bit 13(the c field) of the DSPControl reg. */
                     assign(t0, binop(Iop_Add64,
                                      unop(Iop_32Uto64, getIReg(rs)),
                                      unop(Iop_32Uto64, getIReg(rt))));

                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t0)),
                                            mkU32(0x1)),
                                      mkU32(0x1)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x2000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xffffdfff))));

                     putIReg(rd, unop(Iop_64to32, mkexpr(t0)));
                     break;
                  }
                  case 0x11: {  /* ADDWC */
                     DIP("addwc r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I1);

                     /* Get carry bit from DSPControl register. */
                     assign(t0, binop(Iop_Shr32,
                                       binop(Iop_And32,
                                             getDSPControl(),
                                             mkU32(0x2000)),
                                       mkU8(0xd)));
                     assign(t1, binop(Iop_Add64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64,
                                           binop(Iop_Add32,
                                                 getIReg(rt),
                                                 mkexpr(t0)))));

                     /* Extract bits 32 and 31. */
                     assign(t2, binop(Iop_And32,
                                      unop(Iop_64HIto32, mkexpr(t1)),
                                      mkU32(0x1)));
                     assign(t3, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            unop(Iop_64to32, mkexpr(t1)),
                                            mkU32(0x80000000)),
                                      mkU8(31)));
                     assign(t4, binop(Iop_CmpNE32, mkexpr(t2), mkexpr(t3)));

                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));
                     putIReg(rd, unop(Iop_64to32, mkexpr(t1)));
                     break;
                  }
                  case 0x12: {  /* MODSUB */
                     DIP("modsub r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);

                     /* decr_7..0 */
                     assign(t0,
                            unop(Iop_8Uto32,
                                 unop(Iop_16to8,
                                      unop(Iop_32to16, getIReg(rt)))));

                     /* lastindex_15..0 */
                     assign(t1,
                            unop(Iop_16Uto32,
                                 binop(Iop_8HLto16,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))),
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     /* temp_15..0 */
                     assign(t2,
                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                             getIReg(rs),
                                             mkU32(0x00000000)),
                                       mkexpr(t1),
                                       binop(Iop_Sub32,
                                             getIReg(rs), mkexpr(t0))));
                     putIReg(rd, mkexpr(t2));
                     break;
                  }
                  case 0x14: {  /* RADDU.W.QB */
                     DIP("raddu.w.qb r%u, r%u", rd, rs);
                     vassert(!mode64);
                     putIReg(rd, binop(Iop_Add32,
                                       binop(Iop_Add32,
                                             unop(Iop_8Uto32,
                                                  unop(Iop_16to8,
                                                       unop(Iop_32to16,
                                                            getIReg(rs)))),
                                             unop(Iop_8Uto32,
                                                  unop(Iop_16HIto8,
                                                       unop(Iop_32to16,
                                                            getIReg(rs))))),
                                       binop(Iop_Add32,
                                             unop(Iop_8Uto32,
                                                  unop(Iop_16to8,
                                                       unop(Iop_32HIto16,
                                                            getIReg(rs)))),
                                             unop(Iop_8Uto32,
                                                  unop(Iop_16HIto8,
                                                       unop(Iop_32HIto16,
                                                            getIReg(rs)))))));
                     break;
                  }
                  case 0x16: {  /* ADDQ_S.W */
                     DIP("addq_s.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Add64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));

                     assign(t3, binop(Iop_And32,
                                      unop(Iop_64HIto32, mkexpr(t0)),
                                      mkU32(0x1)));
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t0)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      mkexpr(t3)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             mkexpr(t3),
                                                             mkU32(0x0)),
                                                       mkU32(0x7fffffff),
                                                       mkU32(0x80000000)),
                                            unop(Iop_64to32, mkexpr(t0))));
                     break;
                  }
                  case 0x17: {  /* SUBQ_S.W */
                     DIP("subq_s.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Sub64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));

                     assign(t3, binop(Iop_And32,
                                      unop(Iop_64HIto32, mkexpr(t0)),
                                      mkU32(0x1)));
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t0)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      mkexpr(t3)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00100000)),
                                              getDSPControl()));

                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                                             mkexpr(t3),
                                                             mkU32(0x0)),
                                                       mkU32(0x7fffffff),
                                                       mkU32(0x80000000)),
                                            unop(Iop_64to32, mkexpr(t0))));
                     break;
                  }
                  case 0x1C: {  /* MULEQ_S.W.PHL */
                     DIP("muleq_s.w.phl r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I32);

                     assign(t0,
                            binop(Iop_Shl32,
                                  binop(Iop_Mul32,
                                        unop(Iop_16Sto32,
                                             unop(Iop_32HIto16, getIReg(rt))),
                                        unop(Iop_16Sto32,
                                             unop(Iop_32HIto16, getIReg(rs)))),
                                  mkU8(0x1)));
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0xffff0000)),
                                      mkU32(0x80000000)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0xffff0000)),
                                      mkU32(0x80000000)));
                     assign(t3, IRExpr_ITE(mkexpr(t1),
                                           IRExpr_ITE(mkexpr(t2),
                                                      binop(Iop_Or32,
                                                            getDSPControl(),
                                                            mkU32(0x00200000)),
                                                      getDSPControl()),
                                           getDSPControl()));
                     putDSPControl(mkexpr(t3));

                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(mkexpr(t2),
                                                       mkU32(0x7fffffff),
                                                       mkexpr(t0)),
                                            mkexpr(t0)));
                     break;
                  }
                  case 0x1D: {  /* MULEQ_S.W.PHR */
                     DIP("muleq_s.w.phr r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t0,
                            binop(Iop_Shl32,
                                  binop(Iop_Mul32,
                                        unop(Iop_16Sto32,
                                             unop(Iop_32to16, getIReg(rt))),
                                        unop(Iop_16Sto32,
                                             unop(Iop_32to16, getIReg(rs)))),
                                  mkU8(0x1)));
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0xffff)),
                                      mkU32(0x8000)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0xffff)),
                                      mkU32(0x8000)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              IRExpr_ITE(mkexpr(t2),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(mkexpr(t2),
                                                       mkU32(0x7fffffff),
                                                       mkexpr(t0)),
                                            mkexpr(t0)));
                     break;
                  }
                  case 0x1E: {  /* MULQ_S.PH */
                     DIP("mulq_s.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I16);
                     t3 = newTemp(Ity_I16);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t5,
                            unop(Iop_16Sto32, unop(Iop_32to16, getIReg(rs))));
                     assign(t6,
                            unop(Iop_16Sto32, unop(Iop_32to16, getIReg(rt))));

                     assign(t7,
                            unop(Iop_16Sto32, unop(Iop_32HIto16, getIReg(rs))));
                     assign(t8,
                            unop(Iop_16Sto32, unop(Iop_32HIto16, getIReg(rt))));

                     assign(t0, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t5),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000))),
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t6),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000)))));
                     assign(t1, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t7),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000))),
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t8),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000)))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    binop(Iop_Or32,
                                                          mkexpr(t0),
                                                          mkexpr(t1)),
                                                    mkU32(0x0)),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x200000))));

                     assign(t2, unop(Iop_32HIto16,
                                     binop(Iop_Shl32,
                                           unop(Iop_64to32,
                                                binop(Iop_MullS32,
                                                      mkexpr(t7),
                                                      mkexpr(t8))),
                                           mkU8(0x1))));
                     assign(t3, unop(Iop_32HIto16,
                                     binop(Iop_Shl32,
                                           unop(Iop_64to32,
                                                binop(Iop_MullS32,
                                                      mkexpr(t5),
                                                      mkexpr(t6))),
                                           mkU8(0x1))));
                     putIReg(rd, binop(Iop_16HLto32,
                                       IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        mkexpr(t1),
                                                        mkU32(0x0)),
                                                  mkexpr(t2),
                                                  mkU16(0x7fff)),
                                       IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        mkexpr(t0),
                                                        mkU32(0x0)),
                                                  mkexpr(t3),
                                                  mkU16(0x7fff))));
                     break;
                  }
                  case 0x1F: {  /* MULQ_RS.PH */
                     DIP("mulq_rs.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I16);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I16);

                     /* Multiply and round lower halfwords. */
                     assign(t0, binop(Iop_Add32,
                                      binop(Iop_Shl32,
                                            binop(Iop_Mul32,
                                                  unop(Iop_16Sto32,
                                                       unop(Iop_32to16,
                                                            getIReg(rt))),
                                                  unop(Iop_16Sto32,
                                                       unop(Iop_32to16,
                                                            getIReg(rs)))),
                                            mkU8(0x1)),
                                      mkU32(0x00008000)));
                     assign(t1, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rt), mkU32(0xffff)),
                                      mkU32(0x8000)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rs), mkU32(0xffff)),
                                      mkU32(0x8000)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              IRExpr_ITE(mkexpr(t2),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     assign(t3, IRExpr_ITE(mkexpr(t1),
                                           IRExpr_ITE(mkexpr(t2),
                                                      mkU16(0x7fff),
                                                      unop(Iop_32HIto16,
                                                           mkexpr(t0))),
                                           unop(Iop_32HIto16, mkexpr(t0))));

                     /* Multiply and round higher halfwords. */
                     assign(t4, binop(Iop_Add32,
                                      binop(Iop_Shl32,
                                            binop(Iop_Mul32,
                                                  unop(Iop_16Sto32,
                                                       unop(Iop_32HIto16,
                                                            getIReg(rt))),
                                                  unop(Iop_16Sto32,
                                                       unop(Iop_32HIto16,
                                                            getIReg(rs)))),
                                            mkU8(0x1)),
                                      mkU32(0x00008000)));
                     assign(t5, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0xffff0000)),
                                      mkU32(0x80000000)));
                     assign(t6, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0xffff0000)),
                                      mkU32(0x80000000)));
                     putDSPControl(IRExpr_ITE(mkexpr(t5),
                                             IRExpr_ITE(mkexpr(t6),
                                                        binop(Iop_Or32,
                                                             getDSPControl(),
                                                             mkU32(0x00200000)),
                                                        getDSPControl()),
                                             getDSPControl()));
                     assign(t7, IRExpr_ITE(mkexpr(t5),
                                           IRExpr_ITE(mkexpr(t6),
                                                      mkU16(0x7fff),
                                                      unop(Iop_32HIto16,
                                                           mkexpr(t4))),
                                           unop(Iop_32HIto16, mkexpr(t4))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t7), mkexpr(t3)));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of ADDU.QB */
            }
            case 0x11: {  /* CMPU.EQ.QB */
               switch(sa) {
                  case 0x0: {  /* CMPU.EQ.QB */
                     DIP("cmpu.eq.qb r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);

                     assign(t1,
                            binop(Iop_CmpEQ32,
                                  binop(Iop_And32, getIReg(rs), mkU32(0xff)),
                                  binop(Iop_And32, getIReg(rt), mkU32(0xff))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));
                     break;
                  }
                  case 0x1: {  /* CMPU.LT.QB */
                     DIP("cmpu.lt.qb r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);

                     assign(t1, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));
                     break;
                  }
                  case 0x2: {  /* CMPU.LE.QB */
                     DIP("cmpu.le.qb r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);

                     assign(t1, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));
                     break;
                  }
                  case 0x3: {  /* PICK.QB */
                     DIP("pick.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I8);

                     assign(t0, getDSPControl());
                     assign(t1, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x01000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_16to8,
                                                 unop(Iop_32to16,
                                                      getIReg(rs))),
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt)))));
                     assign(t2, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x02000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs))),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt)))));
                     assign(t3, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x04000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs))),
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt)))));
                     assign(t4, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x08000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs))),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt)))));
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16, mkexpr(t4), mkexpr(t3)),
                                   binop(Iop_8HLto16, mkexpr(t2), mkexpr(t1))));
                     break;
                  }
                  case 0x4: {  /* CMPGU.EQ.QB */
                     DIP("cmpgu.eq.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001), mkU32(0)));

                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));

                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));

                     assign(t4, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  case 0x5: {  /* CMPGU.LT.QB */
                     DIP("cmpgu.lt.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001), mkU32(0)));

                     assign(t2, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));

                     assign(t3, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));

                     assign(t4, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));
                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  case 0x6: {  /* CMPGU.LE.QB */
                     DIP("cmpgu.le.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001), mkU32(0)));

                     assign(t2, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));

                     assign(t3, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));

                     assign(t4, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));
                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  case 0x8: {  /* CMP.EQ.PH */
                     DIP("cmp.eq.ph r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t1, binop(Iop_CmpEQ16,
                                      unop(Iop_32to16, getIReg(rs)),
                                      unop(Iop_32to16, getIReg(rt))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));
                     assign(t2, binop(Iop_CmpEQ16,
                                      unop(Iop_32HIto16, getIReg(rs)),
                                      unop(Iop_32HIto16, getIReg(rt))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));
                     break;
                  }
                  case 0x9: {  /* CMP.LT.PH */
                     DIP("cmp.lt.ph r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t1, binop(Iop_CmpLT32S,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLT32S,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));
                     break;
                  }
                  case 0xA: {  /* CMP.LE.PH */
                     DIP("cmp.le.ph r%u, r%u", rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t1, binop(Iop_CmpLE32S,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLE32S,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));
                     break;
                  }
                  case 0xB: {  /* PICK.PH */
                     DIP("pick.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I16);
                     t2 = newTemp(Ity_I16);

                     assign(t0, getDSPControl());

                     assign(t1, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x01000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_32to16, getIReg(rs)),
                                           unop(Iop_32to16, getIReg(rt))));

                     assign(t2, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0x02000000)),
                                                 mkU32(0x0)),
                                           unop(Iop_32HIto16, getIReg(rs)),
                                           unop(Iop_32HIto16, getIReg(rt))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t2), mkexpr(t1)));
                     break;
                  }
                  case 0xC: {  /* PRECRQ.QB.PH */
                     DIP("precrq.qb.ph r%u, r%u, %u", rd, rs, rt);
                     vassert(!mode64);
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16,
                                         unop(Iop_16HIto8,
                                              unop(Iop_32HIto16, getIReg(rs))),
                                         unop(Iop_16HIto8,
                                              unop(Iop_32to16, getIReg(rs)))),
                                   binop(Iop_8HLto16,
                                         unop(Iop_16HIto8,
                                              unop(Iop_32HIto16, getIReg(rt))),
                                         unop(Iop_16HIto8,
                                              unop(Iop_32to16, getIReg(rt))))));
                     break;
                  }
                  case 0xD: {  /* PRECR.QB.PH */
                     DIP("precr.qb.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);

                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16,
                                         unop(Iop_16to8,
                                              unop(Iop_32HIto16, getIReg(rs))),
                                         unop(Iop_16to8,
                                              unop(Iop_32to16, getIReg(rs)))),
                                   binop(Iop_8HLto16,
                                         unop(Iop_16to8,
                                              unop(Iop_32HIto16, getIReg(rt))),
                                         unop(Iop_16to8,
                                              unop(Iop_32to16, getIReg(rt))))));
                     break;
                  }
                  case 0xF: {  /* PRECRQU_S.QB.PH */
                     DIP("precrqu_s.qb.ph r%u, r%u, %u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I8);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I8);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I8);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I8);
                     t11 = newTemp(Ity_I1);
                     t12 = newTemp(Ity_I32);
                     t13 = newTemp(Ity_I8);
                     t14 = newTemp(Ity_I1);
                     t15 = newTemp(Ity_I32);

                     assign(t4, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                 mkU32(0x7f80),
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32to16,
                                                            getIReg(rs))),
                                                       mkU32(0x7fff))),
                                           mkU8(0xff),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     binop(Iop_Shl32,
                                                           getIReg(rs),
                                                           mkU8(1))))));
                     assign(t0, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32to16,
                                                                 getIReg(rs))),
                                                       mkU32(0x00008000)),
                                                 mkU32(0x0)),
                                           mkexpr(t4),
                                           mkU8(0x0)));
                     assign(t5, binop(Iop_And32,
                                      unop(Iop_16Uto32,
                                            unop(Iop_32to16,
                                                 getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t6, binop(Iop_CmpLT32U,
                                      mkU32(0x7f80),
                                      binop(Iop_And32,
                                            unop(Iop_16Uto32,
                                                 unop(Iop_32to16,
                                                 getIReg(rs))),
                                            mkU32(0x7fff))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    mkexpr(t5),
                                                    mkU32(0x0)),
                                              IRExpr_ITE(mkexpr(t6),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00400000)
                                                              ),
                                                         getDSPControl()),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00400000))));

                     assign(t7, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                 mkU32(0x7f80),
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32HIto16,
                                                                 getIReg(rs))),
                                                       mkU32(0x7fff))),
                                           mkU8(0xff),
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     binop(Iop_Shl32,
                                                           getIReg(rs),
                                                           mkU8(1))))));
                     assign(t1, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32HIto16,
                                                                 getIReg(rs))),
                                                       mkU32(0x00008000)),
                                                 mkU32(0x0)),
                                           mkexpr(t7),
                                           mkU8(0x0)));
                     assign(t8, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            unop(Iop_16Uto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs))),
                                            mkU32(0x00008000)),
                                      mkU32(0x0)));
                     assign(t9, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                 mkU32(0x7f80),
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32HIto16,
                                                                 getIReg(rs))),
                                                       mkU32(0x7fff))),
                                           binop(Iop_Or32,
                                                 getDSPControl(),
                                                 mkU32(0x00400000)),
                                           getDSPControl()));
                     putDSPControl(IRExpr_ITE(mkexpr(t8),
                                              mkexpr(t9),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00400000))));

                     assign(t10, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                  mkU32(0x7f80),
                                                  binop(Iop_And32,
                                                        unop(Iop_16Uto32,
                                                             unop(Iop_32to16,
                                                             getIReg(rt))),
                                                        mkU32(0x7fff))),
                                            mkU8(0xff),
                                            unop(Iop_16HIto8,
                                                 unop(Iop_32to16,
                                                      binop(Iop_Shl32,
                                                            getIReg(rt),
                                                            mkU8(1))))));
                     assign(t2, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32to16,
                                                                 getIReg(rt))),
                                                       mkU32(0x00008000)),
                                                 mkU32(0x0)),
                                           mkexpr(t10),
                                           mkU8(0x0)));
                     assign(t11, binop(Iop_CmpEQ32,
                                       binop(Iop_And32,
                                             unop(Iop_16Uto32,
                                                  unop(Iop_32to16,
                                                       getIReg(rt))),
                                             mkU32(0x00008000)),
                                       mkU32(0x0)));
                     assign(t12, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                  mkU32(0x7f80),
                                                  binop(Iop_And32,
                                                        unop(Iop_16Uto32,
                                                             unop(Iop_32to16,
                                                             getIReg(rt))),
                                                        mkU32(0x7fff))),
                                            binop(Iop_Or32,
                                                  getDSPControl(),
                                                  mkU32(0x00400000)),
                                            getDSPControl()));
                     putDSPControl(IRExpr_ITE(mkexpr(t11),
                                              mkexpr(t12),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00400000))));

                     assign(t13, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                  mkU32(0x7f80),
                                                  binop(Iop_And32,
                                                        unop(Iop_16Uto32,
                                                             unop(Iop_32HIto16,
                                                                  getIReg(rt))),
                                                        mkU32(0x7fff))),
                                            mkU8(0xff),
                                            unop(Iop_16HIto8,
                                                 unop(Iop_32HIto16,
                                                      binop(Iop_Shl32,
                                                            getIReg(rt),
                                                            mkU8(1))))));
                     assign(t3, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       unop(Iop_16Uto32,
                                                            unop(Iop_32HIto16,
                                                                 getIReg(rt))),
                                                       mkU32(0x00008000)),
                                                 mkU32(0x0)),
                                           mkexpr(t13),
                                           mkU8(0x0)));
                     assign(t14, binop(Iop_CmpEQ32,
                                       binop(Iop_And32,
                                             unop(Iop_16Uto32,
                                                  unop(Iop_32HIto16,
                                                       getIReg(rt))),
                                             mkU32(0x00008000)),
                                       mkU32(0x0)));
                     assign(t15, IRExpr_ITE(binop(Iop_CmpLT32U,
                                                  mkU32(0x7f80),
                                                  binop(Iop_And32,
                                                        unop(Iop_16Uto32,
                                                             unop(Iop_32HIto16,
                                                                  getIReg(rt))),
                                                        mkU32(0x7fff))),
                                            binop(Iop_Or32,
                                                  getDSPControl(),
                                                  mkU32(0x00400000)),
                                            getDSPControl()));
                     putDSPControl(IRExpr_ITE(mkexpr(t14),
                                              mkexpr(t15),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00400000))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       binop(Iop_8HLto16,
                                             mkexpr(t1), mkexpr(t0)),
                                       binop(Iop_8HLto16,
                                             mkexpr(t3), mkexpr(t2))));
                     break;
                  }
                  case 0x14: {  /* PRECRQ.PH.W */
                     DIP("precrq.ph.w r%u, r%u, %u", rd, rs, rt);
                     vassert(!mode64);
                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32HIto16, getIReg(rs)),
                                       unop(Iop_32HIto16, getIReg(rt))));
                     break;
                  }
                  case 0x15: {  /* PRECRQ_RS.PH.W */
                     DIP("precrq_rs.ph.w r%u, r%u, %u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_Add64,
                                      binop(Iop_32HLto64,
                                            binop(Iop_Shr32,
                                                  binop(Iop_And32,
                                                        getIReg(rs),
                                                        mkU32(0x80000000)),
                                                  mkU8(31)),
                                            getIReg(rs)),
                                      mkU64(0x0000000000008000ULL)));
                     assign(t1, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t0)),
                                            mkU32(0x1)),
                                      binop(Iop_And32,
                                            binop(Iop_Shr32,
                                                  unop(Iop_64to32, mkexpr(t0)),
                                                  mkU8(31)),
                                            mkU32(0x1))));
                     assign(t2, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x7fffffff),
                                           unop(Iop_64to32, mkexpr(t0))));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              getDSPControl()));
                     assign(t3, binop(Iop_Add64,
                                      binop(Iop_32HLto64,
                                            binop(Iop_Shr32,
                                                  binop(Iop_And32,
                                                        getIReg(rt),
                                                        mkU32(0x80000000)),
                                                  mkU8(31)),
                                            getIReg(rt)),
                                      mkU64(0x0000000000008000ULL)));
                     assign(t4, binop(Iop_CmpNE32,
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t3)),
                                            mkU32(0x1)),
                                      binop(Iop_And32,
                                            binop(Iop_Shr32,
                                                  unop(Iop_64to32, mkexpr(t3)),
                                                  mkU8(31)),
                                            mkU32(0x1))));
                     assign(t5, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x7fffffff),
                                           unop(Iop_64to32, mkexpr(t3))));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              getDSPControl()));
                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32HIto16, mkexpr(t2)),
                                       unop(Iop_32HIto16, mkexpr(t5))));
                     break;
                  }
                  case 0x1E: {  /* PRECR_SRA.PH.W */
                     DIP("precr_sra.ph.w r%u, r%u, %u", rt, rs, rd);
                     vassert(!mode64);

                     if (0 == rd) {
                        putIReg(rt, binop(Iop_16HLto32,
                                          unop(Iop_32to16, getIReg(rt)),
                                          unop(Iop_32to16, getIReg(rs))));
                     } else {
                        putIReg(rt, binop(Iop_16HLto32,
                                          unop(Iop_32to16, binop(Iop_Sar32,
                                                                 getIReg(rt),
                                                                 mkU8(rd))),
                                          unop(Iop_32to16, binop(Iop_Sar32,
                                                                 getIReg(rs),
                                                                 mkU8(rd)))));
                     }
                     break;
                  }
                  case 0x1F: {  /* PRECR_SRA_R.PH.W */
                     DIP("precr_sra_r.ph.w r%u, r%u, %u", rt, rs, rd);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);

                     if (0 == rd) {
                        putIReg(rt, binop(Iop_16HLto32,
                                          unop(Iop_32to16, getIReg(rt)),
                                          unop(Iop_32to16, getIReg(rs))));
                     } else {
                        assign(t0, binop(Iop_Shr32,
                                         binop(Iop_Add32,
                                               binop(Iop_Sar32,
                                                     getIReg(rt),
                                                     mkU8(rd-1)),
                                               mkU32(0x1)),
                                         mkU8(0x1)));
                        assign(t1, binop(Iop_Shr32,
                                         binop(Iop_Add32,
                                               binop(Iop_Sar32,
                                                     getIReg(rs),
                                                     mkU8(rd-1)),
                                               mkU32(0x1)),
                                         mkU8(0x1)));
                        putIReg(rt, binop(Iop_16HLto32,
                                          unop(Iop_32to16, mkexpr(t0)),
                                          unop(Iop_32to16, mkexpr(t1))));
                     };
                     break;
                  }
                  case 0xE: {  /* PACKRL.PH */
                     DIP("packrl.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, getIReg(rs)),
                                       unop(Iop_32HIto16, getIReg(rt))));
                     break;
                  }
                  case 0x18: {  /* CMPGDU.EQ.QB */
                     DIP("cmpgdu.eq.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1,
                            binop(Iop_CmpEQ32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpEQ32,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  case 0x19: {  /* CMPGDU.LT.QB */
                     DIP("cmpgdu.lt.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpLT32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  case 0x1A: {  /* CMPGDU.LE.QB */
                     DIP("cmpgdu.le.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           mkU32(0x00000001),
                                           mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x01000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfeffffff))));

                     assign(t2, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16, getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32to16,
                                                     getIReg(rt))))));
                     assign(t6, IRExpr_ITE(mkexpr(t2),
                                           mkU32(0x00000002), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x02000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfdffffff))));

                     assign(t3, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16to8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t7, IRExpr_ITE(mkexpr(t3),
                                           mkU32(0x00000004), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x04000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xfbffffff))));

                     assign(t4, binop(Iop_CmpLE32U,
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rs)))),
                                      unop(Iop_8Uto32,
                                           unop(Iop_16HIto8,
                                                unop(Iop_32HIto16,
                                                     getIReg(rt))))));
                     assign(t8, IRExpr_ITE(mkexpr(t4),
                                           mkU32(0x00000008), mkU32(0)));
                     putDSPControl(IRExpr_ITE(mkexpr(t4),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x08000000)),
                                              binop(Iop_And32,
                                                    getDSPControl(),
                                                    mkU32(0xf7ffffff))));

                     putIReg(rd, binop(Iop_Or32,
                                       binop(Iop_Or32,
                                             binop(Iop_Or32,
                                                   mkexpr(t5), mkexpr(t6)),
                                             mkexpr(t7)),
                                       mkexpr(t8)));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of CMPU.EQ.QB */
            }
            case 0x13: {  /* SHLL.QB */
               switch(sa) {
                  case 0x0: {  /* SHLL.QB */
                     DIP("shll.qb r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I1);
                     t10 = newTemp(Ity_I1);

                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        /* Shift bits 7..0 and 23..16. */
                        assign(t0, binop(Iop_Shl32,
                                         binop(Iop_And32,
                                               getIReg(rt),
                                               mkU32(0x00ff00ff)),
                                         mkU8(rs)));
                        assign(t1, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t0),
                                              mkU32(0xff000000)),
                                        mkU32(0x00000000)));
                        assign(t2, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t0),
                                              mkU32(0xff000000)),
                                        mkU32(0xff000000)));
                        assign(t7, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t0),
                                              mkU32(0x0000ff00)),
                                        mkU32(0x00000000)));
                        assign(t8, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t0),
                                              mkU32(0x0000ff00)),
                                        mkU32(0x000ff00)));
                        /* Shift bits 15..8 and 31..24. */
                        assign(t3, binop(Iop_Shl32,
                                         binop(Iop_Shr32,
                                               binop(Iop_And32,
                                                     getIReg(rt),
                                                     mkU32(0xff00ff00)),
                                               mkU8(8)),
                                         mkU8(rs)));
                        assign(t4, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t3),
                                              mkU32(0xff000000)),
                                        mkU32(0x00000000)));
                        assign(t5, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t3),
                                              mkU32(0xff000000)),
                                        mkU32(0xff000000)));
                        assign(t9, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t3),
                                              mkU32(0x0000ff00)),
                                        mkU32(0x00000000)));
                        assign(t10, binop(Iop_CmpNE32,
                                        binop(Iop_And32,
                                              mkexpr(t3),
                                              mkU32(0x0000ff00)),
                                        mkU32(0x0000ff00)));

                        assign(t6, binop(Iop_Or32,
                                         binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t1)),
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t2))),
                                               binop(Iop_And32,
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t7)),
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t8)))),
                                         binop(Iop_Or32,
                                               binop(Iop_And32,
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t4)),
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t5))),
                                               binop(Iop_And32,
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t9)),
                                                     unop(Iop_1Uto32,
                                                          mkexpr(t10))))));

                        putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                       mkexpr(t6),
                                                       mkU32(0x0)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putIReg(rd, binop(Iop_Or32,
                                          binop(Iop_Shl32,
                                                binop(Iop_And32,
                                                      mkexpr(t3),
                                                      mkU32(0x00ff00ff)),
                                                mkU8(8)),
                                          binop(Iop_And32,
                                                mkexpr(t0),
                                                mkU32(0x00ff00ff))));
                     }
                     break;
                  }
                  case 0x3: {  /* SHRL.QB */
                     DIP("shrl.qb r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I8);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I8);
                     t9 = newTemp(Ity_I32);

                     assign(t9, binop(Iop_And32, getIReg(rs), mkU32(0x7)));
                     assign(t0, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t1, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           mkexpr(t0),
                                           unop(Iop_32to8, mkexpr(t9)))));

                     assign(t2, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t3, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           mkexpr(t2),
                                           unop(Iop_32to8, mkexpr(t9)))));

                     assign(t4, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t5, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           mkexpr(t4),
                                           unop(Iop_32to8, mkexpr(t9)))));

                     assign(t6, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t7, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           mkexpr(t6),
                                           unop(Iop_32to8, mkexpr(t9)))));
                     putIReg(rd, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  mkexpr(t9),
                                                  mkU32(0x0)),
                                            getIReg(rt),
                                            binop(Iop_16HLto32,
                                                  binop(Iop_8HLto16,
                                                        mkexpr(t7),
                                                        mkexpr(t5)),
                                                  binop(Iop_8HLto16,
                                                        mkexpr(t3),
                                                        mkexpr(t1)))));
                     break;
                  }
                  case 0x2: {  /* SHLLV.QB */
                     DIP("shllv.qb r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I1);
                     t10 = newTemp(Ity_I1);
                     t11 = newTemp(Ity_I8);

                     assign(t11, unop(Iop_32to8,
                                      binop(Iop_And32,
                                            getIReg(rs),
                                            mkU32(0x7))));
                     /* Shift bits 7..0 and 23..16. */
                     assign(t0, binop(Iop_Shl32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0x00ff00ff)),
                                      mkexpr(t11)));
                     assign(t1, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t0),
                                           mkU32(0xff000000)),
                                     mkU32(0x00000000)));
                     assign(t2, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t0),
                                           mkU32(0xff000000)),
                                     mkU32(0xff000000)));
                     assign(t7, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t0),
                                           mkU32(0x0000ff00)),
                                     mkU32(0x00000000)));
                     assign(t8, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t0),
                                           mkU32(0x0000ff00)),
                                     mkU32(0x000ff00)));
                     /* Shift bits 15..8 and 31..24. */
                     assign(t3, binop(Iop_Shl32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0xff00ff00)),
                                            mkU8(8)),
                                      mkexpr(t11)));
                     assign(t4, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t3),
                                           mkU32(0xff000000)),
                                     mkU32(0x00000000)));
                     assign(t5, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t3),
                                           mkU32(0xff000000)),
                                     mkU32(0xff000000)));
                     assign(t9, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t3),
                                           mkU32(0x0000ff00)),
                                     mkU32(0x00000000)));
                     assign(t10, binop(Iop_CmpNE32,
                                     binop(Iop_And32,
                                           mkexpr(t3),
                                           mkU32(0x0000ff00)),
                                     mkU32(0x0000ff00)));

                     assign(t6, binop(Iop_Or32,
                                      binop(Iop_Or32,
                                            binop(Iop_And32,
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t1)),
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t2))),
                                            binop(Iop_And32,
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t7)),
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t8)))),
                                      binop(Iop_Or32,
                                            binop(Iop_And32,
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t4)),
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t5))),
                                            binop(Iop_And32,
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t9)),
                                                  unop(Iop_1Uto32,
                                                       mkexpr(t10))))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t6),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              getDSPControl()));
                     putIReg(rd, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  unop(Iop_8Uto32, mkexpr(t11)),
                                                  mkU32(0)),
                                            getIReg(rt),
                                            binop(Iop_Or32,
                                                  binop(Iop_Shl32,
                                                        binop(Iop_And32,
                                                              mkexpr(t3),
                                                              mkU32(0xff00ff)),
                                                        mkU8(8)),
                                                  binop(Iop_And32,
                                                        mkexpr(t0),
                                                        mkU32(0x00ff00ff)))));
                     break;
                  }
                  case 0x1: {  /* SHRLV.QB */
                     DIP("shrlv.qb r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I8);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I8);

                     assign(t0, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           unop(Iop_8Uto32,
                                                unop(Iop_32to8, getIReg(rt))),
                                           mkU8(rs))));
                     assign(t1, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           unop(Iop_8Uto32,
                                                unop(Iop_16HIto8,
                                                     unop(Iop_32to16,
                                                          getIReg(rt)))),
                                           mkU8(rs))));
                     assign(t2, unop(Iop_32to8,
                                      binop(Iop_Shr32,
                                            unop(Iop_8Uto32,
                                                 unop(Iop_16to8,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rt)))),
                                            mkU8(rs))));
                     assign(t3, unop(Iop_32to8,
                                     binop(Iop_Shr32,
                                           unop(Iop_8Uto32,
                                                unop(Iop_16HIto8,
                                                     unop(Iop_32HIto16,
                                                          getIReg(rt)))),
                                           mkU8(rs))));
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16, mkexpr(t3), mkexpr(t2)),
                                   binop(Iop_8HLto16, mkexpr(t1), mkexpr(t0))));
                     break;
                  }
                  case 0x4: {  /* SHRA.QB */
                     DIP("shra.qb r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I32);

                     /* ========== GPR[rt]_31..24 ========== */
                     assign(t1,
                            unop(Iop_8Uto32,
                                 unop(Iop_16HIto8,
                                      unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t2,
                            binop(Iop_Shr32, mkexpr(t1), mkU8(rs)));
                     /* tempD_7..0 */
                     assign(t0,
                            binop(Iop_Or32,
                                  mkexpr(t2),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t1),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8, mkU8(0x8), mkU8(rs)))));

                     /* ========== GPR[rt]_23..16 ========== */
                     assign(t4,
                            unop(Iop_8Uto32,
                                 unop(Iop_16to8,
                                      unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t5, binop(Iop_Shr32, mkexpr(t4), mkU8(rs)));
                     /* tempC_7..0 */
                     assign(t3,
                            binop(Iop_Or32,
                                  mkexpr(t5),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t4),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8, mkU8(0x8), mkU8(rs)))));

                     /* ========== GPR[rt]_15..8 ========== */
                     assign(t7,
                            unop(Iop_8Uto32,
                                 unop(Iop_16HIto8,
                                      unop(Iop_32to16, getIReg(rt)))));
                     assign(t8, binop(Iop_Shr32, mkexpr(t7), mkU8(rs)));
                     /* tempB_7..0 */
                     assign(t6,
                            binop(Iop_Or32,
                                  mkexpr(t8),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t7),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8, mkU8(0x8), mkU8(rs)))));

                     /* ========== GPR[rt]_7..0 ========== */
                     assign(t10,
                            unop(Iop_8Uto32,
                                 unop(Iop_16to8,
                                      unop(Iop_32to16, getIReg(rt)))));
                     assign(t11, binop(Iop_Shr32, mkexpr(t10), mkU8(rs)));
                     /* tempB_7..0 */
                     assign(t9,
                            binop(Iop_Or32,
                                  mkexpr(t11),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t10),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8, mkU8(0x8), mkU8(rs)))));

                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16,
                                         unop(Iop_32to8, mkexpr(t0)),
                                         unop(Iop_32to8, mkexpr(t3))),
                                   binop(Iop_8HLto16,
                                         unop(Iop_32to8, mkexpr(t6)),
                                         unop(Iop_32to8, mkexpr(t9)))));
                     break;
                  }
                  case 0x5: {  /* SHRA_R.QB */
                     DIP("shra_r.qb r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I8);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I8);

                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        assign(t0, unop(Iop_8Sto32,
                                        unop(Iop_16to8,
                                             unop(Iop_32to16, getIReg(rt)))));
                        assign(t1, unop(Iop_32to8,
                                        binop(Iop_Sar32,
                                              binop(Iop_Add32,
                                                    mkexpr(t0),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(rs-1))),
                                              mkU8(rs))));

                        assign(t2, unop(Iop_8Sto32,
                                        unop(Iop_16HIto8,
                                             unop(Iop_32to16, getIReg(rt)))));
                        assign(t3, unop(Iop_32to8,
                                        binop(Iop_Sar32,
                                              binop(Iop_Add32,
                                                    mkexpr(t2),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(rs-1))),
                                              mkU8(rs))));

                        assign(t4, unop(Iop_8Sto32,
                                        unop(Iop_16to8,
                                             unop(Iop_32HIto16, getIReg(rt)))));
                        assign(t5, unop(Iop_32to8,
                                        binop(Iop_Sar32,
                                              binop(Iop_Add32,
                                                    mkexpr(t4),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(rs-1))),
                                              mkU8(rs))));

                        assign(t6, unop(Iop_8Sto32,
                                        unop(Iop_16HIto8,
                                             unop(Iop_32HIto16, getIReg(rt)))));
                        assign(t7, unop(Iop_32to8,
                                        binop(Iop_Sar32,
                                              binop(Iop_Add32,
                                                    mkexpr(t6),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(rs-1))),
                                              mkU8(rs))));
                        putIReg(rd, binop(Iop_16HLto32,
                                         binop(Iop_8HLto16,
                                               mkexpr(t7), mkexpr(t5)),
                                         binop(Iop_8HLto16,
                                               mkexpr(t3), mkexpr(t1))));
                     }
                     break;
                  }
                  case 0x6: {  /* SHRAV.QB */
                     DIP("shrav.qb r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);

                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);

                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);

                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I32);

                     /* ========== GPR[rt]_31..24 ========== */
                     assign(t1,
                            unop(Iop_8Uto32,
                                 unop(Iop_16HIto8,
                                      unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t2,
                            binop(Iop_Shr32,
                                  mkexpr(t1),
                                  unop(Iop_32to8, binop(Iop_And32,
                                                        getIReg(rs),
                                                        mkU32(0x7)))));
                     /* tempD_7..0 */
                     assign(t0,
                            binop(Iop_Or32,
                                  mkexpr(t2),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t1),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8,
                                              mkU8(0x8),
                                              unop(Iop_32to8, binop(Iop_And32,
                                                                    getIReg(rs),
                                                                    mkU32(0x7)))
                                              ))));

                     /* ========== GPR[rt]_23..16 ========== */
                     assign(t4,
                            unop(Iop_8Uto32,
                                 unop(Iop_16to8,
                                      unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t5,
                            binop(Iop_Shr32,
                                  mkexpr(t4),
                                  unop(Iop_32to8, binop(Iop_And32,
                                                        getIReg(rs),
                                                        mkU32(0x7)))));
                     /* tempC_7..0 */
                     assign(t3,
                            binop(Iop_Or32,
                                  mkexpr(t5),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t4),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8,
                                              mkU8(0x8),
                                              unop(Iop_32to8, binop(Iop_And32,
                                                                    getIReg(rs),
                                                                    mkU32(0x7)))
                                              ))));

                     /* ========== GPR[rt]_15..8 ========== */
                     assign(t7,
                            unop(Iop_8Uto32,
                                 unop(Iop_16HIto8,
                                      unop(Iop_32to16, getIReg(rt)))));
                     assign(t8,
                            binop(Iop_Shr32,
                                  mkexpr(t7),
                                  unop(Iop_32to8, binop(Iop_And32,
                                                        getIReg(rs),
                                                        mkU32(0x7)))));
                     /* tempB_7..0 */
                     assign(t6,
                            binop(Iop_Or32,
                                  mkexpr(t8),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t7),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8,
                                              mkU8(0x8),
                                              unop(Iop_32to8, binop(Iop_And32,
                                                                    getIReg(rs),
                                                                    mkU32(0x7)))
                                              ))));

                     /* ========== GPR[rt]_7..0 ========== */
                     assign(t10,
                            unop(Iop_8Uto32,
                                 unop(Iop_16to8,
                                      unop(Iop_32to16, getIReg(rt)))));
                     assign(t11,
                            binop(Iop_Shr32,
                                  mkexpr(t10),
                                  unop(Iop_32to8, binop(Iop_And32,
                                                        getIReg(rs),
                                                        mkU32(0x7)))));
                     /* tempB_7..0 */
                     assign(t9,
                            binop(Iop_Or32,
                                  mkexpr(t11),
                                  binop(Iop_Shl32,
                                        IRExpr_ITE(binop(Iop_CmpEQ32,
                                                         binop(Iop_And32,
                                                               mkexpr(t10),
                                                               mkU32(0x00000080)
                                                              ),
                                                         mkU32(0x00000080)),
                                                   mkU32(0xFFFFFFFF),
                                                   mkU32(0x00000000)),
                                        binop(Iop_Sub8,
                                              mkU8(0x8),
                                              unop(Iop_32to8, binop(Iop_And32,
                                                                    getIReg(rs),
                                                                    mkU32(0x7)))
                                              ))));

                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   binop(Iop_8HLto16,
                                         unop(Iop_32to8,
                                              IRExpr_ITE(binop(Iop_CmpEQ32,
                                                               binop(Iop_And32,
                                                                     mkU32(rs),
                                                                     mkU32(0x7)
                                                                    ),
                                                               mkU32(0x0)),
                                                         mkexpr(t1),
                                                         mkexpr(t0))),
                                         unop(Iop_32to8,
                                              IRExpr_ITE(binop(Iop_CmpEQ32,
                                                               binop(Iop_And32,
                                                                     mkU32(rs),
                                                                     mkU32(0x7)
                                                                    ),
                                                               mkU32(0x0)),
                                                         mkexpr(t2),
                                                         mkexpr(t3)))),
                                   binop(Iop_8HLto16,
                                         unop(Iop_32to8,
                                              IRExpr_ITE(binop(Iop_CmpEQ32,
                                                               binop(Iop_And32,
                                                                     mkU32(rs),
                                                                     mkU32(0x7)
                                                                    ),
                                                               mkU32(0x0)),
                                                         mkexpr(t5),
                                                         mkexpr(t6))),
                                         unop(Iop_32to8,
                                              IRExpr_ITE(binop(Iop_CmpEQ32,
                                                               binop(Iop_And32,
                                                                     mkU32(rs),
                                                                     mkU32(0x7)
                                                                    ),
                                                               mkU32(0x0)),
                                                         mkexpr(t8),
                                                         mkexpr(t9))))));
                     break;
                  }
                  case 0x7: {  /* SHRAV_R.QB */
                     DIP("shrav_r.qb r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I8);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I8);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I8);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I8);
                     t8 = newTemp(Ity_I8);
                     t9 = newTemp(Ity_I32);

                     assign(t9, binop(Iop_And32, getIReg(rs), mkU32(0x7)));
                     assign(t8, unop(Iop_32to8,
                                     binop(Iop_Sub32, mkexpr(t9), mkU32(0x1))));
                     assign(t0, unop(Iop_8Sto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t1, unop(Iop_32to8,
                                     binop(Iop_Sar32,
                                           binop(Iop_Add32,
                                                 mkexpr(t0),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkexpr(t8))),
                                           unop(Iop_32to8,
                                                mkexpr(t9)))));

                     assign(t2, unop(Iop_8Sto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t3, unop(Iop_32to8,
                                     binop(Iop_Sar32,
                                           binop(Iop_Add32,
                                                 mkexpr(t2),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkexpr(t8))),
                                           unop(Iop_32to8, mkexpr(t9)))));

                     assign(t4, unop(Iop_8Sto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t5, unop(Iop_32to8,
                                     binop(Iop_Sar32,
                                           binop(Iop_Add32,
                                                 mkexpr(t4),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkexpr(t8))),
                                           unop(Iop_32to8, mkexpr(t9)))));

                     assign(t6, unop(Iop_8Sto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t7, unop(Iop_32to8,
                                     binop(Iop_Sar32,
                                           binop(Iop_Add32,
                                                 mkexpr(t6),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkexpr(t8))),
                                           unop(Iop_32to8, mkexpr(t9)))));
                     putIReg(rd, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  mkexpr(t9),
                                                  mkU32(0x0)),
                                            getIReg(rt),
                                            binop(Iop_16HLto32,
                                                  binop(Iop_8HLto16,
                                                        mkexpr(t7),
                                                        mkexpr(t5)),
                                                  binop(Iop_8HLto16,
                                                        mkexpr(t3),
                                                        mkexpr(t1)))));
                     break;
                  }
                  case 0x8: {  /* SHLL.PH */
                     DIP("shll.ph r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);

                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        /* Shift lower 16 bits. */
                        assign(t0, binop(Iop_Shl32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32to16, getIReg(rt))),
                                         mkU8(rs)));

                        assign(t1, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                               binop(Iop_Sar32,
                                                     mkexpr(t0),
                                                     mkU8(16)),
                                               mkU32(0))));
                        assign(t2, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                              binop(Iop_Sar32,
                                                    mkexpr(t0),
                                                    mkU8(16)),
                                              mkU32(0xffffffff))));
                        assign(t3, binop(Iop_And32,
                                         mkexpr(t1),
                                         mkexpr(t2)));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t3),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       binop(Iop_And32,
                                                             getIReg(rt),
                                                             mkU32(0x00008000)),
                                                       binop(Iop_And32,
                                                             mkexpr(t0),
                                                             mkU32(0x00008000))
                                                      ),
                                                 getDSPControl(),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000))));
                        /* Shift higher 16 bits. */
                        assign(t4, binop(Iop_Shl32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32HIto16, getIReg(rt))),
                                         mkU8(rs)));

                        assign(t5, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                               binop(Iop_Sar32,
                                                     mkexpr(t4),
                                                     mkU8(16)),
                                               mkU32(0))));
                        assign(t6, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                              binop(Iop_Sar32,
                                                    mkexpr(t4),
                                                    mkU8(16)),
                                              mkU32(0xffffffff))));
                        assign(t7, binop(Iop_And32,
                                         mkexpr(t5),
                                         mkexpr(t6)));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t7),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t7),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       binop(Iop_And32,
                                                             getIReg(rt),
                                                             mkU32(0x80000000)),
                                                       binop(Iop_Shl32,
                                                             binop(Iop_And32,
                                                                   mkexpr(t4),
                                                                   mkU32(0x00008000)),
                                                             mkU8(16))
                                                      ),
                                                 getDSPControl(),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000))));
                        putIReg(rd, binop(Iop_16HLto32,
                                          unop(Iop_32to16, mkexpr(t4)),
                                          unop(Iop_32to16, mkexpr(t0))));
                     }
                     break;
                  }
                  case 0x9: {  /* SHRA.PH */
                     DIP("shra.ph r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        assign(t0, binop(Iop_Sar32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32to16, getIReg(rt))),
                                         mkU8(rs)));
                        assign(t1, binop(Iop_Sar32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32HIto16, getIReg(rt))),
                                         mkU8(rs)));
                        putIReg(rd, binop(Iop_16HLto32,
                                          unop(Iop_32to16, mkexpr(t1)),
                                          unop(Iop_32to16, mkexpr(t0))));
                     }
                     break;
                  }
                  case 0xA: {  /* SHLLV.PH */
                     DIP("shllv.ph r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I32);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);

                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x0f)));

                     /* Shift lower 16 bits. */
                     assign(t2, binop(Iop_Shl32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x00000000)));
                     assign(t4, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0xffffffff)));
                     assign(t10, binop(Iop_And32,
                                       unop(Iop_1Sto32, mkexpr(t3)),
                                       unop(Iop_1Sto32, mkexpr(t4))));
                     assign(t5, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0x00008000)),
                                      mkU8(15)));
                     assign(t12, binop(Iop_CmpEQ32,
                                       mkexpr(t5),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   mkexpr(t2),
                                                   mkU32(0x00008000)),
                                             mkU8(15))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t10),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              IRExpr_ITE(mkexpr(t12),
                                                         getDSPControl(),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x400000)))
                                             ));
                     /* Shift higher 16 bits. */
                     assign(t6, binop(Iop_Shl32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     assign(t7, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t6))),
                                      mkU32(0x00000000)));
                     assign(t8, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t6))),
                                      mkU32(0xffffffff)));
                     assign(t11, binop(Iop_And32,
                                       unop(Iop_1Sto32, mkexpr(t7)),
                                       unop(Iop_1Sto32, mkexpr(t8))));

                     assign(t9, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0x80000000)),
                                      mkU8(31)));
                     assign(t13, binop(Iop_CmpEQ32,
                                       mkexpr(t9),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   mkexpr(t6),
                                                   mkU32(0x00008000)),
                                             mkU8(15))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t11),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              IRExpr_ITE(mkexpr(t13),
                                                         getDSPControl(),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x400000)))
                                             ));

                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t6)),
                                       unop(Iop_32to16, mkexpr(t2))));
                     break;
                  }
                  case 0xB: {  /* SHRAV.PH */
                     DIP("shrav.ph r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x0f)));
                     assign(t1, binop(Iop_CmpEQ32, mkexpr(t0), mkU32(0x0)));
                     assign(t2, binop(Iop_Sar32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));
                     assign(t3, binop(Iop_Sar32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));
                     putIReg(rd,
                             binop(Iop_16HLto32,
                                   IRExpr_ITE(mkexpr(t1),
                                              unop(Iop_32HIto16, getIReg(rt)),
                                              unop(Iop_32to16, mkexpr(t3))),
                                   IRExpr_ITE(mkexpr(t1),
                                              unop(Iop_32to16, getIReg(rt)),
                                              unop(Iop_32to16, mkexpr(t2)))));
                     break;
                  }
                  case 0xC: {  /* SHLL_S.PH */
                     DIP("shll_s.ph r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I32);
                     t12 = newTemp(Ity_I32);
                     t13 = newTemp(Ity_I32);
                     t14 = newTemp(Ity_I32);

                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        /* Shift lower 16 bits. */
                        assign(t0, binop(Iop_Shl32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32to16, getIReg(rt))),
                                         mkU8(rs)));

                        assign(t1, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                               binop(Iop_Sar32,
                                                     mkexpr(t0),
                                                     mkU8(16)),
                                               mkU32(0))));
                        assign(t2, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                              binop(Iop_Sar32,
                                                    mkexpr(t0),
                                                    mkU8(16)),
                                              mkU32(0xffffffff))));
                        assign(t3, binop(Iop_And32,
                                         mkexpr(t1),
                                         mkexpr(t2)));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t3),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       binop(Iop_And32,
                                                             getIReg(rt),
                                                             mkU32(0x00008000)),
                                                       binop(Iop_And32,
                                                             mkexpr(t0),
                                                             mkU32(0x00008000))
                                                      ),
                                                 getDSPControl(),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000))));
                        assign(t8,
                               IRExpr_ITE(binop(Iop_CmpEQ32,
                                                mkexpr(t3),
                                                mkU32(0x1)),
                                          IRExpr_ITE(binop(Iop_CmpEQ32,
                                                           binop(Iop_And32,
                                                                 getIReg(rt),
                                                                 mkU32(0x8000)),
                                                           mkU32(0)),
                                                     mkU32(0x00007fff),
                                                     mkU32(0x00008000)),
                                          binop(Iop_And32,
                                                mkexpr(t0),
                                                mkU32(0x0000ffff))));
                        assign(t10,
                               IRExpr_ITE(binop(Iop_CmpEQ32,
                                                binop(Iop_And32,
                                                      getIReg(rt),
                                                      mkU32(0x00008000)),
                                                binop(Iop_And32,
                                                      mkexpr(t0),
                                                      mkU32(0x00008000))),
                                          mkexpr(t8),
                                          IRExpr_ITE(binop(Iop_CmpEQ32,
                                                           binop(Iop_And32,
                                                                 getIReg(rt),
                                                                 mkU32(0x8000)),
                                                           mkU32(0)),
                                                     mkU32(0x00007fff),
                                                     mkU32(0x00008000))));
                        /* Shift higher 16 bits. */
                        assign(t4, binop(Iop_Shl32,
                                         unop(Iop_16Sto32,
                                              unop(Iop_32HIto16, getIReg(rt))),
                                         mkU8(rs)));

                        assign(t5, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                               binop(Iop_Sar32,
                                                     mkexpr(t4),
                                                     mkU8(16)),
                                               mkU32(0))));
                        assign(t6, unop(Iop_1Uto32,
                                        binop(Iop_CmpNE32,
                                              binop(Iop_Sar32,
                                                    mkexpr(t4),
                                                    mkU8(16)),
                                              mkU32(0xffffffff))));
                        assign(t7, binop(Iop_And32,
                                         mkexpr(t5),
                                         mkexpr(t6)));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t7),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       mkexpr(t7),
                                                       mkU32(0x1)),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000)),
                                                 getDSPControl()));
                        assign(t12, binop(Iop_Shl32,
                                          binop(Iop_And32,
                                                mkexpr(t4),
                                                mkU32(0x8000)),
                                          mkU8(16)));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       binop(Iop_And32,
                                                             getIReg(rt),
                                                             mkU32(0x80000000)),
                                                       mkexpr(t12)),
                                                 getDSPControl(),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000))));
                        assign(t13, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                     binop(Iop_And32,
                                                           getIReg(rt),
                                                           mkU32(0x80000000)),
                                                     mkU32(0)),
                                               mkU32(0x7fff0000),
                                               mkU32(0x80000000)));
                        assign(t9,
                               IRExpr_ITE(binop(Iop_CmpEQ32,
                                                mkexpr(t7),
                                                mkU32(0x1)),
                                          mkexpr(t13),
                                          binop(Iop_Shl32,
                                                binop(Iop_And32,
                                                      mkexpr(t4),
                                                      mkU32(0x0000ffff)),
                                                mkU8(16))));
                        assign(t14, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                     binop(Iop_And32,
                                                           getIReg(rt),
                                                           mkU32(0x80000000)),
                                                     mkU32(0)),
                                               mkU32(0x7fff0000),
                                               mkU32(0x80000000)));
                        assign(t11,
                               IRExpr_ITE(binop(Iop_CmpEQ32,
                                                binop(Iop_And32,
                                                      getIReg(rt),
                                                      mkU32(0x80000000)),
                                                binop(Iop_Shl32,
                                                      binop(Iop_And32,
                                                            mkexpr(t4),
                                                            mkU32(0x00008000)),
                                                      mkU8(16))),
                                          mkexpr(t9),
                                          mkexpr(t14)));
                        putIReg(rd, binop(Iop_Or32,
                                          mkexpr(t10),
                                          mkexpr(t11)));
                     }
                     break;
                  }
                  case 0xD: {  /* SHRA_R.PH */
                     DIP("shra.ph r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        assign(t0, binop(Iop_Sar32,
                                         binop(Iop_Add32,
                                               unop(Iop_16Sto32,
                                                    unop(Iop_32to16,
                                                         getIReg(rt))),
                                               binop(Iop_Shl32,
                                                     mkU32(0x1),
                                                     mkU8(rs-1))),
                                         mkU8(rs)));
                        assign(t1, binop(Iop_Sar32,
                                         binop(Iop_Add32,
                                               unop(Iop_16Sto32,
                                                    unop(Iop_32HIto16,
                                                         getIReg(rt))),
                                               binop(Iop_Shl32,
                                                     mkU32(0x1),
                                                     mkU8(rs-1))),
                                         mkU8(rs)));
                        putIReg(rd, binop(Iop_16HLto32,
                                          unop(Iop_32to16, mkexpr(t1)),
                                          unop(Iop_32to16, mkexpr(t0))));
                     }
                     break;
                  }
                  case 0xE: {  /* SHLLV_S.PH */
                     DIP("shllv_s.ph r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I32);
                     t12 = newTemp(Ity_I1);
                     t13 = newTemp(Ity_I1);
                     t14 = newTemp(Ity_I16);
                     t15 = newTemp(Ity_I16);
                     t16 = newTemp(Ity_I16);
                     t17 = newTemp(Ity_I16);

                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x0f)));

                     /* Shift lower 16 bits. */
                     assign(t2, binop(Iop_Shl32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     assign(t3, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0x00000000)));
                     assign(t4, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t2))),
                                      mkU32(0xffffffff)));
                     assign(t10, binop(Iop_And32,
                                       unop(Iop_1Sto32, mkexpr(t3)),
                                       unop(Iop_1Sto32, mkexpr(t4))));
                     assign(t5, binop(Iop_Shr32,
                                       binop(Iop_And32,
                                             getIReg(rt),
                                             mkU32(0x00008000)),
                                       mkU8(15)));
                     assign(t12, binop(Iop_CmpEQ32,
                                       mkexpr(t5),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   mkexpr(t2),
                                                   mkU32(0x00008000)),
                                             mkU8(15))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t10),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              IRExpr_ITE(mkexpr(t12),
                                                         getDSPControl(),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x400000)))
                                             ));
                     assign(t14, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t5),
                                                  mkU32(0x0)),
                                            mkU16(0x8000),
                                            mkU16(0x7fff)));
                     assign(t15, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t10),
                                                  mkU32(0x0)),
                                            mkexpr(t14),
                                            IRExpr_ITE(mkexpr(t12),
                                                       unop(Iop_32to16,
                                                            mkexpr(t2)),
                                                       mkexpr(t14))));
                     /* Shift higher 16 bits. */
                     assign(t6, binop(Iop_Shl32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     assign(t7, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t6))),
                                      mkU32(0x00000000)));
                     assign(t8, binop(Iop_CmpNE32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, mkexpr(t6))),
                                      mkU32(0xffffffff)));
                     assign(t11, binop(Iop_And32,
                                       unop(Iop_1Sto32, mkexpr(t7)),
                                       unop(Iop_1Sto32, mkexpr(t8))));

                     assign(t9, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0x80000000)),
                                      mkU8(31)));
                     assign(t13, binop(Iop_CmpEQ32,
                                       mkexpr(t9),
                                       binop(Iop_Shr32,
                                             binop(Iop_And32,
                                                   mkexpr(t6),
                                                   mkU32(0x00008000)),
                                             mkU8(15))));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t11),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              IRExpr_ITE(mkexpr(t13),
                                                         getDSPControl(),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x400000)))
                                             ));

                     assign(t16, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t9),
                                                  mkU32(0x0)),
                                            mkU16(0x8000),
                                            mkU16(0x7fff)));
                     assign(t17, IRExpr_ITE(binop(Iop_CmpNE32,
                                                  mkexpr(t11),
                                                  mkU32(0x0)),
                                            mkexpr(t16),
                                            IRExpr_ITE(mkexpr(t13),
                                                       unop(Iop_32to16,
                                                            mkexpr(t6)),
                                                       mkexpr(t16))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t17), mkexpr(t15)));
                     break;
                  }
                  case 0xF: {  /* SHRAV_R.PH */
                     DIP("shrav_r.ph r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x0f)));
                     assign(t1, binop(Iop_CmpEQ32, mkexpr(t0), mkU32(0x0)));
                     assign(t2, unop(Iop_32to8,
                                     binop(Iop_Sub32, mkexpr(t0), mkU32(1))));

                     assign(t3, binop(Iop_Sar32,
                                      binop(Iop_Add32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16, getIReg(rt))),
                                            binop(Iop_Shl32,
                                                  mkU32(0x1),
                                                  mkexpr(t2))),
                                      unop(Iop_32to8, mkexpr(t0))));
                     assign(t4, binop(Iop_Sar32,
                                      binop(Iop_Add32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rt))),
                                            binop(Iop_Shl32,
                                                  mkU32(0x1),
                                                  mkexpr(t2))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       IRExpr_ITE(mkexpr(t1),
                                                  unop(Iop_32HIto16,
                                                       getIReg(rt)),
                                                  unop(Iop_32to16,
                                                       mkexpr(t4))),
                                       IRExpr_ITE(mkexpr(t1),
                                                  unop(Iop_32to16, getIReg(rt)),
                                                  unop(Iop_32to16,
                                                       mkexpr(t3)))));
                     break;
                  }
                  case 0x14: {  /* SHLL_S.W */
                     DIP("shll_s.w r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);

                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        /* t0-bits that will be discarded, sign extended to
                           32bits. */
                        assign(t0, binop(Iop_Sar32,
                                         binop(Iop_And32,
                                               getIReg(rt),
                                               binop(Iop_Sar32,
                                                     mkU32(0x80000000),
                                                     mkU8(rs-1))),
                                         mkU8(32-rs)));

                        assign(t1, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    binop(Iop_And32,
                                                          getIReg(rt),
                                                          mkU32(0x80000000)),
                                                    mkU32(0x0)),
                                              mkU32(0x7fffffff),
                                              mkU32(0x80000000)));

                        assign(t2, binop(Iop_Shl32, getIReg(rt), mkU8(rs)));
                        assign(t3, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    binop(Iop_And32,
                                                          getIReg(rt),
                                                          mkU32(0x80000000)),
                                                    binop(Iop_And32,
                                                          mkexpr(t2),
                                                          mkU32(0x80000000))),
                                              mkexpr(t2),
                                              mkexpr(t1)));

                        assign(t4, IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t0),
                                                    mkU32(0x0)),
                                              IRExpr_ITE(binop(Iop_CmpNE32,
                                                               mkexpr(t0),
                                                               mkU32(0xffffffff)
                                                              ),
                                                         mkexpr(t1),
                                                         mkexpr(t3)),
                                              mkexpr(t3)));
                        assign(t5, IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t0),
                                                    mkU32(0xffffffff)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x400000)),
                                              getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                       mkexpr(t0),
                                                       mkU32(0x0)),
                                                 mkexpr(t5),
                                                 getDSPControl()));
                        putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                       binop(Iop_And32,
                                                             getIReg(rt),
                                                             mkU32(0x80000000)),
                                                       binop(Iop_And32,
                                                             mkexpr(t2),
                                                             mkU32(0x80000000))
                                                            ),
                                                 getDSPControl(),
                                                 binop(Iop_Or32,
                                                       getDSPControl(),
                                                       mkU32(0x400000))));
                        putIReg(rd, mkexpr(t4));
                     }
                     break;
                  }
                  case 0x15: {  /* SHRA_R.W */
                     DIP("shra_r.w r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     if (0 == rs) {
                        putIReg(rd, getIReg(rt));
                     } else {
                        putIReg(rd, binop(Iop_Add32,
                                          binop(Iop_Sar32,
                                                getIReg(rt), mkU8(rs)),
                                          binop(Iop_Shr32,
                                                binop(Iop_And32,
                                                      getIReg(rt),
                                                      binop(Iop_Shl32,
                                                            mkU32(0x1),
                                                            mkU8(rs-1))),
                                                mkU8(rs-1))));
                     }
                     break;
                  }
                  case 0x16: {  /* SHLLV_S.W */
                     DIP("shllv_s.w r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I1);
                     t5 = newTemp(Ity_I1);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I32);

                     /* Check if shift amount is zero. */
                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x1f)));
                     assign(t1, binop(Iop_CmpEQ32, mkexpr(t0), mkU32(0x0)));

                     /* t2 = sign of the input value. */
                     assign(t2, binop(Iop_Shr32,
                                      binop(Iop_And32,
                                            getIReg(rt),
                                            mkU32(0x80000000)),
                                      mkU8(31)));
                     /* Shift left input value and check for overflow. */
                     assign(t3, binop(Iop_Shl64,
                                      unop(Iop_32Sto64, getIReg(rt)),
                                      unop(Iop_32to8, mkexpr(t0))));
                     assign(t4, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32, mkexpr(t3)),
                                      mkU32(0x00000000)));
                     assign(t5, binop(Iop_CmpNE32,
                                      unop(Iop_64HIto32, mkexpr(t3)),
                                      mkU32(0xffffffff)));
                     assign(t6, binop(Iop_And32,
                                      unop(Iop_1Uto32, mkexpr(t4)),
                                      unop(Iop_1Uto32, mkexpr(t5))));
                     assign(t7, binop(Iop_CmpEQ32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  getIReg(rt),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t3)),
                                                  mkU32(0x80000000)),
                                            mkU8(31))));

                     putDSPControl(IRExpr_ITE(unop(Iop_32to1, mkexpr(t6)),
                                                   binop(Iop_Or32,
                                                         getDSPControl(),
                                                         mkU32(0x400000)),
                                              IRExpr_ITE(mkexpr(t7),
                                                         getDSPControl(),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x400000)))
                                             ));

                     assign(t8, IRExpr_ITE(unop(Iop_32to1,
                                                mkexpr(t2)),
                                           mkU32(0x80000000),
                                           mkU32(0x7fffffff)));
                     putIReg(rd, IRExpr_ITE(unop(Iop_32to1, mkexpr(t6)),
                                            IRExpr_ITE(unop(Iop_32to1,
                                                            mkexpr(t2)),
                                                       mkU32(0x80000000),
                                                       mkU32(0x7fffffff)),
                                            IRExpr_ITE(mkexpr(t7),
                                                       unop(Iop_64to32,
                                                            mkexpr(t3)),
                                                       mkexpr(t8))));
                     break;
                  }
                  case 0x17: {  /* SHRAV_R.W */
                     DIP("shrav_r.w r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x1f)));
                     assign(t1, binop(Iop_CmpEQ32, mkexpr(t0), mkU32(0x0)));
                     assign(t2, unop(Iop_32to8,
                                     binop(Iop_Sub32, mkexpr(t0), mkU32(1))));

                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            getIReg(rt),
                                            binop(Iop_Sar32,
                                                  binop(Iop_Add32,
                                                        binop(Iop_Sar32,
                                                              getIReg(rt),
                                                              mkexpr(t2)),
                                                        mkU32(0x1)),
                                                  mkU8(1))));
                     break;
                  }
                  case 0x19: {  /* SHRL.PH */
                     DIP("shrl.ph r%u, r%u, %u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     assign(t0, binop(Iop_Shr32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU8(rs)));
                     assign(t1, binop(Iop_Shr32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU8(rs)));
                     putIReg(rd, binop(Iop_16HLto32,
                                       unop(Iop_32to16, mkexpr(t1)),
                                       unop(Iop_32to16, mkexpr(t0))));
                     break;
                  }
                  case 0x1B: {  /* SHRLV.PH */
                     DIP("shrlv.ph r%u, r%u, r%u", rd, rt, rs);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I16);
                     t5 = newTemp(Ity_I16);

                     /* Get shift amount from lower 5 bits of rs
                        and check if it is zero. */
                     assign(t0, binop(Iop_And32, getIReg(rs), mkU32(0x0f)));
                     assign(t1, binop(Iop_CmpEQ32, mkexpr(t0), mkU32(0x0)));

                     assign(t2, binop(Iop_Shr32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));
                     assign(t3, binop(Iop_Shr32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      unop(Iop_32to8, mkexpr(t0))));

                     assign(t4, IRExpr_ITE(mkexpr(t1),
                                           unop(Iop_32HIto16, getIReg(rt)),
                                           unop(Iop_32to16, mkexpr(t3))));
                     assign(t5, IRExpr_ITE(mkexpr(t1),
                                           unop(Iop_32to16, getIReg(rt)),
                                           unop(Iop_32to16, mkexpr(t2))));
                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t4), mkexpr(t5)));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of SHLL.QB */
            }
            case 0x18: {  /* ADDUH.QB/MUL.PH */
               switch(sa) {
                  case 0x00: {  /* ADDUH.QB */
                     DIP("adduh.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_HAdd8Ux4, getIReg(rs), getIReg(rt)));

                     putIReg(rd, mkexpr(t0));
                     break;
                  }
                  case 0x1: {  /* SUBUH.QB */
                     DIP("subuh.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);

                     assign(t0, binop(Iop_HSub8Ux4, getIReg(rs), getIReg(rt)));

                     putIReg(rd, mkexpr(t0));
                     break;
                  }
                  case 0x02: {  /* ADDUH_R.QB */
                     DIP("adduh_r.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I8);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I8);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I8);
                     t9 = newTemp(Ity_I32);
                     t10 = newTemp(Ity_I32);
                     t11 = newTemp(Ity_I8);

                     /* Extract input bytes, add values, add 1 and half the
                        result. */
                     assign(t0, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rs)))));
                     assign(t1, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t2, unop(Iop_16to8,
                                     unop(Iop_32to16,
                                          binop(Iop_Shr32,
                                                binop(Iop_Add32,
                                                      binop(Iop_Add32,
                                                            mkexpr(t0),
                                                            mkexpr(t1)),
                                                      mkU32(0x00000001)),
                                                mkU8(0x01)))));

                     assign(t3, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rs)))));
                     assign(t4, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t5, unop(Iop_16to8,
                                     unop(Iop_32to16,
                                          binop(Iop_Shr32,
                                                binop(Iop_Add32,
                                                      binop(Iop_Add32,
                                                            mkexpr(t3),
                                                            mkexpr(t4)),
                                                      mkU32(0x00000001)),
                                                mkU8(0x01)))));

                     assign(t6, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rs)))));
                     assign(t7, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t8, unop(Iop_16to8,
                                     unop(Iop_32to16,
                                          binop(Iop_Shr32,
                                                binop(Iop_Add32,
                                                      binop(Iop_Add32,
                                                            mkexpr(t7),
                                                            mkexpr(t6)),
                                                      mkU32(0x00000001)),
                                                mkU8(0x01)))));

                     assign(t9, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32HIto16, getIReg(rs)))));
                     assign(t10, unop(Iop_8Uto32,
                                      unop(Iop_16HIto8,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t11, unop(Iop_16to8,
                                      unop(Iop_32to16,
                                           binop(Iop_Shr32,
                                                 binop(Iop_Add32,
                                                       binop(Iop_Add32,
                                                             mkexpr(t9),
                                                             mkexpr(t10)),
                                                       mkU32(0x00000001)),
                                                 mkU8(0x01)))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       binop(Iop_8HLto16,
                                             mkexpr(t11), mkexpr(t8)),
                                       binop(Iop_8HLto16,
                                             mkexpr(t5), mkexpr(t2))));
                     break;
                  }
                  case 0x3: {  /* SUBUH_R.QB */
                     DIP("subuh_r.qb r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I32);
                     t7 = newTemp(Ity_I32);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I8);
                     t10 = newTemp(Ity_I8);
                     t11 = newTemp(Ity_I8);
                     t12 = newTemp(Ity_I8);

                     /* Extract each byte of rs and rt. */
                     assign(t1, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rs)))));
                     assign(t2, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rs)))));
                     assign(t3, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rs)))));
                     assign(t4, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32HIto16, getIReg(rs)))));

                     assign(t5, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t6, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32to16, getIReg(rt)))));
                     assign(t7, unop(Iop_8Uto32,
                                     unop(Iop_16to8,
                                          unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t8, unop(Iop_8Uto32,
                                     unop(Iop_16HIto8,
                                          unop(Iop_32HIto16, getIReg(rt)))));

                     /* Add 1 to each resulting byte and half the results. */
                     assign(t9, unop(Iop_16to8,
                                     unop(Iop_32to16,
                                          binop(Iop_Shr32,
                                                binop(Iop_Add32,
                                                      binop(Iop_Sub32,
                                                            mkexpr(t1),
                                                            mkexpr(t5)),
                                                      mkU32(0x00000001)),
                                                mkU8(0x01)))));
                     assign(t10, unop(Iop_16to8,
                                      unop(Iop_32to16,
                                           binop(Iop_Shr32,
                                                 binop(Iop_Add32,
                                                       binop(Iop_Sub32,
                                                             mkexpr(t2),
                                                             mkexpr(t6)),
                                                       mkU32(0x00000001)),
                                                 mkU8(0x01)))));
                     assign(t11, unop(Iop_16to8,
                                      unop(Iop_32to16,
                                            binop(Iop_Shr32,
                                                  binop(Iop_Add32,
                                                        binop(Iop_Sub32,
                                                              mkexpr(t3),
                                                              mkexpr(t7)),
                                                        mkU32(0x00000001)),
                                                  mkU8(0x01)))));
                     assign(t12, unop(Iop_16to8,
                                      unop(Iop_32to16,
                                           binop(Iop_Shr32,
                                                 binop(Iop_Add32,
                                                       binop(Iop_Sub32,
                                                             mkexpr(t4),
                                                             mkexpr(t8)),
                                                       mkU32(0x00000001)),
                                                 mkU8(0x01)))));

                     putIReg(rd, binop(Iop_16HLto32,
                                       binop(Iop_8HLto16,
                                             mkexpr(t12), mkexpr(t11)),
                                       binop(Iop_8HLto16,
                                             mkexpr(t10), mkexpr(t9))));
                     break;
                  }
                  case 0x8: {  /* ADDQH.PH */
                     DIP("addqh.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I16);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I16);

                     /* Add lower halfs of rs and rt
                        and right shift the result by 1. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));
                     assign(t1, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 mkexpr(t0),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));
                     /* Add higher halfs of rs and rt
                        and right shift the result by 1. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t3, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 mkexpr(t2),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));
                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t3), mkexpr(t1)));
                     break;
                  }
                  case 0x9: {  /* SUBQH.PH */
                     DIP("subqh.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);

                     putIReg(rd, binop(Iop_HSub16Sx2,
                                       getIReg(rs), getIReg(rt)));
                     break;
                  }
                  case 0xA: {/* ADDQH_R.PH */
                     DIP("addqh_r.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I16);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I16);

                     /* Add lower halfs of rs and rt, add 1
                        and right shift the result by 1. */
                     assign(t0, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));
                     assign(t1, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 binop(Iop_Add32,
                                                       mkexpr(t0),
                                                       mkU32(0x1)),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));
                     /* Add higher halfs of rs and rt, add 1
                        and right shift the result by 1. */
                     assign(t2, binop(Iop_Add32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t3, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 binop(Iop_Add32,
                                                       mkexpr(t2),
                                                       mkU32(0x1)),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t3), mkexpr(t1)));
                     break;
                  }
                  case 0xB: {  /* SUBQH_R.PH */
                     DIP("subqh_r.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I16);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I16);

                     /* Sub lower halfs of rs and rt, add 1
                        and right shift the result by 1. */
                     assign(t0, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));
                     assign(t1, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 binop(Iop_Add32,
                                                       mkexpr(t0),
                                                       mkU32(0x1)),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));
                     /* Sub higher halfs of rs and rt, add 1
                        and right shift the result by 1. */
                     assign(t2, binop(Iop_Sub32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt)))));
                     assign(t3, unop(Iop_32to16,
                                     binop(Iop_Shr32,
                                           binop(Iop_And32,
                                                 binop(Iop_Add32,
                                                       mkexpr(t2),
                                                       mkU32(0x1)),
                                                 mkU32(0x0001fffe)),
                                           mkU8(0x1))));

                     putIReg(rd, binop(Iop_16HLto32, mkexpr(t3), mkexpr(t1)));
                     break;
                  }
                  case 0xC: {  /* MUL.PH */
                     DIP("mul.ph r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);

                     assign(t0,
                            binop(Iop_Mul32,
                                  unop(Iop_16Sto32,
                                       unop(Iop_32HIto16, getIReg(rs))),
                                  unop(Iop_16Sto32,
                                       unop(Iop_32HIto16, getIReg(rt)))));
                     /* DSP Control flag. */
                     putDSPControl(IRExpr_ITE(unop(Iop_Not1,
                                                   binop(Iop_CmpLE32S,
                                                         mkexpr(t0),
                                                         mkU32(0x7FFF))),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00200000)),
                                              IRExpr_ITE(binop(Iop_CmpLT32S,
                                                               mkexpr(t0),
                                                               mkU32(0xFFFF8000)
                                                             ),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl())));

                     assign(t1,
                            binop(Iop_Mul32,
                                  unop(Iop_16Sto32,
                                       unop(Iop_32to16, getIReg(rs))),
                                  unop(Iop_16Sto32,
                                       unop(Iop_32to16, getIReg(rt)))));
                     /* DSP Control flag. */
                     putDSPControl(IRExpr_ITE(unop(Iop_Not1,
                                                   binop(Iop_CmpLE32S,
                                                         mkexpr(t1),
                                                         mkU32(0x7FFF))),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00200000)),
                                              IRExpr_ITE(binop(Iop_CmpLT32S,
                                                               mkexpr(t1),
                                                               mkU32(0xFFFF8000)
                                                              ),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl())));

                     assign(t2, binop(Iop_16HLto32,
                                      unop(Iop_32to16, mkexpr(t0)),
                                      unop(Iop_32to16, mkexpr(t1))));
                     putIReg(rd, mkexpr(t2));
                     break;
                  }
                  case 0xE: {  /* MUL_S.PH */
                     DIP("mul_s.ph r%u r%u, r%u", rd, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);

                     /* t0 - signed intermediate result. */
                     assign(t0,
                           binop(Iop_Mul32,
                                 unop(Iop_16Sto32,
                                      unop(Iop_32HIto16, getIReg(rs))),
                                 unop(Iop_16Sto32,
                                      unop(Iop_32HIto16, getIReg(rt)))));

                     assign(t1,
                            IRExpr_ITE(unop(Iop_Not1,
                                            binop(Iop_CmpLE32S,
                                                  mkexpr(t0),
                                                  mkU32(0x7FFF))),
                                       mkU32(0x00007FFF),
                                       IRExpr_ITE(binop(Iop_CmpLT32S,
                                                        mkexpr(t0),
                                                        mkU32(0xFFFF8000)),
                                                  mkU32(0xFFFF8000),
                                                  mkexpr(t0))));

                     /* DSP Control flag. */
                     putDSPControl(IRExpr_ITE(unop(Iop_Not1,
                                                   binop(Iop_CmpLE32S,
                                                         mkexpr(t0),
                                                         mkU32(0x7FFF))),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00200000)),
                                              IRExpr_ITE(binop(Iop_CmpLT32S,
                                                               mkexpr(t0),
                                                               mkU32(0xFFFF8000)
                                                              ),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl())));

                     /* t2 - signed intermediate result. */
                     assign(t2, binop(Iop_Mul32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt)))));

                     assign(t3, IRExpr_ITE(unop(Iop_Not1,
                                                binop(Iop_CmpLE32S,
                                                      mkexpr(t2),
                                                      mkU32(0x7FFF))),
                                           mkU32(0x00007FFF),
                                           IRExpr_ITE(binop(Iop_CmpLT32S,
                                                            mkexpr(t2),
                                                            mkU32(0xFFFF8000)),
                                                      mkU32(0xFFFF8000),
                                                      mkexpr(t2))));

                     /* DSP Control flag. */
                     putDSPControl(IRExpr_ITE(unop(Iop_Not1,
                                                   binop(Iop_CmpLE32S,
                                                         mkexpr(t2),
                                                         mkU32(0x7FFF))),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    mkU32(0x00200000)),
                                              IRExpr_ITE(binop(Iop_CmpLT32S,
                                                               mkexpr(t2),
                                                               mkU32(0xFFFF8000)
                                                              ),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl())));

                     assign(t4, binop(Iop_16HLto32,
                                      unop(Iop_32to16, mkexpr(t1)),
                                      unop(Iop_32to16, mkexpr(t3))));
                     putIReg(rd, mkexpr(t4));
                     break;
                  }
                  case 0x10: {  /* ADDQH.W */
                     DIP("addqh.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);

                     assign(t0, binop(Iop_Add64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));
                     assign(t1, binop(Iop_And64,
                                      mkexpr(t0),
                                      mkU64(0x00000001fffffffeULL)));
                     putIReg(rd, unop(Iop_64to32,
                                      binop(Iop_Shr64, mkexpr(t1), mkU8(0x1))));
                     break;
                  }
                  case 0x11: {  /* SUBQH.W */
                     DIP("subqh.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);

                     assign(t0, binop(Iop_Sub64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));
                     assign(t1, binop(Iop_And64,
                                      mkexpr(t0),
                                      mkU64(0x00000001fffffffeULL)));
                     putIReg(rd, unop(Iop_64to32,
                                      binop(Iop_Shr64, mkexpr(t1), mkU8(0x1))));
                     break;
                  }
                  case 0x12: {  /* ADDQH_R.W */
                     DIP("addqh_r.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0, binop(Iop_Add64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));
                     assign(t1, binop(Iop_Add64,
                                      mkexpr(t0),
                                      mkU64(0x0000000000000001ULL)));
                     assign(t2, binop(Iop_And64,
                                      mkexpr(t1),
                                      mkU64(0x00000001fffffffeULL)));
                     putIReg(rd, unop(Iop_64to32,
                                      binop(Iop_Shr64, mkexpr(t2), mkU8(0x1))));
                     break;
                  }
                  case 0x13: {  /* SUBQH_R.W */
                     DIP("subqh_r.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0, binop(Iop_Sub64,
                                      unop(Iop_32Sto64, getIReg(rs)),
                                      unop(Iop_32Sto64, getIReg(rt))));
                     assign(t1, binop(Iop_Add64,
                                      mkexpr(t0),
                                      mkU64(0x0000000000000001ULL)));
                     assign(t2, binop(Iop_And64,
                                      mkexpr(t1),
                                      mkU64(0x00000001fffffffeULL)));
                     putIReg(rd, unop(Iop_64to32,
                                      binop(Iop_Shr64, mkexpr(t2), mkU8(0x1))));
                     break;
                  }
                  case 0x16: {  /* MULQ_S.W */
                     DIP("mulq_s.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t0, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            getIReg(rt), getIReg(rs)),
                                      mkU8(0x1)));
                     assign(t1, binop(Iop_CmpEQ32,
                                      getIReg(rt), mkU32(0x80000000)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      getIReg(rs), mkU32(0x80000000)));

                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              IRExpr_ITE(mkexpr(t2),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(mkexpr(t2),
                                                       mkU32(0x7fffffff),
                                                       unop(Iop_64HIto32,
                                                            mkexpr(t0))),
                                            unop(Iop_64HIto32, mkexpr(t0))));
                     break;
                  }
                  case 0x17: {  /* MULQ_RS.W */
                     DIP("mulq_rs.w r%u, r%u, r%u", rd, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I1);
                     t2 = newTemp(Ity_I1);

                     assign(t0, binop(Iop_Add64,
                                      binop(Iop_Shl64,
                                            binop(Iop_MullS32,
                                                  getIReg(rt),
                                                  getIReg(rs)),
                                            mkU8(0x1)),
                                      mkU64(0x0000000080000000ULL)));
                     assign(t1,
                            binop(Iop_CmpEQ32, getIReg(rt), mkU32(0x80000000)));
                     assign(t2,
                            binop(Iop_CmpEQ32, getIReg(rs), mkU32(0x80000000)));
                     putDSPControl(IRExpr_ITE(mkexpr(t1),
                                              IRExpr_ITE(mkexpr(t2),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               mkU32(0x00200000)
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     putIReg(rd, IRExpr_ITE(mkexpr(t1),
                                            IRExpr_ITE(mkexpr(t2),
                                                       mkU32(0x7fffffff),
                                                       unop(Iop_64HIto32,
                                                            mkexpr(t0))),
                                            unop(Iop_64HIto32, mkexpr(t0))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of ADDUH.QB/MUL.PH */
            }
            case 0x30: {  /* DPAQ.W.PH */
               switch(sa) {
                  case 0x0: {  /* DPA.W.PH */
                     DIP("dpa.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t1,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t2,
                            binop(Iop_Add64,
                                  getAcc(ac),
                                  binop(Iop_Add64, mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, mkexpr(t2));
                     break;
                  }
                  case 0x1: {  /* DPS.W.PH */
                     DIP("dps.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t1,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t2,
                            binop(Iop_Sub64,
                                  getAcc(ac),
                                  binop(Iop_Add64, mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, mkexpr(t2));
                     break;
                  }
                  case 0x2: {  /* MULSA.W.PH */
                     DIP("mulsa.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);

                     assign(t4, getAcc(ac));
                     assign(t0, binop(Iop_Mul32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32to16, getIReg(rs)))));
                     assign(t1, binop(Iop_Mul32,
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      unop(Iop_16Sto32,
                                           unop(Iop_32HIto16, getIReg(rs)))));
                     assign(t2, binop(Iop_Sub32, mkexpr(t1), mkexpr(t0)));
                     putAcc(ac, binop(Iop_Add64,
                                      mkexpr(t4),
                                      unop(Iop_32Sto64, mkexpr(t2))));
                     break;
                  }
                  case 0x3: {  /* DPAU.H.QBL */
                     DIP("dpau.h.qbl ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_I64);

                     assign(t0,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t1,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t2,
                            unop(Iop_32Uto64,
                                 binop(Iop_Add32,
                                       mkexpr(t0),
                                       mkexpr(t1))));
                     assign(t3,
                            binop(Iop_Add64, getAcc(ac), mkexpr(t2)));
                     putAcc(ac, mkexpr(t3));
                     break;
                  }
                  case 0x4: {  /* DPAQ_S.W.PH */
                     DIP("dpaq_s.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16, getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16, getIReg(rt)))
                                           ),
                                      mkU8(0x1)));
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t8,
                            IRExpr_ITE(mkexpr(t6),
                                       IRExpr_ITE(mkexpr(t7),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t5)),
                                       mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              IRExpr_ITE(mkexpr(t7),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t9, binop(Iop_Add64,
                                      binop(Iop_Add64, mkexpr(t4), mkexpr(t8)),
                                      mkexpr(t0)));
                     putAcc(ac, mkexpr(t9));
                     break;
                  }
                  case 0x5: {  /* DPSQ_S.W.PH */
                     DIP("dpsq_s.w.ph ac%u r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5,
                            binop(Iop_Shl64,
                                  binop(Iop_MullS32,
                                        unop(Iop_16Sto32,
                                             unop(Iop_32to16, getIReg(rs))),
                                        unop(Iop_16Sto32,
                                             unop(Iop_32to16, getIReg(rt)))),
                                  mkU8(0x1)));
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t8,
                            IRExpr_ITE(mkexpr(t6),
                                       IRExpr_ITE(mkexpr(t7),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t5)),
                                       mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              IRExpr_ITE(mkexpr(t7),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t9,
                            binop(Iop_Sub64,
                                  mkexpr(t0),
                                  binop(Iop_Add64, mkexpr(t4), mkexpr(t8))));
                     putAcc(ac, mkexpr(t9));
                     break;
                  }
                  case 0x6: {  /* MULSAQ_S.W.PH */
                     DIP("mulsaq_s.w.ph ac%u r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I32);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I64);
                     t8 = newTemp(Ity_I32);
                     t9 = newTemp(Ity_I32);

                     assign(t0, unop(Iop_16Sto32,
                                     unop(Iop_32HIto16, getIReg(rs))));
                     assign(t1, unop(Iop_16Sto32,
                                     unop(Iop_32HIto16, getIReg(rt))));

                     assign(t8, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 unop(Iop_16Uto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rs))),
                                                 mkU32(0x8000))),
                                    unop(Iop_1Sto32,
                                         binop(Iop_CmpEQ32,
                                               unop(Iop_16Uto32,
                                                    unop(Iop_32HIto16,
                                                         getIReg(rt))),
                                               mkU32(0x8000)))));
                     /* DSPControl_outflag:16+acc <- 1 */
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t8),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x00010000),
                                                          mkU8(ac))),
                                              getDSPControl()));

                     /* tempB_31..0 */
                     assign(t2,
                            IRExpr_ITE(binop(Iop_CmpNE32,
                                             mkexpr(t8), mkU32(0x0)),
                                       mkU32(0x7FFFFFFF),
                                       binop(Iop_Shl32,
                                             binop(Iop_Mul32,
                                                   mkexpr(t0), mkexpr(t1)),
                                             mkU8(1))));

                     assign(t3, unop(Iop_16Sto32,
                                     unop(Iop_32to16, getIReg(rs))));
                     assign(t4, unop(Iop_16Sto32,
                                     unop(Iop_32to16, getIReg(rt))));

                     assign(t9, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 unop(Iop_16Uto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rs))),
                                                 mkU32(0x8000))),
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 unop(Iop_16Uto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rt))),
                                                 mkU32(0x8000)))));
                     /* DSPControl_outflag:16+acc <- 1 */
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    mkexpr(t9),
                                                    mkU32(0x0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x00010000),
                                                          mkU8(ac))),
                                              getDSPControl()));
                     /* tempA_31..0 */
                     assign(t5,
                            IRExpr_ITE(binop(Iop_CmpNE32,
                                             mkexpr(t9),
                                             mkU32(0x0)),
                                       mkU32(0x7FFFFFFF),
                                       binop(Iop_Shl32,
                                             binop(Iop_Mul32,
                                                   mkexpr(t3),
                                                   mkexpr(t4)),
                                             mkU8(1))));
                     /* dotp_63..0 */
                     assign(t6,
                            binop(Iop_Sub64,
                                  unop(Iop_32Sto64, mkexpr(t2)),
                                  unop(Iop_32Sto64, mkexpr(t5))));
                     /* tempC_63..0 */
                     assign(t7, binop(Iop_Add64, getAcc(ac), mkexpr(t6)));

                     putAcc(ac, mkexpr(t7));
                     break;
                  }
                  case 0x7: {  /* DPAU.H.QBR */
                     DIP("dpau.h.qbr ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_I64);

                     assign(t0,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t1,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t2, unop(Iop_32Uto64,
                                     binop(Iop_Add32, mkexpr(t0), mkexpr(t1))));
                     assign(t3, binop(Iop_Add64, getAcc(ac), mkexpr(t2)));
                     putAcc(ac, mkexpr(t3));
                     break;
                  }
                  case 0x8: {  /* DPAX.W.PH */
                     DIP("dpax.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t1,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t2,
                            binop(Iop_Add64,
                                  getAcc(ac),
                                  binop(Iop_Add64, mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, mkexpr(t2));
                     break;
                  }
                  case 0x9: {  /* DPSX.W.PH */
                     DIP("dpsx.w.ph ac%u r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I64);

                     assign(t0,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t1,
                            unop(Iop_32Sto64,
                                 binop(Iop_Mul32,
                                       unop(Iop_16Sto32,
                                            unop(Iop_32to16, getIReg(rs))),
                                       unop(Iop_16Sto32,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t2,
                            binop(Iop_Sub64,
                                  getAcc(ac),
                                  binop(Iop_Add64, mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, mkexpr(t2));
                     break;
                  }
                  case 0xB: {  /* DPSU.H.QBL */
                     DIP("dpsu.h.qbl ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_I64);

                     assign(t0,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t1,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32HIto16, getIReg(rt))))));
                     assign(t2,
                            unop(Iop_32Uto64,
                                 binop(Iop_Add32, mkexpr(t0), mkexpr(t1))));
                     assign(t3,
                            binop(Iop_Sub64, getAcc(ac), mkexpr(t2)));
                     putAcc(ac, mkexpr(t3));
                     break;
                  }
                  case 0xC: {  /* DPAQ_SA.L.W */
                     DIP("dpaq_sa.l.w ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I64);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I1);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            getIReg(rs), getIReg(rt)),
                                      mkU8(0x1)));

                     assign(t2, binop(Iop_CmpEQ32,
                                      getIReg(rs),
                                      mkU32(0x80000000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      getIReg(rt),
                                      mkU32(0x80000000)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x7fffffffffffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5, binop(Iop_Add64,
                                      unop(Iop_32Uto64,
                                           unop(Iop_64to32, mkexpr(t0))),
                                      unop(Iop_32Uto64,
                                           unop(Iop_64to32, mkexpr(t4)))));
                     assign(t6,
                            binop(Iop_Add64,
                                  binop(Iop_Add64,
                                        unop(Iop_32Sto64,
                                             unop(Iop_64HIto32, mkexpr(t0))),
                                        unop(Iop_32Sto64,
                                             unop(Iop_64HIto32, mkexpr(t4)))),
                                  unop(Iop_32Uto64,
                                       binop(Iop_And32,
                                             unop(Iop_64HIto32, mkexpr(t5)),
                                             mkU32(0x1)))));
                     assign(t7, binop(Iop_32HLto64,
                                      unop(Iop_64to32, mkexpr(t6)),
                                      unop(Iop_64to32, mkexpr(t5))));
                     assign(t8, binop(Iop_CmpEQ32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t6)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t6)),
                                            mkU32(0x00000001))));
                     assign(t9, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32,
                                                 mkexpr(t6)),
                                            mkU32(0x00000001)),
                                      mkU32(0x1)));
                     putDSPControl(IRExpr_ITE(mkexpr(t8),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     putAcc(ac,
                            IRExpr_ITE(mkexpr(t8),
                                       mkexpr(t7),
                                       IRExpr_ITE(mkexpr(t9),
                                                  mkU64(0x8000000000000000ULL),
                                                  mkU64(0x7fffffffffffffffULL)))
                           );
                     break;
                  }
                  case 0xD: {  /* DPSQ_SA.L.W */
                     DIP("dpsq_sa.l.w ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     t7 = newTemp(Ity_I64);
                     t8 = newTemp(Ity_I1);
                     t9 = newTemp(Ity_I1);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            getIReg(rs), getIReg(rt)),
                                      mkU8(0x1)));

                     assign(t2, binop(Iop_CmpEQ32,
                                      getIReg(rs),
                                      mkU32(0x80000000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      getIReg(rt),
                                      mkU32(0x80000000)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x7fffffffffffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5, binop(Iop_Sub64,
                                      unop(Iop_32Uto64,
                                           unop(Iop_64to32, mkexpr(t0))),
                                      unop(Iop_32Uto64,
                                           unop(Iop_64to32, mkexpr(t4)))));
                     assign(t6, binop(Iop_Sub64,
                                      binop(Iop_Add64,
                                            unop(Iop_32Sto64,
                                                 unop(Iop_64HIto32, mkexpr(t0))
                                                ),
                                            unop(Iop_32Sto64,
                                                 unop(Iop_1Sto32,
                                                      binop(Iop_CmpLT32U,
                                                            unop(Iop_64to32,
                                                                 mkexpr(t0)),
                                                            unop(Iop_64to32,
                                                                mkexpr(t4)))))),
                                      unop(Iop_32Sto64,
                                           unop(Iop_64HIto32, mkexpr(t4)))));
                     assign(t7, binop(Iop_32HLto64,
                                      unop(Iop_64to32, mkexpr(t6)),
                                      unop(Iop_64to32, mkexpr(t5))));
                     assign(t8, binop(Iop_CmpEQ32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t6)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t6)),
                                            mkU32(0x00000001))));
                     assign(t9, binop(Iop_CmpEQ32,
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t6)),
                                            mkU32(0x00000001)),
                                      mkU32(0x1)));
                     putDSPControl(IRExpr_ITE(mkexpr(t8),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     putAcc(ac,
                            IRExpr_ITE(mkexpr(t8),
                                       mkexpr(t7),
                                       IRExpr_ITE(mkexpr(t9),
                                                  mkU64(0x8000000000000000ULL),
                                                  mkU64(0x7fffffffffffffffULL)))
                           );
                     break;
                  }
                  case 0xF: {  /* DPSU.H.QBR */
                     DIP("dpsu.h.qbr ac%u r%u, r%u", ac, rs, rt);
                     vassert(!mode64);

                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_I64);

                     assign(t0,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16HIto8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t1,
                            binop(Iop_Mul32,
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rs)))),
                                  unop(Iop_8Uto32,
                                       unop(Iop_16to8,
                                            unop(Iop_32to16, getIReg(rt))))));
                     assign(t2, unop(Iop_32Uto64,
                                     binop(Iop_Add32, mkexpr(t0), mkexpr(t1))));
                     assign(t3, binop(Iop_Sub64, getAcc(ac), mkexpr(t2)));
                     putAcc(ac, mkexpr(t3));

                     break;
                  }
                  case 0x10: {  /* MAQ_SA.W.PHL */
                     DIP("maq_sa.w.phl ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));
                     assign(t1, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl register.
                     */
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     /* Add intermediate product and value in the
                        accumulator. */
                     assign(t5, binop(Iop_Add64, mkexpr(t0), mkexpr(t4)));

                     /* Compare bits 31 and 32 of the value in t5. */
                     assign(t6, binop(Iop_CmpEQ32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t5)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t5)),
                                            mkU32(1))));
                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     assign(t7,
                            IRExpr_ITE(mkexpr(t6),
                                       mkexpr(t5),
                                       IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        binop(Iop_And32,
                                                              unop(Iop_64HIto32,
                                                                   mkexpr(t5)),
                                                              mkU32(1)),
                                                        mkU32(0x0)),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkU64(0xffffffff80000000ULL)))
                           );
                     putAcc(ac, mkexpr(t7));
                     break;
                  }
                  case 0x12: {  /* MAQ_SA.W.PHR */
                     DIP("maq_sa.w.phr ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));
                     assign(t1, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl
                        register. */
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));
                     /* Add intermediate product and value in the
                        accumulator. */
                     assign(t5, binop(Iop_Add64, mkexpr(t0), mkexpr(t4)));

                     /* Compare bits 31 and 32 of the value in t5. */
                     assign(t6, binop(Iop_CmpEQ32,
                                      binop(Iop_Shr32,
                                            binop(Iop_And32,
                                                  unop(Iop_64to32, mkexpr(t5)),
                                                  mkU32(0x80000000)),
                                            mkU8(31)),
                                      binop(Iop_And32,
                                            unop(Iop_64HIto32, mkexpr(t5)),
                                            mkU32(1))));
                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     assign(t7,
                            IRExpr_ITE(mkexpr(t6),
                                       mkexpr(t5),
                                       IRExpr_ITE(binop(Iop_CmpEQ32,
                                                        binop(Iop_And32,
                                                              unop(Iop_64HIto32,
                                                                   mkexpr(t5)),
                                                              mkU32(1)),
                                                        mkU32(0x0)),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkU64(0xffffffff80000000ULL)))
                           );
                     putAcc(ac, mkexpr(t7));
                     break;
                  }
                  case 0x14: {  /* MAQ_S.W.PHL */
                     DIP("maq_s.w.phl ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I64);

                     assign(t5, getAcc(ac));

                     assign(t0, unop(Iop_16Sto32,
                                     unop(Iop_32HIto16, getIReg(rs))));
                     assign(t1, unop(Iop_16Sto32,
                                     unop(Iop_32HIto16, getIReg(rt))));

                     assign(t2, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000))),
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t1),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000)))));

                     assign(t3, binop(Iop_CmpEQ32, mkexpr(t2), mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));

                     assign(t4, unop(Iop_64to32,
                                     binop(Iop_MullS32,
                                           mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Add64,
                                                 unop(Iop_32Sto64,
                                                      binop(Iop_Shl32,
                                                            mkexpr(t4),
                                                            mkU8(0x1))),
                                                 mkexpr(t5)),
                                           binop(Iop_Add64,
                                                 mkexpr(t5),
                                                 unop(Iop_32Sto64,
                                                      mkU32(0x7fffffff)))));
                     break;
                  }
                  case 0x16: {  /* MAQ_S.W.PHR */
                     DIP("maq_s.w.phr ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I32);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I32);
                     t5 = newTemp(Ity_I64);

                     assign(t5, getAcc(ac));

                     assign(t0, unop(Iop_16Sto32,
                                     unop(Iop_32to16, getIReg(rs))));
                     assign(t1, unop(Iop_16Sto32,
                                     unop(Iop_32to16, getIReg(rt))));

                     assign(t2, binop(Iop_And32,
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t0),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000))),
                                      unop(Iop_1Sto32,
                                           binop(Iop_CmpEQ32,
                                                 binop(Iop_And32,
                                                       mkexpr(t1),
                                                       mkU32(0xffff)),
                                                 mkU32(0x8000)))));

                     assign(t3, binop(Iop_CmpEQ32, mkexpr(t2), mkU32(0x0)));

                     putDSPControl(IRExpr_ITE(mkexpr(t3),
                                              getDSPControl(),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));

                     assign(t4, unop(Iop_64to32,
                                     binop(Iop_MullS32,
                                           mkexpr(t0), mkexpr(t1))));
                     putAcc(ac, IRExpr_ITE(mkexpr(t3),
                                           binop(Iop_Add64,
                                                 unop(Iop_32Sto64,
                                                      binop(Iop_Shl32,
                                                            mkexpr(t4),
                                                            mkU8(0x1))),
                                                 mkexpr(t5)),
                                           binop(Iop_Add64,
                                                 mkexpr(t5),
                                                 unop(Iop_32Sto64,
                                                      mkU32(0x7fffffff)))));
                     break;
                  }
                  case 0x18: {  /* DPAQX_S.W.PH */
                     DIP("dpaqx_s.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                  mkU32(0x1),
                                                                  mkU8(ac+16))),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t8,
                            IRExpr_ITE(mkexpr(t6),
                                       IRExpr_ITE(mkexpr(t7),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t5)),
                                       mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              IRExpr_ITE(mkexpr(t7),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t9, binop(Iop_Add64,
                                      binop(Iop_Add64, mkexpr(t4), mkexpr(t8)),
                                      mkexpr(t0)));
                     putAcc(ac, mkexpr(t9));
                     break;
                  }
                  case 0x19: {  /* DPSQX_S.W.PH */
                     DIP("dpsqx_s.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);

                     assign(t0, getAcc(ac));

                     assign(t1, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t4,
                            IRExpr_ITE(mkexpr(t2),
                                       IRExpr_ITE(mkexpr(t3),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t1)),
                                       mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(mkexpr(t2),
                                              IRExpr_ITE(mkexpr(t3),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t5, binop(Iop_Shl64,
                                      binop(Iop_MullS32,
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32to16,
                                                      getIReg(rs))),
                                            unop(Iop_16Sto32,
                                                 unop(Iop_32HIto16,
                                                      getIReg(rt)))),
                                      mkU8(0x1)));
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));
                     assign(t8,
                            IRExpr_ITE(mkexpr(t6),
                                       IRExpr_ITE(mkexpr(t7),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t5)),
                                       mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(mkexpr(t6),
                                              IRExpr_ITE(mkexpr(t7),
                                                         binop(Iop_Or32,
                                                               getDSPControl(),
                                                               binop(Iop_Shl32,
                                                                     mkU32(0x1),
                                                                     mkU8(ac+16)
                                                                    )
                                                              ),
                                                         getDSPControl()),
                                              getDSPControl()));

                     assign(t9, binop(Iop_Sub64,
                                     mkexpr(t0),
                                     binop(Iop_Add64, mkexpr(t4), mkexpr(t8))));
                     putAcc(ac, mkexpr(t9));
                     break;
                  }
                  case 0x1A: {  /* DPAQX_SA.W.PH */
                     DIP("dpaqx_sa.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     /* Calculate the first cross dot product and saturate if
                        needed. */
                     assign(t1, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl
                        register. */
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t4, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t2)),
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t3))),
                                                 mkU32(0)),
                                           mkU64(0x000000007fffffffULL),
                                           mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    binop(Iop_And32,
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t2)),
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t3))),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16))),
                                              getDSPControl()));
                     /* Calculate second cross dot product and saturate if
                        needed. */
                     assign(t5, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl
                        register. */
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t8, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t6)),
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t7))),
                                                 mkU32(0)),
                                           mkU64(0x000000007fffffffULL),
                                           mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    binop(Iop_And32,
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t6)),
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t7))),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16))),
                                              getDSPControl()));
                     /* Subtract intermediate products from value in the
                        accumulator. */
                     assign(t9,
                            binop(Iop_Add64,
                                  mkexpr(t0),
                                  binop(Iop_Add64, mkexpr(t8), mkexpr(t4))));

                     putAcc(ac,
                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                             binop(Iop_And32,
                                                   unop(Iop_64HIto32,
                                                        mkexpr(t9)),
                                                   mkU32(0x80000000)),
                                             mkU32(0x0)),
                                       IRExpr_ITE(binop(Iop_CmpNE32,
                                                        unop(Iop_64HIto32,
                                                             binop(Iop_Shl64,
                                                                   mkexpr(t9),
                                                                   mkU8(1))),
                                                        mkU32(0x0)),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t9)),
                                       IRExpr_ITE(binop(Iop_CmpNE32,
                                                        unop(Iop_64HIto32,
                                                             binop(Iop_Shl64,
                                                                   mkexpr(t9),
                                                                   mkU8(1))),
                                                        mkU32(0xffffffff)),
                                                  mkU64(0xffffffff80000000ULL),
                                                  mkexpr(t9))));
                     assign(t10, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  unop(Iop_64to32,
                                                       mkexpr(t9)),
                                                  unop(Iop_64to32,
                                                       getAcc(ac))),
                                           getDSPControl(),
                                           binop(Iop_Or32,
                                                 getDSPControl(),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkU8(ac+16)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    unop(Iop_64HIto32,
                                                         mkexpr(t9)),
                                                    unop(Iop_64HIto32,
                                                         getAcc(ac))),
                                              mkexpr(t10),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     break;
                  }
                  case 0x1B: {  /* DPSQX_SA.W.PH */
                     DIP("dpsqx_sa.w.ph ac%u, r%u, r%u", ac, rs, rt);
                     vassert(!mode64);
                     t0 = newTemp(Ity_I64);
                     t1 = newTemp(Ity_I64);
                     t2 = newTemp(Ity_I1);
                     t3 = newTemp(Ity_I1);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I1);
                     t7 = newTemp(Ity_I1);
                     t8 = newTemp(Ity_I64);
                     t9 = newTemp(Ity_I64);
                     t10 = newTemp(Ity_I32);

                     assign(t0, getAcc(ac));
                     /* Calculate the first cross dot product and saturate if
                        needed. */
                     assign(t1, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl
                        register. */
                     assign(t2, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t3, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t4, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t2)),
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t3))),
                                                 mkU32(0)),
                                           mkU64(0x000000007fffffffULL),
                                           mkexpr(t1)));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    binop(Iop_And32,
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t2)),
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t3))),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16))),
                                              getDSPControl()));
                     /* Calculate second cross dot product and saturate if
                        needed. */
                     assign(t5, unop(Iop_32Sto64,
                                     binop(Iop_Shl32,
                                           binop(Iop_Mul32,
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32to16,
                                                           getIReg(rs))),
                                                 unop(Iop_16Sto32,
                                                      unop(Iop_32HIto16,
                                                           getIReg(rt)))),
                                           mkU8(0x1))));

                     /* If both input arguments are equal 0x8000, saturate
                        intermediate product and write to DSPControl
                        register. */
                     assign(t6, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32to16, getIReg(rs))),
                                      mkU32(0x00008000)));
                     assign(t7, binop(Iop_CmpEQ32,
                                      unop(Iop_16Uto32,
                                           unop(Iop_32HIto16, getIReg(rt))),
                                      mkU32(0x00008000)));

                     assign(t8, IRExpr_ITE(binop(Iop_CmpNE32,
                                                 binop(Iop_And32,
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t6)),
                                                       unop(Iop_1Sto32,
                                                            mkexpr(t7))),
                                                 mkU32(0)),
                                           mkU64(0x000000007fffffffULL),
                                           mkexpr(t5)));

                     putDSPControl(IRExpr_ITE(binop(Iop_CmpNE32,
                                                    binop(Iop_And32,
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t6)),
                                                          unop(Iop_1Sto32,
                                                               mkexpr(t7))),
                                                    mkU32(0)),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16))),
                                              getDSPControl()));
                     /* Subtract intermediate products from value in the
                        accumulator. */
                     assign(t9,
                            binop(Iop_Sub64,
                                  mkexpr(t0),
                                  binop(Iop_Add64, mkexpr(t8), mkexpr(t4))));

                     putAcc(ac,
                            IRExpr_ITE(binop(Iop_CmpEQ32,
                                             binop(Iop_And32,
                                                   unop(Iop_64HIto32,
                                                        mkexpr(t9)),
                                                   mkU32(0x80000000)),
                                             mkU32(0x0)),
                                       IRExpr_ITE(binop(Iop_CmpNE32,
                                                        unop(Iop_64HIto32,
                                                             binop(Iop_Shl64,
                                                                   mkexpr(t9),
                                                                   mkU8(1))),
                                                        mkU32(0x0)),
                                                  mkU64(0x000000007fffffffULL),
                                                  mkexpr(t9)),
                                       IRExpr_ITE(binop(Iop_CmpNE32,
                                                        unop(Iop_64HIto32,
                                                             binop(Iop_Shl64,
                                                                   mkexpr(t9),
                                                                   mkU8(1))),
                                                        mkU32(0xffffffff)),
                                                  mkU64(0xffffffff80000000ULL),
                                                  mkexpr(t9))));
                     assign(t10, IRExpr_ITE(binop(Iop_CmpEQ32,
                                                  unop(Iop_64to32,
                                                       mkexpr(t9)),
                                                  unop(Iop_64to32,
                                                       getAcc(ac))),
                                           getDSPControl(),
                                           binop(Iop_Or32,
                                                 getDSPControl(),
                                                 binop(Iop_Shl32,
                                                       mkU32(0x1),
                                                       mkU8(ac+16)))));
                     putDSPControl(IRExpr_ITE(binop(Iop_CmpEQ32,
                                                    unop(Iop_64HIto32,
                                                         mkexpr(t9)),
                                                    unop(Iop_64HIto32,
                                                         getAcc(ac))),
                                              mkexpr(t10),
                                              binop(Iop_Or32,
                                                    getDSPControl(),
                                                    binop(Iop_Shl32,
                                                          mkU32(0x1),
                                                          mkU8(ac+16)))));
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of DPAQ.W.PH */
            }
            case 0x31: {  /* APPEND */
               switch(sa) {
                  case 0x0: {  /* APPEND */
                     DIP("append r%u, r%u, %u", rt, rs, rd);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     assign(t1, binop(Iop_Shl32, getIReg(rt), mkU8(rd)));

                     if (31 == rd) {
                        putIReg(rt, binop(Iop_Or32,
                                          mkexpr(t1),
                                          binop(Iop_And32,
                                                getIReg(rs),
                                                mkU32(0x7fffffff))));
                     } else if (1 == rd) {
                        putIReg(rt,
                                binop(Iop_Or32,
                                      mkexpr(t1),
                                      binop(Iop_And32,
                                            getIReg(rs), mkU32(0x1))));
                     } else {
                        assign(t2,
                               unop(Iop_Not32,
                                    binop(Iop_Shl32,
                                          mkU32(0xffffffff), mkU8(rd))));

                        putIReg(rt, binop(Iop_Or32,
                                          mkexpr(t1),
                                          binop(Iop_And32,
                                                getIReg(rs), mkexpr(t2))));
                     }
                     break;
                  }
                  case 0x1: {  /* PREPEND */
                     DIP("prepend r%u, r%u, %u", rt, rs, rd);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     if (0 != rd) {
                        assign(t1, binop(Iop_Shr32, getIReg(rt), mkU8(rd)));

                        if (31 == rd) {
                           putIReg(rt, binop(Iop_Or32,
                                             mkexpr(t1),
                                             binop(Iop_Shl32,
                                                   binop(Iop_And32,
                                                         getIReg(rs),
                                                         mkU32(0x7fffffff)),
                                                   mkU8(1))));
                        } else if (1 == rd) {
                           putIReg(rt, binop(Iop_Or32,
                                             mkexpr(t1),
                                             binop(Iop_Shl32,
                                                   binop(Iop_And32,
                                                         getIReg(rs),
                                                         mkU32(0x1)),
                                                   mkU8(31))));
                        } else {
                           assign(t2, binop(Iop_Add32, mkU32(rd), mkU32(0x1)));

                           assign(t3, unop(Iop_Not32,
                                           binop(Iop_Shl32,
                                                 mkU32(0xffffffff),
                                                 unop(Iop_32to8, mkexpr(t2)))));

                           putIReg(rt, binop(Iop_Or32,
                                             mkexpr(t1),
                                             binop(Iop_Shl32,
                                                   binop(Iop_And32,
                                                         getIReg(rs),
                                                         mkexpr(t3)),
                                                   mkU8(32-rd))));
                        }
                     }
                     break;
                  }
                  case 0x10: {  /* BALIGN */
                     DIP("balign r%u, r%u, %u", rt, rs, rd);
                     vassert(!mode64);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I32);
                     t3 = newTemp(Ity_I32);

                     if ((2 != rd) && (0 != rd)) {
                        assign(t1, binop(Iop_Shl32,
                                         binop(Iop_And32,
                                               mkU32(rd), mkU32(0x3)),
                                         mkU8(0x3)));
                        assign(t2, binop(Iop_Shl32,
                                         getIReg(rt),
                                         unop(Iop_32to8, mkexpr(t1))));
                        assign(t3, binop(Iop_Shr32,
                                         getIReg(rs),
                                         unop(Iop_32to8,
                                              binop(Iop_Shl32,
                                                    binop(Iop_Sub32,
                                                          mkU32(0x4),
                                                          binop(Iop_And32,
                                                                mkU32(rd),
                                                                mkU32(0x3))),
                                                    mkU8(0x3)))));
                        putIReg(rt, binop(Iop_Or32, mkexpr(t2), mkexpr(t3)));
                     }
                     break;
                  }
                  default:
                     return -1;
               }
               break;  /* end of APPEND */
            }
            default:
               return -1;
         }
         break;
      }
      default:
            return -1;
   }
   return 0;
}

static Int msa_I8_logical(UInt cins, UChar wd, UChar ws) {
   IRTemp t1, t2;
   UShort operation;
   UChar i8;

   operation = (cins >> 24) & 3;
   i8 = (cins & 0x00FF0000) >> 16;
   switch (operation) {
      case 0x00: {  /* ANDI.B */
            DIP("ANDI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t1, getWReg(ws));
            assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            putWReg(wd, binop(Iop_AndV128, mkexpr(t1), mkexpr(t2)));
            break;
         }

      case 0x01: { /* ORI.B */
            DIP("ORI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t1, getWReg(ws));
            assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            putWReg(wd, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            break;
         }

      case 0x02: { /* NORI.B */
            DIP("NORI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t1, getWReg(ws));
            assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            putWReg(wd, unop(Iop_NotV128, binop(Iop_OrV128,
                                                mkexpr(t1), mkexpr(t2))));
            break;
         }

      case 0x03: {  /* XORI.B */
            DIP("XORI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t1, getWReg(ws));
            assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            putWReg(wd, binop(Iop_XorV128, mkexpr(t1), mkexpr(t2)));
            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_I8_branch(UInt cins, UChar wd, UChar ws) {
   IRTemp t1, t2, t3, t4;
   UShort operation;
   UChar i8;

   operation = (cins >> 24) & 3;
   i8 = (cins & 0x00FF0000) >> 16;
   switch (operation) {
      case 0x00: { /* BMNZI.B */
            DIP("BMNZI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            t4 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t4, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            assign(t1, binop(Iop_AndV128, getWReg(ws), mkexpr(t4)));
            assign(t2, binop(Iop_AndV128, getWReg(wd),
                                          unop(Iop_NotV128, mkexpr(t4))));
            assign(t3, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x01: { /* BMZI.B */
            DIP("BMZI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            t4 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t4, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            assign(t1, binop(Iop_AndV128, getWReg(wd), mkexpr(t4)));
            assign(t2, binop(Iop_AndV128, getWReg(ws),
                             unop(Iop_NotV128, mkexpr(t4))));
            assign(t3, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x02: { /* BSELI.B */
            DIP("BSELI.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            t4 = newTemp(Ity_V128);
            ULong tmp = i8;
            tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                   (tmp << 32) | (tmp << 24) | (tmp << 16) |
                   (tmp << 8);
            assign(t4, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            assign(t1, binop(Iop_AndV128, getWReg(wd), mkexpr(t4)));
            assign(t2, binop(Iop_AndV128, getWReg(ws),
                             unop(Iop_NotV128, getWReg(wd))));
            assign(t3, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_I8_shift(UInt cins, UChar wd, UChar ws) {
   IRTemp t1, t2;
   UShort operation;
   UChar i8;

   operation = (cins >> 24) & 3;
   i8 = (cins & 0x00FF0000) >> 16;
   switch (operation) {
      case 0x00: { /* SHF.B */
            DIP("SHF.B w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(wd));
            assign(t2, getWReg(ws));
            Int i;
            IRTemp tmp[16];

            for (i = 0; i < 16; i++) {
               tmp[i] = newTemp(Ity_I8);
               assign(tmp[i],
                      binop(Iop_GetElem8x16, mkexpr(t2),
                            mkU8(i - (i % 4) +
                                 ((i8 >> (i % 4) * 2) & 0x03))));
            }

            putWReg(wd, binop(Iop_64HLtoV128,
                              binop(Iop_32HLto64,
                                    binop(Iop_16HLto32,
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[15]),
                                                mkexpr(tmp[14])),
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[13]),
                                                mkexpr(tmp[12]))),
                                    binop(Iop_16HLto32,
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[11]),
                                                mkexpr(tmp[10])),
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[9]),
                                                mkexpr(tmp[8])))),
                              binop(Iop_32HLto64,
                                    binop(Iop_16HLto32,
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[7]),
                                                mkexpr(tmp[6])),
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[5]),
                                                mkexpr(tmp[4]))),
                                    binop(Iop_16HLto32,
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[3]),
                                                mkexpr(tmp[2])),
                                          binop(Iop_8HLto16,
                                                mkexpr(tmp[1]),
                                                mkexpr(tmp[0]))))));
            break;
         }

      case 0x01: { /* SHF.H */
            DIP("SHF.H w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(wd));
            assign(t2, getWReg(ws));
            Int i;
            IRTemp tmp[8];

            for (i = 0; i < 8; i++) {
               tmp[i] = newTemp(Ity_I16);
               assign(tmp[i],
                      binop(Iop_GetElem16x8, mkexpr(t2),
                            mkU8(i - (i % 4) +
                                 ((i8 >> (i % 4) * 2) & 0x03))));
            }

            putWReg(wd, binop(Iop_64HLtoV128,
                              binop(Iop_32HLto64,
                                    binop(Iop_16HLto32,
                                          mkexpr(tmp[7]), mkexpr(tmp[6])),
                                    binop(Iop_16HLto32,
                                          mkexpr(tmp[5]), mkexpr(tmp[4]))),
                              binop(Iop_32HLto64,
                                    binop(Iop_16HLto32,
                                          mkexpr(tmp[3]), mkexpr(tmp[2])),
                                    binop(Iop_16HLto32,
                                          mkexpr(tmp[1]), mkexpr(tmp[0])))));
            break;
         }

      case 0x02: { /* SHF.W */
            DIP("SHF.W w%d, w%d, %d", wd, ws, i8);
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(wd));
            assign(t2, getWReg(ws));
            Int i;
            IRTemp tmp[4];

            for (i = 0; i < 4; i++) {
               tmp[i] = newTemp(Ity_I32);
               assign(tmp[i],
                      binop(Iop_GetElem32x4, mkexpr(t2),
                            mkU8(i - (i % 4) +
                                 ((i8 >> (i % 4) * 2) & 0x03))));
            }

            putWReg(wd, binop(Iop_64HLtoV128,
                              binop(Iop_32HLto64,
                                    mkexpr(tmp[3]), mkexpr(tmp[2])),
                              binop(Iop_32HLto64,
                                    mkexpr(tmp[1]), mkexpr(tmp[0]))));
            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_I5_06(UInt cins, UChar wd, UChar ws) { /* I5 (0x06) */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* ADDVI */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* ADDVI.B */
                     DIP("ADDVI.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Add8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ADDVI.H */
                     DIP("ADDVI.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Add16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ADDVI.W */
                     DIP("ADDVI.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Add32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ADDVI.D */
                     DIP("ADDVI.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Add64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x01: { /* SUBVI */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* SUBVI.B */
                     DIP("SUBVI.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Sub8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SUBVI.H */
                     DIP("SUBVI.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Sub16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SUBVI.W */
                     DIP("SUBVI.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Sub32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SUBVI.D */
                     DIP("SUBVI.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Sub64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x02: { /* MAXI_S */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* MAXI_S.B */
                     DIP("MAXI_S.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     char stemp = ((int)tmp << 27) >> 27;
                     tmp = (UChar)stemp;
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2,binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MAXI_S.H */
                     DIP("MAXI_S.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     short stemp = ((int)tmp << 27) >> 27;
                     tmp = (UShort)stemp;
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MAXI_S.W */
                     DIP("MAXI_S.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     int stemp = ((int)tmp << 27) >> 27;
                     tmp = (UInt)stemp;
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MAXI_S.D */
                     DIP("MAXI_S.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     Long stemp = ((Long)tmp << 59) >> 59;
                     tmp = stemp;
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x03: { /* MAXI_U */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* MAXI_U.B */
                     DIP("MAXI_U.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MAXI_U.H */
                     DIP("MAXI_U.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MAXI_U.W */
                     DIP("MAXI_U.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MAXI_U.D */
                     DIP("MAXI_U.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Max64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x04: { /* MINI_S */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* MINI_S.B */
                     DIP("MINI_S.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     char stemp = ((int)tmp << 27) >> 27;
                     tmp = (UChar)stemp;
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MINI_S.H */
                     DIP("MINI_S.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     short stemp = ((int)tmp << 27) >> 27;
                     tmp = (UShort)stemp;
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MINI_S.W */
                     DIP("MINI_S.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     int stemp = ((int)tmp << 27) >> 27;
                     tmp = (UInt)stemp;
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MINI_S.D */
                     DIP("MINI_S.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     Long stemp = ((Long)tmp << 59) >> 59;
                     tmp = stemp;
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x05: { /* MINI_U */
            ULong tmp = wt;

            switch (df) {
               case 0x00: { /* MINI_U.B */
                     DIP("MINI_U.B w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MINI_U.H */
                     DIP("MINI_U.H w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MINI_U.W */
                     DIP("MINI_U.W w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MINI_U.D */
                     DIP("MINI_U.D w%d, w%d, %d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_Min64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      default: {
            return -1;
         }
   }

   return 0;
}

static Int msa_I5_07(UInt cins, UChar wd, UChar ws) { /* I5 (0x07) / I10 */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, i5;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   i5 = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: {
            ULong tmp = i5;

            switch (df) {
               case 0x00: { /* CEQI.B */
                     DIP("CEQI.B w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     char stemp = ((int)tmp << 27) >> 27;
                     tmp = (UChar)stemp;
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpEQ8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CEQI.H */
                     DIP("CEQI.H w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     short stemp = ((int)tmp << 27) >> 27;
                     tmp = (UShort)stemp;
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpEQ16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CEQI.W */
                     DIP("CEQI.W w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     int stemp = ((int)tmp << 27) >> 27;
                     tmp = (UInt)stemp;
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpEQ32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CEQI.D */
                     DIP("CEQI.D w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     Long stemp = ((Long)tmp << 59) >> 59;
                     tmp = stemp;
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpEQ64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x02: { /* CLTI_S.df */
            ULong tmp = i5;

            switch (df) {
               case 0x00: { /* CLTI_S.B */
                     DIP("CLTI_S.B w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     char stemp = ((int)tmp << 27) >> 27;
                     tmp = (UChar)stemp;
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT8Sx16, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLTI_S.H */
                     DIP("CLTI_S.H w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     short stemp = ((int)tmp << 27) >> 27;
                     tmp = (UShort)stemp;
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT16Sx8, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLTI_S.W */
                     DIP("CLTI_S.W w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     int stemp = ((int)tmp << 27) >> 27;
                     tmp = (UInt)stemp;
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT32Sx4, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLTI_S.D */
                     DIP("CLTI_S.D w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     Long stemp = ((Long)tmp << 59) >> 59;
                     tmp = stemp;
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT64Sx2, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* CLTI_U.df */
            ULong tmp = i5;

            switch (df) {
               case 0x00: { /* CLTI_U.B */
                     DIP("CLTI_U.B w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT8Ux16, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLTI_U.H */
                     DIP("CLTI_U.H w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT16Ux8, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLTI_U.W */
                     DIP("CLTI_U.W w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT32Ux4, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLTI_U.D */
                     DIP("CLTI_U.D w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_CmpGT64Ux2, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x04: { /* CLEI_S.df */
            ULong tmp = i5;

            switch (df) {
               case 0x00: { /* CLEI_S.B */
                     DIP("CLEI_S.B w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     char stemp = ((int)tmp << 27) >> 27;
                     tmp = (UChar)stemp;
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128, binop(Iop_CmpGT8Sx16,
                                                      mkexpr(t2), mkexpr(t1)),
                                                  binop(Iop_CmpEQ8x16,
                                                      mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLEI_S.H */
                     DIP("CLEI_S.H w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     short stemp = ((int)tmp << 27) >> 27;
                     tmp = (UShort)stemp;
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128, binop(Iop_CmpGT16Sx8,
                                                      mkexpr(t2), mkexpr(t1)),
                                                  binop(Iop_CmpEQ16x8,
                                                      mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLEI_S.W */
                     DIP("CLEI_S.W w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     int stemp = ((int)tmp << 27) >> 27;
                     tmp = (UInt)stemp;
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT32Sx4,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ32x4,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLEI_S.D */
                     DIP("CLEI_S.D w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     Long stemp = ((Long)tmp << 59) >> 59;
                     tmp = stemp;
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT64Sx2,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ64x2,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* CLEI_U.df */
            ULong tmp = i5;

            switch (df) {
               case 0x00: { /* CLEI_U.B */
                     DIP("CLEI_U.B w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT8Ux16,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ8x16,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLEI_U.H */
                     DIP("CLEI_U.H w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT16Ux8,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ16x8,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLEI_U.W */
                     DIP("CLEI_U.W w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     tmp |= (tmp << 32);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT32Ux4,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ32x4,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLEI_U.D */
                     DIP("CLEI_U.D w%d, w%d, %d", wd, ws, i5);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT64Ux2,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ64x2,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x06: { /* LDI.df */
            ULong tmp;
            UShort s10;
            s10 = (cins & 0x001FF800) >> 11;
            switch (df) {
               case 0x00: /* LDI.B */
                  DIP("LDI.B w%d, %d", wd, s10);
                  tmp = s10 & 0xFFl;
                  tmp = tmp | (tmp << 8) | (tmp << 16) | (tmp << 24)
                        | (tmp << 32) | (tmp << 40) | (tmp << 48) |
                        (tmp << 56);
                  break;

               case 0x01: /* LDI.H */
                  DIP("LDI.H w%d, %d", wd, s10);
                  tmp = extend_s_10to16(s10);
                  tmp = tmp | (tmp << 16) | (tmp << 32) | (tmp << 48);
                  break;

               case 0x02: /* LDI.W */
                  DIP("LDI.W w%d, %d", wd, s10);
                  tmp = extend_s_10to32(s10);
                  tmp = tmp | (tmp << 32);
                  break;

               case 0x03: /* LDI.D */
                  DIP("LDI.D w%d, %d", wd, s10);
                  tmp = extend_s_10to64(s10);
                  break;

               default:
                  return -1;
            }

            putWReg(wd, binop(Iop_64HLtoV128, mkU64(tmp), mkU64(tmp)));
            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_BIT_09(UInt cins, UChar wd, UChar ws) { /* BIT (0x09) */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, m;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x007F0000) >> 16;

   if ((df & 0x70) == 0x70) {        // 111mmmm; b
      m = df & 0x07;
      df = 0;
   } else if ((df & 0x60) == 0x60) { // 110mmmm; h
      m = df & 0x0F;
      df = 1;
   } else if ((df & 0x40) == 0x40) { // 10mmmmm; w
      m = df & 0x1F;
      df = 2;
   } else if ((df & 0x00) == 0x00) { // 0mmmmmm; d
      m = df & 0x3F;
      df = 3;
   }

   switch (operation) {
      case 0x00: { /* SLLI.df */
            switch (df) {
               case 0x00: { /* SLLI.B */
                     DIP("SLLI.B w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShlN8x16, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x01: { /* SLLI.H */
                     DIP("SLLI.H w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShlN16x8, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x02: { /* SLLI.W */
                     DIP("SLLI.W w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShlN32x4, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x03: { /* SLLI.D */
                     DIP("SLLI.D w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShlN64x2, getWReg(ws), mkU8(m)));
                     break;
                  }
            }

            break;
         }

      case 0x01: { /* SRAI.df */
            switch (df) {
               case 0x00: { /* SRAI.B */
                     DIP("SRAI.B w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_SarN8x16, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x01: { /* SRAI.H */
                     DIP("SRAI.H w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_SarN16x8, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x02: { /* SRAI.W */
                     DIP("SRAI.W w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_SarN32x4, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x03: { /* SRAI.D */
                     DIP("SRAI.D w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_SarN64x2, getWReg(ws), mkU8(m)));
                     break;
                  }
            }

            break;
         }

      case 0x02: { /* SRLI.df */
            switch (df) {
               case 0x00: { /* SRLI.B */
                     DIP("SRLI.B w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShrN8x16, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x01: { /* SRLI.H */
                     DIP("SRLI.H w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShrN16x8, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x02: { /* SRLI.W */
                     DIP("SRLI.W w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShrN32x4, getWReg(ws), mkU8(m)));
                     break;
                  }

               case 0x03: { /* SRLI.D */
                     DIP("SRLI.D w%d, w%d, %d", wd, ws, m);
                     putWReg(wd, binop(Iop_ShrN64x2, getWReg(ws), mkU8(m)));
                     break;
                  }
            }

            break;
         }

      case 0x03: { /* BCLRI.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BCLRI.B */
                     DIP("BCLRI.B w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2, binop(Iop_ShlN8x16,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)),mkU8(m)));
                     break;
                  }

               case 0x01: { /* BCLRI.H */
                     DIP("BCLRI.H w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2, binop(Iop_ShlN16x8,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x02: { /* BCLRI.W */
                     DIP("BCLRI.W w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 32);
                     assign(t2, binop(Iop_ShlN32x4,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x03: { /* BCLRI.D */
                     DIP("BCLRI.D w%d, w%d, %d", wd, ws, m);
                     assign(t2, binop(Iop_ShlN64x2,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }
            }

            assign(t3, binop(Iop_AndV128,
                             mkexpr(t1), unop(Iop_NotV128, mkexpr(t2))));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x04: { /* BSETI */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BSETI.B */
                     DIP("BSETI.B w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2, binop(Iop_ShlN8x16,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x01: { /* BSETI.H */
                     DIP("BSETI.H w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2, binop(Iop_ShlN16x8,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x02: { /* BSETI.W */
                     DIP("BSETI.W w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 32);
                     assign(t2, binop(Iop_ShlN32x4,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x03: { /* BSETI.D */
                     DIP("BSETI.D w%d, w%d, %d", wd, ws, m);
                     assign(t2, binop(Iop_ShlN64x2,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }
            }

            assign(t3, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x05: { /* BNEGI.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BNEGI.B */
                     DIP("BNEGI.B w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2, binop(Iop_ShlN8x16,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x01: { /* BNEGI.H */
                     DIP("BNEGI.H w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2, binop(Iop_ShlN16x8,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x02: { /* BNEGI.W */
                     DIP("BNEGI.W w%d, w%d, %d", wd, ws, m);
                     tmp |= (tmp << 32);
                     assign(t2, binop(Iop_ShlN32x4,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }

               case 0x03: { /* BNEGI.D */
                     DIP("BNEGI.D w%d, w%d, %d", wd, ws, m);
                     assign(t2, binop(Iop_ShlN64x2,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     break;
                  }
            }

            assign(t3, binop(Iop_XorV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x06: { /* BINSLI.df */
            switch (df) {
               case 0x00: { /* BINSLI.B */
                     DIP("BINSLI.B w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8080808080808080ULL;
                     assign(t1, binop(Iop_SarN8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)), getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* BINSLI.H */
                     DIP("BINSLI.H w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000800080008000ULL;
                     assign(t1,
                            binop(Iop_SarN16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)), getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* BINSLI.W */
                     DIP("BINSLI.W w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000000080000000ULL;
                     assign(t1,
                            binop(Iop_SarN32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)), getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* BINSLI.D */
                     DIP("BINSLI.D w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000000000000000ULL;
                     assign(t1,
                            binop(Iop_SarN64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)), getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: {
            switch (df) {
               case 0x00: { /* BINSRI.B */
                     DIP("BINSRI.B w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFEFEFEFEFEFEFEFEULL;
                     assign(t1,
                            binop(Iop_ShlN8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)), mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)), getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* BINSRI.H */
                     DIP("BINSRI.H w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFFFEFFFEFFFEFFFEULL;
                     assign(t1,
                            binop(Iop_ShlN16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* BINSRI.W */
                     DIP("BINSRI.W w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFFFFFFFEFFFFFFFEULL;
                     assign(t1,
                            binop(Iop_ShlN32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* BINSRI.D */
                     DIP("BINSRI.D w%d, w%d, w%d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = -2;
                     assign(t1,
                            binop(Iop_ShlN64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_BIT_0A(UInt cins, UChar wd, UChar ws) { /* BIT (0x0A) */
   IRTemp t1, t2;
   UShort operation;
   UChar df, m;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x007F0000) >> 16;

   if ((df & 0x70) == 0x70) {        // 111mmmm; b
      m = df & 0x07;
      df = 0;
   } else if ((df & 0x60) == 0x60) { // 110mmmm; h
      m = df & 0x0F;
      df = 1;
   } else if ((df & 0x40) == 0x40) { // 10mmmmm; w
      m = df & 0x1F;
      df = 2;
   } else if ((df & 0x00) == 0x00) { // 0mmmmmm; d
      m = df & 0x3F;
      df = 3;
   }

   switch (operation) {
      case 0x00: { /* SAT_S.df */
            switch (df) {
               case 0x00: { /* SAT_S.B */
                     DIP("SAT_S.B w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN8x16, getWReg(ws), mkU8(7)));

                     if (m == 0) {
                        putWReg(wd, mkexpr(t1));
                     } else {
                        t2 = newTemp(Ity_V128);
                        assign(t2,
                               binop(Iop_SarN8x16, getWReg(ws), mkU8(m)));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_OrV128,
                                            binop(Iop_AndV128,
                                                  binop(Iop_CmpEQ8x16,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  getWReg(ws)),
                                            binop(Iop_ShlN8x16,
                                                  binop(Iop_CmpGT8Sx16,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(m))),
                                      binop(Iop_ShrN8x16,
                                            binop(Iop_CmpGT8Sx16,
                                                  mkexpr(t2),
                                                  mkexpr(t1)),
                                            mkU8(8 - m))));
                     }

                     break;
                  }

               case 0x01: { /* SAT_S.H */
                     DIP("SAT_S.H w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN16x8, getWReg(ws), mkU8(15)));

                     if (m == 0) {
                        putWReg(wd, mkexpr(t1));
                     } else {
                        t2 = newTemp(Ity_V128);
                        assign(t2,
                               binop(Iop_SarN16x8,
                                     getWReg(ws),
                                     mkU8(m)));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_OrV128,
                                            binop(Iop_AndV128,
                                                  binop(Iop_CmpEQ16x8,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  getWReg(ws)),
                                            binop(Iop_ShlN16x8,
                                                  binop(Iop_CmpGT16Sx8,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(m))),
                                      binop(Iop_ShrN16x8,
                                            binop(Iop_CmpGT16Sx8,
                                                  mkexpr(t2),
                                                  mkexpr(t1)),
                                            mkU8(16 - m))));
                     }

                     break;
                  }

               case 0x02: { /* SAT_S.W */
                     DIP("SAT_S.W w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN32x4, getWReg(ws), mkU8(31)));

                     if (m == 0) {
                        putWReg(wd, mkexpr(t1));
                     } else {
                        t2 = newTemp(Ity_V128);
                        assign(t2,
                               binop(Iop_SarN32x4,
                                     getWReg(ws),
                                     mkU8(m)));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_OrV128,
                                            binop(Iop_AndV128,
                                                  binop(Iop_CmpEQ32x4,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  getWReg(ws)),
                                            binop(Iop_ShlN32x4,
                                                  binop(Iop_CmpGT32Sx4,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(m))),
                                      binop(Iop_ShrN32x4,
                                            binop(Iop_CmpGT32Sx4,
                                                  mkexpr(t2),
                                                  mkexpr(t1)),
                                            mkU8(32 - m))));
                     }

                     break;
                  }

               case 0x03: { /* SAT_S.D */
                     DIP("SAT_S.D w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN64x2, getWReg(ws), mkU8(63)));

                     if (m == 0) {
                        putWReg(wd, mkexpr(t1));
                     } else {
                        t2 = newTemp(Ity_V128);
                        assign(t2,
                               binop(Iop_SarN64x2,
                                     getWReg(ws),
                                     mkU8(m)));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_OrV128,
                                            binop(Iop_AndV128,
                                                  binop(Iop_CmpEQ64x2,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  getWReg(ws)),
                                            binop(Iop_ShlN64x2,
                                                  binop(Iop_CmpGT64Sx2,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(m))),
                                      binop(Iop_ShrN64x2,
                                            binop(Iop_CmpGT64Sx2,
                                                  mkexpr(t2),
                                                  mkexpr(t1)),
                                            mkU8(64 - m))));
                     }

                     break;
                  }
            }

            break;
         }

      case 0x01: { /* SAT_U.df */
            switch (df) {
               case 0x00: { /* SAT_U.B */
                     DIP("SAT_U.B w%d, w%d, %d", wd, ws, m);

                     if (m == 7) {
                        putWReg(wd, getWReg(ws));
                     } else {
                        t1 = newTemp(Ity_V128);
                        assign(t1,
                               binop(Iop_CmpEQ8x16,
                                     binop(Iop_ShrN8x16,
                                           getWReg(ws),
                                           mkU8(m + 1)),
                                     binop(Iop_64HLtoV128,
                                           mkU64(0), mkU64(0))));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      binop(Iop_ShrN8x16,
                                            unop(Iop_NotV128,
                                                 mkexpr(t1)),
                                            mkU8(7 - m))));
                     }

                     break;
                  }

               case 0x01: { /* SAT_U.H */
                     DIP("SAT_U.H w%d, w%d, %d", wd, ws, m);

                     if (m == 15) {
                        putWReg(wd, getWReg(ws));
                     } else {
                        t1 = newTemp(Ity_V128);
                        assign(t1,
                               binop(Iop_CmpEQ16x8,
                                     binop(Iop_ShrN16x8,
                                           getWReg(ws),
                                           mkU8(m + 1)),
                                     binop(Iop_64HLtoV128,
                                           mkU64(0), mkU64(0))));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      binop(Iop_ShrN16x8,
                                            unop(Iop_NotV128,
                                                 mkexpr(t1)),
                                            mkU8(15 - m))));
                     }

                     break;
                  }

               case 0x02: { /* SAT_U.W */
                     DIP("SAT_U.W w%d, w%d, %d", wd, ws, m);

                     if (m == 31) {
                        putWReg(wd, getWReg(ws));
                     } else {
                        t1 = newTemp(Ity_V128);
                        assign(t1,
                               binop(Iop_CmpEQ32x4,
                                     binop(Iop_ShrN32x4,
                                           getWReg(ws),
                                           mkU8(m + 1)),
                                     binop(Iop_64HLtoV128,
                                           mkU64(0), mkU64(0))));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t1), \
                                            getWReg(ws)),
                                      binop(Iop_ShrN32x4,
                                            unop(Iop_NotV128,
                                                 mkexpr(t1)),
                                            mkU8(31 - m))));
                     }

                     break;
                  }

               case 0x03: { /* SAT_U.D */
                     DIP("SAT_U.D w%d, w%d, %d", wd, ws, m);

                     if (m == 63) {
                        putWReg(wd, getWReg(ws));
                     } else {
                        t1 = newTemp(Ity_V128);
                        assign(t1,
                               binop(Iop_CmpEQ64x2,
                                     binop(Iop_ShrN64x2,
                                           getWReg(ws),
                                           mkU8(m + 1)),
                                     binop(Iop_64HLtoV128,
                                           mkU64(0), mkU64(0))));
                        putWReg(wd,
                                binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      binop(Iop_ShrN64x2,
                                            unop(Iop_NotV128,
                                                 mkexpr(t1)),
                                            mkU8(63 - m))));
                     }

                     break;
                  }
            }

            break;
         }

      case 0x02: { /* SRARI.df */
            switch (df) {
               case 0x00: { /* SRARI.B */
                     DIP("SRARI.B w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_SarN8x16,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN8x16,
                                  binop(Iop_ShlN8x16,
                                        getWReg(ws),
                                        mkU8(8 - m)),
                                  mkU8(7)));

                     if (m) putWReg(wd, binop(Iop_Add8x16,
                                              mkexpr(t1),
                                              mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x01: { /* SRARI.H */
                     DIP("SRARI.H w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_SarN16x8,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN16x8,
                                  binop(Iop_ShlN16x8,
                                        getWReg(ws),
                                        mkU8(16 - m)),
                                  mkU8(15)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add16x8,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x02: { /* SRARI.W */
                     DIP("SRARI.W w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_SarN32x4,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN32x4,
                                  binop(Iop_ShlN32x4,
                                        getWReg(ws),
                                        mkU8(32 - m)),
                                  mkU8(31)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add32x4,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x03: { /* SRARI.D */
                     DIP("SRARI.D w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_SarN64x2,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN64x2,
                                  binop(Iop_ShlN64x2,
                                        getWReg(ws),
                                        mkU8(64 - m)),
                                  mkU8(63)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add64x2,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }
            }

            break;
         }

      case 0x03: { /* SRLRI.df */
            switch (df) {
               case 0x00: { /* SRLRI.B */
                     DIP("SRLRI.B w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_ShrN8x16,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN8x16,
                                  binop(Iop_ShlN8x16,
                                        getWReg(ws),
                                        mkU8(8 - m)),
                                  mkU8(7)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add8x16,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x01: { /* SRLRI.H */
                     DIP("SRLRI.H w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_ShrN16x8,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN16x8,
                                  binop(Iop_ShlN16x8,
                                        getWReg(ws),
                                        mkU8(16 - m)),
                                  mkU8(15)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add16x8,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x02: { /* SRLRI.W */
                     DIP("SRLRI.W w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_ShrN32x4,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN32x4,
                                  binop(Iop_ShlN32x4,
                                        getWReg(ws),
                                        mkU8(32 - m)),
                                  mkU8(31)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add32x4,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }

               case 0x03: { /* SRLRI.D */
                     DIP("SRLRI.D w%d, w%d, %d", wd, ws, m);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_ShrN64x2,
                                  getWReg(ws),
                                  mkU8(m)));
                     assign(t2,
                            binop(Iop_ShrN64x2,
                                  binop(Iop_ShlN64x2,
                                        getWReg(ws),
                                        mkU8(64 - m)),
                                  mkU8(63)));

                     if (m)
                        putWReg(wd,
                                binop(Iop_Add64x2,
                                      mkexpr(t1), mkexpr(t2)));
                     else putWReg(wd, mkexpr(t1));

                     break;
                  }
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_0D(UInt cins, UChar wd, UChar ws) { /* 3R (0x0D) */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* SLL.df */
            switch (df) {
               case 0x00: { /* SLL.B */
                     DIP("SLL.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shl8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SLL.H */
                     DIP("SLL.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shl16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SLL.W */
                     DIP("SLL.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shl32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SLL.D */
                     DIP("SLL.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shl64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* SRA.df */
            switch (df) {
               case 0x00: { /* SRA.B */
                     DIP("SRA.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sar8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SRA.H */
                     DIP("SRA.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sar16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SRA.W */
                     DIP("SRA.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sar32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SRA.D */
                     DIP("SRA.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sar64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* SRL.df */
            switch (df) {
               case 0x00: { /* SRL.B */
                     DIP("SRL.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shr8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SRL.H */
                     DIP("SRL.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shr16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SRL.W */
                     DIP("SRL.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shr32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SRL.D */
                     DIP("SRL.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Shr64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* BCLR.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BCLR.B */
                     DIP("BCLR.B w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2, binop(Iop_Shl8x16,
                                      binop(Iop_64HLtoV128,
                                            mkU64(tmp), mkU64(tmp)),
                                      getWReg(wt)));
                     break;
                  }

               case 0x01: { /* BCLR.H */
                     DIP("BCLR.H w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2,
                            binop(Iop_Shl16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x02: { /* BCLR.W */
                     DIP("BCLR.W w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 32);
                     assign(t2,
                            binop(Iop_Shl32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x03: { /* BCLR.D */
                     DIP("BCLR.D w%d, w%d, w%d", wd, ws, wt);
                     assign(t2,
                            binop(Iop_Shl64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }
            }

            assign(t3,
                   binop(Iop_AndV128,
                         mkexpr(t1), unop(Iop_NotV128, mkexpr(t2))));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x04: { /* BSET.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BSET.B */
                     DIP("BSET.B w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2,
                            binop(Iop_Shl8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x01: { /* BSET.H */
                     DIP("BSET.H w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2,
                            binop(Iop_Shl16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x02: { /* BSET.W */
                     DIP("BSET.W w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 32);
                     assign(t2,
                            binop(Iop_Shl32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x03: { /* BSET.D */
                     DIP("BSET.D w%d, w%d, w%d", wd, ws, wt);
                     assign(t2,
                            binop(Iop_Shl64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }
            }

            assign(t3, binop(Iop_OrV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x05: { /* BNEG.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            ULong tmp = 1;
            assign(t1, getWReg(ws));

            switch (df) {
               case 0x00: { /* BNEG.B */
                     DIP("BNEG.B w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 56) | (tmp << 48) | (tmp << 40) |
                            (tmp << 32) | (tmp << 24) | (tmp << 16) |
                            (tmp << 8);
                     assign(t2,
                            binop(Iop_Shl8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x01: { /* BNEG.H */
                     DIP("BNEG.H w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 48) | (tmp << 32) | (tmp << 16);
                     assign(t2,
                            binop(Iop_Shl16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x02: { /* BNEG.W */
                     DIP("BNEG.W w%d, w%d, w%d", wd, ws, wt);
                     tmp |= (tmp << 32);
                     assign(t2,
                            binop(Iop_Shl32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }

               case 0x03: { /* BNEG.D */
                     DIP("BNEG.D w%d, w%d, w%d", wd, ws, wt);
                     assign(t2,
                            binop(Iop_Shl64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     break;
                  }
            }

            assign(t3, binop(Iop_XorV128, mkexpr(t1), mkexpr(t2)));
            putWReg(wd, mkexpr(t3));
            break;
         }

      case 0x06: { /* BINSL.df */
            switch (df) {
               case 0x00: { /* BINSL.B */
                     DIP("BINSL.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8080808080808080ULL;
                     assign(t1,
                            binop(Iop_Sar8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* BINSL.H */
                     DIP("BINSL.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000800080008000ULL;
                     assign(t1,
                            binop(Iop_Sar16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* BINSL.W */
                     DIP("BINSL.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000000080000000ULL;
                     assign(t1,
                            binop(Iop_Sar32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* BINSL.D */
                     DIP("BINSL.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0x8000000000000000ULL;
                     assign(t1,
                            binop(Iop_Sar64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(wd)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(ws)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* BINSR.df */
            switch (df) {
               case 0x00: { /* BINSR.B */
                     DIP("BINSR.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFEFEFEFEFEFEFEFEULL;
                     assign(t1,
                            binop(Iop_Shl8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* BINSR.H */
                     DIP("BINSR.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFFFEFFFEFFFEFFFEULL;
                     assign(t1,
                            binop(Iop_Shl16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* BINSR.W */
                     DIP("BINSR.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = 0xFFFFFFFEFFFFFFFEULL;
                     assign(t1,
                            binop(Iop_Shl32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* BINSR.D */
                     DIP("BINSR.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     ULong tmp = -2;
                     assign(t1,
                            binop(Iop_Shl64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(tmp), mkU64(tmp)),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_AndV128,
                                  unop(Iop_NotV128, mkexpr(t1)),
                                  getWReg(ws)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  mkexpr(t1), getWReg(wd)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_0E(UInt cins, UChar wd, UChar ws) { /* 3R (0x0E) */
   IRTemp t1, t2, t3, t4;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* ADDV.df */
            switch (df) {
               case 0x00: { /* ADDV.B */
                     DIP("ADDV.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ADDV.H */
                     DIP("ADDV.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ADDV.W */
                     DIP("ADDV.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ADDV.D */
                     DIP("ADDV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* SUBV.df */
            switch (df) {
               case 0x00: { /* SUBV.B */
                     DIP("SUBV.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sub8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SUBV.H */
                     DIP("SUBV.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sub16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SUBV.W */
                     DIP("SUBV.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sub32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SUBV.D */
                     DIP("SUBV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Sub64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* MAX_S.df */
            switch (df) {
               case 0x00: { /* MAX_S.B */
                     DIP("MAX_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MAX_S.H */
                     DIP("MAX_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MAX_S.W */
                     DIP("MAX_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MAX_S.D */
                     DIP("MAX_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* MAX_U.df */
            switch (df) {
               case 0x00: { /* MAX_U.B */
                     DIP("MAX_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MAX_U.H */
                     DIP("MAX_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MAX_U.W */
                     DIP("MAX_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MAX_U.D */
                     DIP("MAX_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Max64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* MIN_S.df */
            switch (df) {
               case 0x00: { /* MIN_S.B */
                     DIP("MIN_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MIN_S.H */
                     DIP("MIN_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MIN_S.W */
                     DIP("MIN_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MIN_S.D */
                     DIP("MIN_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* MIN_U.df */
            switch (df) {
               case 0x00: { /* MIN_U.B */
                     DIP("MIN_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MIN_U.H */
                     DIP("MIN_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MIN_U.W */
                     DIP("MIN_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MIN_U.D */
                     DIP("MIN_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Min64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x06: { /* MAX_A.df */
            switch (df) {
               case 0x00: { /* MAX_A.B */
                     DIP("MAX_A.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs8x16, getWReg(ws)));
                     assign(t2, unop(Iop_Abs8x16, getWReg(wt)));
                     assign(t4, binop(Iop_CmpGT8Ux16, mkexpr(t1), mkexpr(t2)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(ws)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(wt))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MAX_A.H */
                     DIP("MAX_A.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs16x8, getWReg(ws)));
                     assign(t2, unop(Iop_Abs16x8, getWReg(wt)));
                     assign(t4, binop(Iop_CmpGT16Ux8, mkexpr(t1), mkexpr(t2)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(ws)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(wt))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MAX_A.W */
                     DIP("MAX_A.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs32x4, getWReg(ws)));
                     assign(t2, unop(Iop_Abs32x4, getWReg(wt)));
                     assign(t4, binop(Iop_CmpGT32Ux4, mkexpr(t1), mkexpr(t2)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(ws)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(wt))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MAX_A.D */
                     DIP("MAX_A.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs64x2, getWReg(ws)));
                     assign(t2, unop(Iop_Abs64x2, getWReg(wt)));
                     assign(t4, binop(Iop_CmpGT64Ux2, mkexpr(t1), mkexpr(t2)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(ws)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(wt))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* MIN_A.df */
            switch (df) {
               case 0x00: { /* MIN_A.B */
                     DIP("MIN_A.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs8x16, getWReg(ws)));
                     assign(t2, unop(Iop_Abs8x16, getWReg(wt)));
                     assign(t4, binop(Iop_OrV128,
                                      binop(Iop_CmpGT8Ux16,
                                            mkexpr(t1), mkexpr(t2)),
                                      binop(Iop_CmpEQ8x16,
                                            mkexpr(t1), mkexpr(t2))));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(wt)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(ws))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* MIN_A.H */
                     DIP("MIN_A.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs16x8, getWReg(ws)));
                     assign(t2, unop(Iop_Abs16x8, getWReg(wt)));
                     assign(t4, binop(Iop_OrV128,
                                      binop(Iop_CmpGT16Ux8,
                                            mkexpr(t1), mkexpr(t2)),
                                      binop(Iop_CmpEQ16x8,
                                            mkexpr(t1), mkexpr(t2))));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(wt)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(ws))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* MIN_A.W */
                     DIP("MIN_A.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs32x4, getWReg(ws)));
                     assign(t2, unop(Iop_Abs32x4, getWReg(wt)));
                     assign(t4, binop(Iop_OrV128,
                                      binop(Iop_CmpGT32Ux4,
                                            mkexpr(t1), mkexpr(t2)),
                                      binop(Iop_CmpEQ32x4,
                                            mkexpr(t1), mkexpr(t2))));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(wt)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(ws))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* MIN_A.D */
                     DIP("MIN_A.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs64x2, getWReg(ws)));
                     assign(t2, unop(Iop_Abs64x2, getWReg(wt)));
                     assign(t4, binop(Iop_OrV128,
                                      binop(Iop_CmpGT64Ux2,
                                            mkexpr(t1), mkexpr(t2)),
                                      binop(Iop_CmpEQ64x2,
                                            mkexpr(t1), mkexpr(t2))));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_AndV128,
                                            mkexpr(t4),
                                            getWReg(wt)),
                                      binop(Iop_AndV128,
                                            unop(Iop_NotV128, mkexpr(t4)),
                                            getWReg(ws))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_0F(UInt cins, UChar wd, UChar ws) { /* 3R (0x0F) */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* CEQ.df */
            switch (df) {
               case 0x00: { /* CEQ.B */
                     DIP("CEQ.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpEQ8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CEQ.H */
                     DIP("CEQ.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpEQ16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CEQ.W */
                     DIP("CEQ.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpEQ32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CEQ.D */
                     DIP("CEQ.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpEQ64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* CLT_S.df */
            switch (df) {
               case 0x00: { /* CLT_S.B */
                     DIP("CLT_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT8Sx16, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLT_S.H */
                     DIP("CLT_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT16Sx8, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLT_S.W */
                     DIP("CLT_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT32Sx4, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLT_S.D */
                     DIP("CLT_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT64Sx2, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* CLT_U.df */
            switch (df) {
               case 0x00: { /* CLT_U.B */
                     DIP("CLT_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT8Ux16, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLT_U.H */
                     DIP("CLT_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT16Ux8, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLT_U.W */
                     DIP("CLT_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT32Ux4, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLT_U.D */
                     DIP("CLT_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_CmpGT64Ux2, mkexpr(t2), mkexpr(t1)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* CLE_S.df */
            switch (df) {
               case 0x00: { /* CLE_S.B */
                     DIP("CLE_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT8Sx16,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ8x16,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLE_S.H */
                     DIP("CLE_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT16Sx8,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ16x8,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLE_S.W */
                     DIP("CLE_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT32Sx4,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ32x4,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLE_S.D */
                     DIP("CLE_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT64Sx2,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ64x2,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* CLE_U.df */
            switch (df) {
               case 0x00: { /* CLE_U.B */
                     DIP("CLE_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT8Ux16,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ8x16,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* CLE_U.H */
                     DIP("CLE_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT16Ux8,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ16x8,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* CLE_U.W */
                     DIP("CLE_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT32Ux4,
                                            mkexpr(t2), mkexpr(t1)),
                                      binop(Iop_CmpEQ32x4,
                                            mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* CLE_U.D */
                     DIP("CLE_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_OrV128,
                                  binop(Iop_CmpGT64Ux2,
                                        mkexpr(t2), mkexpr(t1)),
                                  binop(Iop_CmpEQ64x2,
                                        mkexpr(t1), mkexpr(t2))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_10(UInt cins, UChar wd, UChar ws) { /* 3R (0x10) */
   IRTemp t1, t2, t3, t4;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* ADD_A.df */
            switch (df) {
               case 0x00: { /* ADD_A.B */
                     DIP("ADD_A.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs8x16, getWReg(ws)));
                     assign(t2, unop(Iop_Abs8x16, getWReg(wt)));
                     assign(t3, binop(Iop_Add8x16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ADD_A.H */
                     DIP("ADD_A.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs16x8, getWReg(ws)));
                     assign(t2, unop(Iop_Abs16x8, getWReg(wt)));
                     assign(t3, binop(Iop_Add16x8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ADD_A.W */
                     DIP("ADD_A.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs32x4, getWReg(ws)));
                     assign(t2, unop(Iop_Abs32x4, getWReg(wt)));
                     assign(t3, binop(Iop_Add32x4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ADD_A.D */
                     DIP("ADD_A.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs64x2, getWReg(ws)));
                     assign(t2, unop(Iop_Abs64x2, getWReg(wt)));
                     assign(t3, binop(Iop_Add64x2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* ADDS_A.df */
            switch (df) {
               case 0x00: { /* ADDS_A.B */
                     DIP("ADDS_A.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs8x16, getWReg(ws)));
                     assign(t2, unop(Iop_Abs8x16, getWReg(wt)));
                     assign(t3, binop(Iop_SarN8x16,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      mkU8(7)));
                     assign(t4, binop(Iop_SarN8x16,
                                      binop(Iop_AndV128,
                                            mkexpr(t2),
                                            getWReg(wt)),
                                      mkU8(7)));
                     putWReg(wd, binop(Iop_QAdd8Sx16,
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t3)),
                                                   mkexpr(t1)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t1)),
                                                   mkexpr(t3))),
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t4)),
                                                   mkexpr(t2)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t2)),
                                                   mkexpr(t4)))));
                     break;
                  }

               case 0x01: { /* ADDS_A.H */
                     DIP("ADDS_A.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs16x8, getWReg(ws)));
                     assign(t2, unop(Iop_Abs16x8, getWReg(wt)));
                     assign(t3, binop(Iop_SarN16x8,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      mkU8(15)));
                     assign(t4, binop(Iop_SarN16x8,
                                   binop(Iop_AndV128,
                                         mkexpr(t2),
                                         getWReg(wt)),
                                   mkU8(15)));
                     putWReg(wd, binop(Iop_QAdd16Sx8,
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t3)),
                                                   mkexpr(t1)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t1)),
                                                   mkexpr(t3))),
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t4)),
                                                   mkexpr(t2)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t2)),
                                                   mkexpr(t4)))));
                     break;
                  }

               case 0x02: { /* ADDS_A.W */
                     DIP("ADDS_A.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs32x4, getWReg(ws)));
                     assign(t2, unop(Iop_Abs32x4, getWReg(wt)));
                     assign(t3, binop(Iop_SarN32x4,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      mkU8(31)));
                     assign(t4, binop(Iop_SarN32x4,
                                      binop(Iop_AndV128,
                                            mkexpr(t2),
                                            getWReg(wt)),
                                      mkU8(31)));
                     putWReg(wd, binop(Iop_QAdd32Sx4,
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t3)),
                                                   mkexpr(t1)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t1)),
                                                   mkexpr(t3))),
                                       binop(Iop_OrV128,
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t4)),
                                                   mkexpr(t2)),
                                             binop(Iop_AndV128,
                                                   unop(Iop_NotV128,
                                                        mkexpr(t2)),
                                                   mkexpr(t4)))));
                     break;
                  }

               case 0x03: { /* ADDS_A.D */
                     DIP("ADDS_A.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1, unop(Iop_Abs64x2, getWReg(ws)));
                     assign(t2, unop(Iop_Abs64x2, getWReg(wt)));
                     assign(t3, binop(Iop_SarN64x2,
                                      binop(Iop_AndV128,
                                            mkexpr(t1),
                                            getWReg(ws)),
                                      mkU8(63)));
                     assign(t4, binop(Iop_SarN64x2,
                                      binop(Iop_AndV128,
                                            mkexpr(t2),
                                            getWReg(wt)),
                                      mkU8(63)));
                     putWReg(wd,
                             binop(Iop_QAdd64Sx2,
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    mkexpr(t3)),
                                               mkexpr(t1)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    mkexpr(t1)),
                                               mkexpr(t3))),
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    mkexpr(t4)),
                                               mkexpr(t2)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    mkexpr(t2)),
                                               mkexpr(t4)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* ADDS_S.df */
            switch (df) {
               case 0x00: { /* ADDS_S.B */
                     DIP("ADDS_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ADDS_S.H */
                     DIP("ADDS_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ADDS_S.W */
                     DIP("ADDS_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ADDS_S.D */
                     DIP("ADDS_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* ADDS_U.df */
            switch (df) {
               case 0x00: { /* ADDS_U.B */
                     DIP("ADDS_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ADDS_U.H */
                     DIP("ADDS_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ADDS_U.W */
                     DIP("ADDS_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ADDS_U.D */
                     DIP("ADDS_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QAdd64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* AVE_S.df */
            switch (df) {
               case 0x00: { /* AVE_S.B */
                     DIP("AVE_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add8x16,
                                      binop(Iop_Add8x16,
                                            binop(Iop_SarN8x16,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_SarN8x16,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN8x16,
                                            binop(Iop_ShlN8x16,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(7)),
                                            mkU8(7))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* AVE_S.H */
                     DIP("AVE_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add16x8,
                                  binop(Iop_Add16x8,
                                        binop(Iop_SarN16x8,
                                              mkexpr(t1), mkU8(1)),
                                        binop(Iop_SarN16x8,
                                              mkexpr(t2), mkU8(1))),
                                  binop(Iop_ShrN16x8,
                                        binop(Iop_ShlN16x8,
                                              binop(Iop_AndV128,
                                                    mkexpr(t1),
                                                    mkexpr(t2)),
                                              mkU8(15)),
                                        mkU8(15))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* AVE_S.W */
                     DIP("AVE_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add32x4,
                                      binop(Iop_Add32x4,
                                            binop(Iop_SarN32x4,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_SarN32x4,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN32x4,
                                            binop(Iop_ShlN32x4,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(31)),
                                            mkU8(31))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* AVE_S.D */
                     DIP("AVE_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add64x2,
                                      binop(Iop_Add64x2,
                                            binop(Iop_SarN64x2,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_SarN64x2,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN64x2,
                                            binop(Iop_ShlN64x2,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(63)),
                                            mkU8(63))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* AVE_U.df */
            switch (df) {
               case 0x00: { /* AVE_U.B */
                     DIP("AVE_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add16x8,
                                      binop(Iop_Add8x16,
                                            binop(Iop_ShrN8x16,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_ShrN8x16,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN8x16,
                                            binop(Iop_ShlN8x16,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(7)),
                                            mkU8(7))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* AVE_U.H */
                     DIP("AVE_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add16x8,
                                      binop(Iop_Add16x8,
                                            binop(Iop_ShrN16x8,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_ShrN16x8,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN16x8,
                                            binop(Iop_ShlN16x8,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(15)),
                                            mkU8(15))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* AVE_U.W */
                     DIP("AVE_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add32x4,
                                      binop(Iop_Add32x4,
                                            binop(Iop_ShrN32x4,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_ShrN32x4,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN32x4,
                                            binop(Iop_ShlN32x4,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(31)),
                                            mkU8(31))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* AVE_U.D */
                     DIP("AVE_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add64x2,
                                      binop(Iop_Add64x2,
                                            binop(Iop_ShrN64x2,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_ShrN64x2,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN64x2,
                                            binop(Iop_ShlN64x2,
                                                  binop(Iop_AndV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(63)),
                                            mkU8(63))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x06: { /* AVER_S.df */
            switch (df) {
               case 0x00: { /* AVER_S.B */
                     DIP("AVER_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* AVER_S.H */
                     DIP("AVER_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* AVER_S.W */
                     DIP("AVER_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* AVER_S.D */
                     DIP("AVER_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add64x2,
                                      binop(Iop_Add64x2,
                                            binop(Iop_SarN64x2,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_SarN64x2,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN64x2,
                                            binop(Iop_ShlN64x2,
                                                  binop(Iop_OrV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(63)),
                                            mkU8(63))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* AVER_U.df */
            switch (df) {
               case 0x00: { /* AVER_U.B */
                     DIP("AVER_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* AVER_U.H */
                     DIP("AVER_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* AVER_U.W */
                     DIP("AVER_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Avg32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* AVER_U.D */
                     DIP("AVER_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_Add64x2,
                                      binop(Iop_Add64x2,
                                            binop(Iop_ShrN64x2,
                                                  mkexpr(t1), mkU8(1)),
                                            binop(Iop_ShrN64x2,
                                                  mkexpr(t2), mkU8(1))),
                                      binop(Iop_ShrN64x2,
                                            binop(Iop_ShlN64x2,
                                                  binop(Iop_OrV128,
                                                        mkexpr(t1),
                                                        mkexpr(t2)),
                                                  mkU8(63)),
                                            mkU8(63))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_11(UInt cins, UChar wd, UChar ws) { /* 3R (0x11) */
   IRTemp t1, t2, t3;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* SUBS_S.df */
            switch (df) {
               case 0x00: { /* SUBS_S.B */
                     DIP("SUBS_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub8Sx16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SUBS_S.H */
                     DIP("SUBS_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub16Sx8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SUBS_S.W */
                     DIP("SUBS_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub32Sx4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SUBS_S.D */
                     DIP("SUBS_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub64Sx2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* SUBS_U.df */
            switch (df) {
               case 0x00: { /* SUBS_U.B */
                     DIP("SUBS_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub8Ux16, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* SUBS_U.H */
                     DIP("SUBS_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub16Ux8, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* SUBS_U.W */
                     DIP("SUBS_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub32Ux4, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* SUBS_U.D */
                     DIP("SUBS_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3, binop(Iop_QSub64Ux2, mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* SUBSUS_U.df */
            switch (df) {
               case 0x00: { /* SUBSUS_U.B */
                     DIP("SUBSUS_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub8x16, getWReg(ws), getWReg(wt)));
                     assign(t2, binop(Iop_SarN8x16, getWReg(wt), mkU8(7)));
                     assign(t3, binop(Iop_OrV128,
                                      binop(Iop_CmpGT8Ux16,
                                            getWReg(ws),
                                            getWReg(wt)),
                                      binop(Iop_CmpEQ8x16,
                                            getWReg(ws),
                                            getWReg(wt))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t3), mkexpr(t2)),
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         binop(Iop_XorV128,
                                               mkexpr(t3),
                                               mkexpr(t2)))));
                     break;
                  }

               case 0x01: { /* SUBSUS_U.H */
                     DIP("SUBSUS_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub16x8, getWReg(ws), getWReg(wt)));
                     assign(t2, binop(Iop_SarN16x8, getWReg(wt), mkU8(15)));
                     assign(t3,
                            binop(Iop_OrV128,
                                  binop(Iop_CmpGT16Ux8,
                                        getWReg(ws),
                                        getWReg(wt)),
                                  binop(Iop_CmpEQ16x8,
                                        getWReg(ws),
                                        getWReg(wt))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t3), mkexpr(t2)),
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         binop(Iop_XorV128,
                                               mkexpr(t3),
                                               mkexpr(t2)))));
                     break;
                  }

               case 0x02: { /* SUBSUS_U.W */
                     DIP("SUBSUS_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub32x4, getWReg(ws), getWReg(wt)));
                     assign(t2, binop(Iop_SarN32x4, getWReg(wt), mkU8(31)));
                     assign(t3,
                            binop(Iop_OrV128,
                                  binop(Iop_CmpGT32Ux4,
                                        getWReg(ws),
                                        getWReg(wt)),
                                  binop(Iop_CmpEQ32x4,
                                        getWReg(ws),
                                        getWReg(wt))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t3), mkexpr(t2)),
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         binop(Iop_XorV128,
                                               mkexpr(t3),
                                               mkexpr(t2)))));
                     break;
                  }

               case 0x03: { /* SUBSUS_U.D */
                     DIP("SUBSUS_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub64x2, getWReg(ws), getWReg(wt)));
                     assign(t2, binop(Iop_SarN64x2, getWReg(wt), mkU8(63)));
                     assign(t3,
                            binop(Iop_OrV128,
                                  binop(Iop_CmpGT64Ux2,
                                        getWReg(ws),
                                        getWReg(wt)),
                                  binop(Iop_CmpEQ64x2,
                                        getWReg(ws),
                                        getWReg(wt))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t3), mkexpr(t2)),
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         binop(Iop_XorV128,
                                               mkexpr(t3),
                                               mkexpr(t2)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* SUBSUU_S.df */
            switch (df) {
               case 0x00: { /* SUBSUU_S.B */
                     DIP("SUBSUU_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub8x16, getWReg(ws), getWReg(wt)));
                     assign(t2,
                            binop(Iop_SarN8x16,
                                  binop (Iop_AndV128,
                                         binop(Iop_XorV128,
                                               getWReg(ws),
                                               getWReg(wt)),
                                         binop(Iop_XorV128,
                                               mkexpr(t1),
                                               getWReg(wt))),
                                  mkU8(7)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  binop(Iop_SarN8x16,
                                        getWReg(ws), mkU8(7)),
                                  mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         unop(Iop_NotV128,
                                              mkexpr(t2))),
                                   binop(Iop_XorV128,
                                         binop(Iop_ShlN8x16,
                                               mkexpr(t2), mkU8(7)),
                                         mkexpr(t3))));
                     break;
                  }

               case 0x01: { /* SUBSUU_S.H */
                     DIP("SUBSUU_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub16x8, getWReg(ws), getWReg(wt)));
                     assign(t2,
                            binop(Iop_SarN16x8,
                                  binop (Iop_AndV128,
                                         binop(Iop_XorV128,
                                               getWReg(ws),
                                               getWReg(wt)),
                                         binop(Iop_XorV128,
                                               mkexpr(t1),
                                               getWReg(wt))),
                                  mkU8(15)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  binop(Iop_SarN16x8,
                                        getWReg(ws),
                                        mkU8(15)),
                                  mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         unop(Iop_NotV128,
                                              mkexpr(t2))),
                                   binop(Iop_XorV128,
                                         binop(Iop_ShlN16x8,
                                               mkexpr(t2), mkU8(15)),
                                         mkexpr(t3))));
                     break;
                  }

               case 0x02: { /* SUBSUU_S.W */
                     DIP("SUBSUU_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub32x4, getWReg(ws), getWReg(wt)));
                     assign(t2,
                            binop(Iop_SarN32x4,
                                  binop (Iop_AndV128,
                                         binop(Iop_XorV128,
                                               getWReg(ws),
                                               getWReg(wt)),
                                         binop(Iop_XorV128,
                                               mkexpr(t1),
                                               getWReg(wt))),
                                  mkU8(31)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  binop(Iop_SarN32x4,
                                        getWReg(ws),
                                        mkU8(31)),
                                  mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         unop(Iop_NotV128,
                                              mkexpr(t2))),
                                   binop(Iop_XorV128,
                                         binop(Iop_ShlN32x4,
                                               mkexpr(t2),
                                               mkU8(31)),
                                         mkexpr(t3))));
                     break;
                  }

               case 0x03: { /* SUBSUU_S.D */
                     DIP("SUBSUU_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_Sub64x2, getWReg(ws), getWReg(wt)));
                     assign(t2,
                            binop(Iop_SarN64x2,
                                  binop (Iop_AndV128,
                                         binop(Iop_XorV128,
                                               getWReg(ws),
                                               getWReg(wt)),
                                         binop(Iop_XorV128,
                                               mkexpr(t1),
                                               getWReg(wt))),
                                  mkU8(63)));
                     assign(t3,
                            binop(Iop_AndV128,
                                  binop(Iop_SarN64x2,
                                        getWReg(ws),
                                        mkU8(63)),
                                  mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         mkexpr(t1),
                                         unop(Iop_NotV128,
                                              mkexpr(t2))),
                                   binop(Iop_XorV128,
                                         binop(Iop_ShlN64x2,
                                               mkexpr(t2), mkU8(63)),
                                         mkexpr(t3))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* ASUB_S.df */
            switch (df) {
               case 0x00: { /* ASUB_S.B */
                     DIP("ASUB_S.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN8x16, getWReg(ws), mkU8(7)));
                     assign(t2, binop(Iop_SarN8x16, getWReg(wt), mkU8(7)));
                     assign(t3, binop(Iop_Sub8x16, getWReg(ws), getWReg(wt)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               binop(Iop_AndV128,
                                                     unop(Iop_NotV128,
                                                          mkexpr(t1)),
                                                     mkexpr(t2)),
                                               mkexpr(t3)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    binop(Iop_XorV128,
                                                          mkexpr(t1),
                                                          mkexpr(t2))),
                                               unop(Iop_Abs8x16,
                                                    mkexpr(t3)))),
                                   binop(Iop_AndV128,
                                         binop(Iop_AndV128,
                                               mkexpr(t1),
                                               unop(Iop_NotV128,
                                                    mkexpr(t2))),
                                         binop(Iop_Sub8x16,
                                               getWReg(wt),
                                               getWReg(ws)))));
                     break;
                  }

               case 0x01: { /* ASUB_S.H */
                     DIP("ASUB_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN16x8, getWReg(ws), mkU8(15)));
                     assign(t2, binop(Iop_SarN16x8, getWReg(wt), mkU8(15)));
                     assign(t3, binop(Iop_Sub16x8, getWReg(ws), getWReg(wt)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               binop(Iop_AndV128,
                                                     unop(Iop_NotV128,
                                                          mkexpr(t1)),
                                                     mkexpr(t2)),
                                               mkexpr(t3)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    binop(Iop_XorV128,
                                                          mkexpr(t1),
                                                          mkexpr(t2))),
                                               unop(Iop_Abs16x8,
                                                    mkexpr(t3)))),
                                   binop(Iop_AndV128,
                                         binop(Iop_AndV128,
                                               mkexpr(t1),
                                               unop(Iop_NotV128,
                                                    mkexpr(t2))),
                                         binop(Iop_Sub16x8,
                                               getWReg(wt),
                                               getWReg(ws)))));
                     break;
                  }

               case 0x02: { /* ASUB_S.W */
                     DIP("ASUB_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN32x4, getWReg(ws), mkU8(31)));
                     assign(t2, binop(Iop_SarN32x4, getWReg(wt), mkU8(31)));
                     assign(t3, binop(Iop_Sub32x4, getWReg(ws), getWReg(wt)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               binop(Iop_AndV128,
                                                     unop(Iop_NotV128,
                                                          mkexpr(t1)),
                                                     mkexpr(t2)),
                                               mkexpr(t3)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    binop(Iop_XorV128,
                                                          mkexpr(t1),
                                                          mkexpr(t2))),
                                               unop(Iop_Abs32x4,
                                                    mkexpr(t3)))),
                                   binop(Iop_AndV128,
                                         binop(Iop_AndV128,
                                               mkexpr(t1),
                                               unop(Iop_NotV128,
                                                    mkexpr(t2))),
                                         binop(Iop_Sub32x4,
                                               getWReg(wt),
                                               getWReg(ws)))));
                     break;
                  }

               case 0x03: { /* ASUB_S.D */
                     DIP("ASUB_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, binop(Iop_SarN64x2, getWReg(ws), mkU8(63)));
                     assign(t2, binop(Iop_SarN64x2, getWReg(wt), mkU8(63)));
                     assign(t3, binop(Iop_Sub64x2, getWReg(ws), getWReg(wt)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_OrV128,
                                         binop(Iop_AndV128,
                                               binop(Iop_AndV128,
                                                     unop(Iop_NotV128,
                                                          mkexpr(t1)),
                                                     mkexpr(t2)),
                                               mkexpr(t3)),
                                         binop(Iop_AndV128,
                                               unop(Iop_NotV128,
                                                    binop(Iop_XorV128,
                                                          mkexpr(t1),
                                                          mkexpr(t2))),
                                               unop(Iop_Abs64x2,
                                                    mkexpr(t3)))),
                                   binop(Iop_AndV128,
                                         binop(Iop_AndV128,
                                               mkexpr(t1),
                                               unop(Iop_NotV128,
                                                    mkexpr(t2))),
                                         binop(Iop_Sub64x2,
                                               getWReg(wt),
                                               getWReg(ws)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* ASUB_U.df */
            switch (df) {
               case 0x00: { /* ASUB_U.B */
                     DIP("ASUB_U.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_SarN8x16,
                                  binop(Iop_XorV128,
                                        mkexpr(t1), mkexpr(t2)),
                                  mkU8(7)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         unop(Iop_NotV128, mkexpr(t3)),
                                         unop(Iop_Abs8x16,
                                              binop(Iop_Sub8x16,
                                                    mkexpr(t1),
                                                    mkexpr(t2)))),
                                   binop(Iop_AndV128, mkexpr(t3),
                                         binop(Iop_Sub8x16,
                                               binop(Iop_Max8Ux16,
                                                     mkexpr(t1),
                                                     mkexpr(t2)),
                                               binop(Iop_Min8Ux16,
                                                     mkexpr(t1),
                                                     mkexpr(t2))))));
                     break;
                  }

               case 0x01: { /* ASUB_U.H */
                     DIP("ASUB_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_SarN16x8,
                                  binop(Iop_XorV128,
                                        mkexpr(t1), mkexpr(t2)),
                                  mkU8(15)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         unop(Iop_NotV128,
                                              mkexpr(t3)),
                                         unop(Iop_Abs16x8,
                                              binop(Iop_Sub16x8,
                                                    mkexpr(t1),
                                                    mkexpr(t2)))),
                                   binop(Iop_AndV128,
                                         mkexpr(t3),
                                         binop(Iop_Sub16x8,
                                               binop(Iop_Max16Ux8,
                                                     mkexpr(t1),
                                                     mkexpr(t2)),
                                               binop(Iop_Min16Ux8,
                                                     mkexpr(t1),
                                                     mkexpr(t2))))));
                     break;
                  }

               case 0x02: { /* ASUB_U.W */
                     DIP("ASUB_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_SarN32x4,
                                  binop(Iop_XorV128,
                                        mkexpr(t1), mkexpr(t2)),
                                  mkU8(31)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         unop(Iop_NotV128, mkexpr(t3)),
                                         unop(Iop_Abs32x4,
                                              binop(Iop_Sub32x4,
                                                    mkexpr(t1),
                                                    mkexpr(t2)))),
                                   binop(Iop_AndV128,
                                         mkexpr(t3),
                                         binop(Iop_Sub32x4,
                                               binop(Iop_Max32Ux4,
                                                     mkexpr(t1),
                                                     mkexpr(t2)),
                                               binop(Iop_Min32Ux4,
                                                     mkexpr(t1),
                                                     mkexpr(t2))))));
                     break;
                  }

               case 0x03: { /* ASUB_U.D */
                     DIP("ASUB_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_SarN64x2,
                                  binop(Iop_XorV128,
                                        mkexpr(t1), mkexpr(t2)),
                                  mkU8(63)));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_AndV128,
                                         unop(Iop_NotV128, mkexpr(t3)),
                                         unop(Iop_Abs64x2,
                                              binop(Iop_Sub64x2,
                                                    mkexpr(t1),
                                                    mkexpr(t2)))),
                                   binop(Iop_AndV128,
                                         mkexpr(t3),
                                         binop(Iop_Sub64x2,
                                               binop(Iop_Max64Ux2,
                                                     mkexpr(t1),
                                                     mkexpr(t2)),
                                               binop(Iop_Min64Ux2,
                                                     mkexpr(t1),
                                                     mkexpr(t2))))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_12(UInt cins, UChar wd, UChar ws) { /* 3R (0x12) */
   IRTemp t1, t2, t3, t4, t5, t6;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* MULV.df */
            switch (df) {
               case 0x00: { /* MULV.B */
                     DIP("MULV.B w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd, binop(Iop_Mul8x16, getWReg(ws), getWReg(wt)));
                     break;
                  }

               case 0x01: { /* MULV.H */
                     DIP("MULV.H w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd, binop(Iop_Mul16x8, getWReg(ws), getWReg(wt)));
                     break;
                  }

               case 0x02: { /* MULV.W */
                     DIP("MULV.W w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd, binop(Iop_Mul32x4, getWReg(ws), getWReg(wt)));
                     break;
                  }

               case 0x03: { /* MULV.D */
                     DIP("MULV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_Mul64,
                                         unop(Iop_V128HIto64,
                                              mkexpr(t1)),
                                         unop(Iop_V128HIto64,
                                              mkexpr(t2))),
                                   binop(Iop_Mul64,
                                         unop(Iop_V128to64,
                                              mkexpr(t1)),
                                         unop(Iop_V128to64,
                                              mkexpr(t2)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* MADDV.df */
            switch (df) {
               case 0x00: { /* MADDV.B */
                     DIP("MADDV.B w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Add8x16,
                                   getWReg(wd),
                                   binop(Iop_Mul8x16,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* MADDV.H */
                     DIP("MADDV.H w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Add16x8,
                                   getWReg(wd),
                                   binop(Iop_Mul16x8,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x02: { /* MADDV.W */
                     DIP("MADDV.W w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Add32x4,
                                   getWReg(wd),
                                   binop(Iop_Mul32x4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x03: { /* MADDV.D */
                     DIP("MADDV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     putWReg(wd,
                             binop(Iop_Add64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_Mul64,
                                               unop(Iop_V128HIto64,
                                                    mkexpr(t1)),
                                               unop(Iop_V128HIto64,
                                                    mkexpr(t2))),
                                         binop(Iop_Mul64,
                                               unop(Iop_V128to64,
                                                    mkexpr(t1)),
                                               unop(Iop_V128to64,
                                                    mkexpr(t2))))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* MSUBV.df */
            switch (df) {
               case 0x00: { /* MSUBV.B */
                     DIP("MSUBV.B w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Sub8x16,
                                   getWReg(wd),
                                   binop(Iop_Mul8x16,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* MSUBV.H */
                     DIP("MSUBV.H w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Sub16x8,
                                   getWReg(wd),
                                   binop(Iop_Mul16x8,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x02: { /* MSUBV.W */
                     DIP("MSUBV.W w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_Sub32x4,
                                   getWReg(wd),
                                   binop(Iop_Mul32x4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x03: { /* MSUBV.D */
                     DIP("MSUBV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     putWReg(wd,
                             binop(Iop_Sub64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_Mul64,
                                               unop(Iop_V128HIto64,
                                                    mkexpr(t1)),
                                               unop(Iop_V128HIto64,
                                                    mkexpr(t2))),
                                         binop(Iop_Mul64,
                                               unop(Iop_V128to64,
                                                    mkexpr(t1)),
                                               unop(Iop_V128to64,
                                                    mkexpr(t2))))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* DIV_S.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x00: { /* DIV_S.B */
                     DIP("DIV_S.B w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[16];
                     Int i;

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                                 binop(Iop_And32,
                                       mkU32(0xFF),
                                       binop(Iop_DivS32,
                                             unop(Iop_8Sto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t1),
                                                        mkU8(i))),
                                             unop(Iop_8Sto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t2),
                                                        mkU8(i))))),
                                 mkU8((i & 3) << 3)));
                     }

                     putWReg(wd,
                          binop(Iop_64HLtoV128,
                             binop(Iop_32HLto64,
                                   binop(Iop_Or32,
                                         mkexpr(tmp[15]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[14]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[13]),
                                                     mkexpr(tmp[12])))),
                                   binop(Iop_Or32,
                                         mkexpr(tmp[11]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[10]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[9]),
                                                     mkexpr(tmp[8]))))),
                             binop(Iop_32HLto64,
                                   binop(Iop_Or32,
                                         mkexpr(tmp[7]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[6]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[5]),
                                                     mkexpr(tmp[4])))),
                                   binop(Iop_Or32,
                                         mkexpr(tmp[3]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[2]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[1]),
                                                     mkexpr(tmp[0]))))))
                     );
                     break;
                  }

               case 0x01: { /* DIV_S.H */
                     DIP("DIV_S.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                            binop(Iop_Shl32,
                                  binop(Iop_And32,
                                     mkU32(0xFFFF),
                                     binop(Iop_DivS32,
                                           unop(Iop_16Sto32,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t1),
                                                      mkU8(i))),
                                           unop(Iop_16Sto32,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t2),
                                                      mkU8(i))))),
                                  mkU8((i & 1) << 4)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* DIV_S.W */
                     DIP("DIV_S.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_DivS32,
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t1), mkU8(i)),
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t2), mkU8(i))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128, \
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* DIV_S.D */
                     DIP("DIV_S.D w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_DivS64,
                                         unop(Iop_V128HIto64,
                                              mkexpr(t1)),
                                         unop(Iop_V128HIto64,
                                              mkexpr(t2))),
                                   binop(Iop_DivS64,
                                         unop(Iop_V128to64,
                                              mkexpr(t1)),
                                         unop(Iop_V128to64,
                                              mkexpr(t2)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* DIV_U.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x00: { /* DIV_U.B */
                     DIP("DIV_U.B w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[16];
                     Int i;

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                            binop(Iop_Shl32,
                               binop(Iop_And32,
                                     mkU32(0xFF),
                                     binop(Iop_DivU32,
                                           unop(Iop_8Uto32,
                                                binop(Iop_GetElem8x16,
                                                      mkexpr(t1),
                                                      mkU8(i))),
                                           unop(Iop_8Uto32,
                                                binop(Iop_GetElem8x16,
                                                      mkexpr(t2),
                                                      mkU8(i))))),
                               mkU8((i & 3) << 3)));
                     }

                     putWReg(wd,
                        binop(Iop_64HLtoV128,
                             binop(Iop_32HLto64,
                                   binop(Iop_Or32,
                                         mkexpr(tmp[15]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[14]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[13]),
                                                     mkexpr(tmp[12])))),
                                   binop(Iop_Or32,
                                         mkexpr(tmp[11]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[10]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[9]),
                                                     mkexpr(tmp[8]))))),
                             binop(Iop_32HLto64,
                                   binop(Iop_Or32,
                                         mkexpr(tmp[7]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[6]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[5]),
                                                     mkexpr(tmp[4])))),
                                   binop(Iop_Or32,
                                         mkexpr(tmp[3]),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[2]),
                                               binop(Iop_Or32,
                                                     mkexpr(tmp[1]),
                                                     mkexpr(tmp[0]))))))
                     );
                     break;
                  }

               case 0x01: { /* DIV_U.H */
                     DIP("DIV_U.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                               binop(Iop_And32,
                                     mkU32(0xFFFF),
                                     binop(Iop_DivU32,
                                           unop(Iop_16Uto32,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t1),
                                                      mkU8(i))),
                                           unop(Iop_16Uto32,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t2),
                                                      mkU8(i))))),
                               mkU8((i & 1) << 4)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* DIV_U.W */
                     DIP("DIV_U.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_DivU32,
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t1), mkU8(i)),
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t2), mkU8(i))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* DIV_U.D */
                     DIP("DIV_U.D w%d, w%d, w%d", wd, ws, wt);
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_DivU64,
                                         unop(Iop_V128HIto64,
                                              mkexpr(t1)),
                                         unop(Iop_V128HIto64,
                                              mkexpr(t2))),
                                   binop(Iop_DivU64,
                                         unop(Iop_V128to64,
                                              mkexpr(t1)),
                                         unop(Iop_V128to64,
                                              mkexpr(t2)))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x06: { /* MOD_S.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x00: { /* MOD_S.B */
                     DIP("MOD_S.B w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[16];
                     Int i;

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                              binop(Iop_And32,
                                  mkU32(0xFF),
                                  unop(Iop_64HIto32,
                                       binop(Iop_DivModS32to32,
                                             unop(Iop_8Sto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t1),
                                                        mkU8(i))),
                                             unop(Iop_8Sto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t2),
                                                        mkU8(i)))))),
                              mkU8((i & 3) << 3)));
                     }

                     putWReg(wd,
                        binop(Iop_64HLtoV128,
                           binop(Iop_32HLto64,
                                binop(Iop_Or32,
                                      mkexpr(tmp[15]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[14]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[13]),
                                                  mkexpr(tmp[12])))),
                                binop(Iop_Or32,
                                      mkexpr(tmp[11]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[10]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[9]),
                                                  mkexpr(tmp[8]))))),
                           binop(Iop_32HLto64,
                                binop(Iop_Or32,
                                      mkexpr(tmp[7]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[6]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4])))),
                                binop(Iop_Or32,
                                      mkexpr(tmp[3]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[2]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0])))))));
                     break;
                  }

               case 0x01: { /* MOD_S.H */
                     DIP("MOD_S.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                              binop(Iop_And32,
                                  mkU32(0xFFFF),
                                  unop(Iop_64HIto32,
                                       binop(Iop_DivModS32to32,
                                             unop(Iop_16Sto32,
                                                  binop(Iop_GetElem16x8,
                                                        mkexpr(t1),
                                                        mkU8(i))),
                                             unop(Iop_16Sto32,
                                                  binop(Iop_GetElem16x8,
                                                        mkexpr(t2),
                                                        mkU8(i)))))),
                              mkU8((i & 1) << 4)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* MOD_S.W */
                     DIP("MOD_S.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               unop(Iop_64HIto32,
                                    binop(Iop_DivModS32to32,
                                            binop(Iop_GetElem32x4,
                                                  mkexpr(t1),
                                                  mkU8(i)),
                                          binop(Iop_GetElem32x4,
                                                mkexpr(t2),
                                                mkU8(i)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* MOD_S.D */
                     DIP("MOD_S.D w%d, w%d, w%d", wd, ws, wt);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     assign(t3, unop(Iop_V128HIto64, mkexpr(t1)));
                     assign(t4, unop(Iop_V128HIto64, mkexpr(t2)));
                     assign(t5, unop(Iop_V128to64, mkexpr(t1)));
                     assign(t6, unop(Iop_V128to64, mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_Sub64,
                                         mkexpr(t3),
                                         binop(Iop_Mul64,
                                               mkexpr(t4),
                                               binop(Iop_DivS64,
                                                     mkexpr(t3),
                                                     mkexpr(t4)))),
                                   binop(Iop_Sub64,
                                         mkexpr(t5),
                                         binop(Iop_Mul64,
                                               mkexpr(t6),
                                               binop(Iop_DivS64,
                                                     mkexpr(t5),
                                                     mkexpr(t6))))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* MOD_U.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x00: { /* MOD_U.B */
                     DIP("MOD_U.B w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[16];
                     Int i;

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                              binop(Iop_And32,
                                  mkU32(0xFF),
                                  unop(Iop_64HIto32,
                                       binop(Iop_DivModU32to32,
                                             unop(Iop_8Uto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t1),
                                                        mkU8(i))),
                                             unop(Iop_8Uto32,
                                                  binop(Iop_GetElem8x16,
                                                        mkexpr(t2),
                                                        mkU8(i)))))),
                              mkU8((i & 3) << 3)));
                     }

                     putWReg(wd,
                        binop(Iop_64HLtoV128,
                          binop(Iop_32HLto64,
                                binop(Iop_Or32,
                                      mkexpr(tmp[15]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[14]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[13]),
                                                  mkexpr(tmp[12])))),
                                binop(Iop_Or32,
                                      mkexpr(tmp[11]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[10]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[9]),
                                                  mkexpr(tmp[8]))))),
                          binop(Iop_32HLto64,
                                binop(Iop_Or32,
                                      mkexpr(tmp[7]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[6]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4])))),
                                binop(Iop_Or32,
                                      mkexpr(tmp[3]),
                                      binop(Iop_Or32,
                                            mkexpr(tmp[2]),
                                            binop(Iop_Or32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0])))))));
                     break;
                  }

               case 0x01: { /* MOD_U.H */
                     DIP("MOD_U.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                           binop(Iop_Shl32,
                              binop(Iop_And32,
                                  mkU32(0xFFFF),
                                  unop(Iop_64HIto32,
                                       binop(Iop_DivModU32to32,
                                             unop(Iop_16Uto32,
                                                  binop(Iop_GetElem16x8,
                                                        mkexpr(t1),
                                                        mkU8(i))),
                                             unop(Iop_16Uto32,
                                                  binop(Iop_GetElem16x8,
                                                        mkexpr(t2),
                                                        mkU8(i)))))),
                              mkU8((i & 1) << 4)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_Or32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_Or32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* MOD_U.W */
                     DIP("MOD_U.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               unop(Iop_64HIto32,
                                    binop(Iop_DivModU32to32,
                                          binop(Iop_GetElem32x4,
                                                mkexpr(t1),
                                                mkU8(i)),
                                          binop(Iop_GetElem32x4,
                                                mkexpr(t2),
                                                mkU8(i)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* MOD_U.D */
                     DIP("MOD_U.D w%d, w%d, w%d", wd, ws, wt);
                     t3 = newTemp(Ity_I64);
                     t4 = newTemp(Ity_I64);
                     t5 = newTemp(Ity_I64);
                     t6 = newTemp(Ity_I64);
                     assign(t3, unop(Iop_V128HIto64, mkexpr(t1)));
                     assign(t4, unop(Iop_V128HIto64, mkexpr(t2)));
                     assign(t5, unop(Iop_V128to64, mkexpr(t1)));
                     assign(t6, unop(Iop_V128to64, mkexpr(t2)));
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_Sub64,
                                         mkexpr(t3),
                                         binop(Iop_Mul64,
                                               mkexpr(t4),
                                               binop(Iop_DivU64,
                                                     mkexpr(t3),
                                                     mkexpr(t4)))),
                                   binop(Iop_Sub64,
                                         mkexpr(t5),
                                         binop(Iop_Mul64,
                                               mkexpr(t6),
                                               binop(Iop_DivU64,
                                                     mkexpr(t5),
                                                     mkexpr(t6))))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_13(UInt cins, UChar wd, UChar ws) { /* 3R (0x13) */
   IRTemp t1, t2;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* DOTP_S.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DOTP_S.H */
                     DIP("DOTP_S.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* DOTP_S.W */
                     DIP("DOTP_S.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* DOTP_S.D */
                     DIP("DOTP_S.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkexpr(tmp[1]), mkexpr(tmp[0])));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* DOTP_U.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DOTP_U.H */
                     DIP("DOTP_U.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* DOTP_U.W */
                     DIP("DOTP_U.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* DOTP_U.D */
                     DIP("DOTP_U.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkexpr(tmp[1]), mkexpr(tmp[0])));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* DPADD_S.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DPADD_S.H */
                     DIP("DPADD_S.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Add16x8,
                                getWReg(wd),
                                binop(Iop_64HLtoV128,
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x02: { /* DPADD_S.W */
                     DIP("DPADD_S.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Add32x4,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x03: { /* DPADD_S.D */
                     DIP("DPADD_S.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Add64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* DPADD_U.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DPADD_U.H */
                     DIP("DPADD_U.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                          binop(Iop_Add16x8,
                                getWReg(wd),
                                binop(Iop_64HLtoV128,
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x02: { /* DPADD_U.W */
                     DIP("DPADD_U.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Add32x4,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x03: { /* DPADD_U.D */
                     DIP("DPADD_U.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Add64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* DPSUB_S.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DPSUB_S.H */
                     DIP("DPSUB_S.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                          binop(Iop_Sub16x8,
                                getWReg(wd),
                                binop(Iop_64HLtoV128,
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x02: { /* DPSUB_S.W */
                     DIP("DPSUB_S.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Sub32x4,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x03: { /* DPSUB_S.D */
                     DIP("DPSUB_S.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullS32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Sub64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* DPSUB_U.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            assign(t1, getWReg(ws));
            assign(t2, getWReg(wt));

            switch (df) {
               case 0x01: { /* DPSUB_U.H */
                     DIP("DPSUB_U.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_Add16,
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU8,
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                          binop(Iop_Sub16x8,
                                getWReg(wd),
                                binop(Iop_64HLtoV128,
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_32HLto64,
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_16HLto32,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x02: { /* DPSUB_U.W */
                     DIP("DPSUB_U.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_Add32,
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU16,
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem16x8,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Sub32x4,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_32HLto64,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x03: { /* DPSUB_U.D */
                     DIP("DPSUB_U.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_Add64,
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i))),
                                     binop(Iop_MullU32,
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t1),
                                                 mkU8(2 * i + 1)),
                                           binop(Iop_GetElem32x4,
                                                 mkexpr(t2),
                                                 mkU8(2 * i + 1)))));
                     }

                     putWReg(wd,
                             binop(Iop_Sub64x2,
                                   getWReg(wd),
                                   binop(Iop_64HLtoV128,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_14(UInt cins, UChar wd, UChar ws) { /* 3R (0x14) */
   IRTemp t1, t2, t3, t4;
   IRType ty;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;
   ty = mode64 ? Ity_I64 : Ity_I32;

   switch (operation) {
      case 0x00: { /* SLD.df */
            switch (df) {
               case 0x00: {
                     DIP("SLD.B w%d, w%d[%d]", wd, ws, wt);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shl32,
                                  binop(Iop_And32,
                                        mkNarrowTo32(ty,
                                                     getIReg(wt)),
                                        mkU32(15)),
                                  mkU8(3)));
                     assign(t2,
                            binop(Iop_ShrV128,
                                  getWReg(ws),
                                  unop(Iop_32to8, mkexpr(t1))));
                     assign(t3,
                            binop(Iop_ShlV128,
                                  getWReg(wd),
                                  unop(Iop_32to8,
                                       binop(Iop_Sub32,
                                             mkU32(128),
                                             mkexpr(t1)))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t2), mkexpr(t3)));
                     break;
                  }

               case 0x01: {/* SLD.H */
                     DIP("SLD.H w%d, w%d[%d]", wd, ws, wt);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shl32,
                                  binop(Iop_And32,
                                        mkNarrowTo32(ty,
                                                     getIReg(wt)),
                                        mkU32(7)),
                                  mkU8(3)));
                     assign(t2,
                            binop(Iop_32HLto64, mkU32(0), mkexpr(t1)));
                     assign(t3,
                            binop(Iop_Shr64x2,
                                  getWReg(ws),
                                  binop(Iop_64HLtoV128,
                                        mkexpr(t2), mkexpr(t2))));
                     assign(t4,
                            binop(Iop_Shl64x2,
                                  getWReg(wd),
                                  binop(Iop_Sub64x2,
                                        binop(Iop_64HLtoV128,
                                              mkU64(0x40ul),
                                              mkU64(0x40ul)),
                                        binop(Iop_64HLtoV128,
                                              mkexpr(t2),
                                              mkexpr(t2)))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t3),
                                   IRExpr_ITE(
                                      binop(Iop_CmpNE32,
                                            mkexpr(t1), mkU32(0)),
                                      mkexpr(t4),
                                      binop(Iop_64HLtoV128,
                                            mkU64(0), mkU64(0)))));
                     break;
                  }

               case 0x02: {/* SLD.W */
                     DIP("SLD.W w%d, w%d[%d]", wd, ws, wt);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shl32,
                                  binop(Iop_And32,
                                        mkNarrowTo32(ty,
                                                     getIReg(wt)),
                                        mkU32(3)),
                                  mkU8(3)));
                     assign(t2,
                            binop(Iop_32HLto64,
                                  mkexpr(t1), mkexpr(t1)));
                     assign(t3,
                            binop(Iop_Shr32x4,
                                  getWReg(ws),
                                  binop(Iop_64HLtoV128,
                                        mkexpr(t2), mkexpr(t2))));
                     assign(t4,
                            binop(Iop_Shl32x4,
                                  getWReg(wd),
                                  binop(Iop_Sub32x4,
                                        binop(Iop_64HLtoV128,
                                              mkU64(0x2000000020ul),
                                              mkU64(0x2000000020ul)),
                                        binop(Iop_64HLtoV128,
                                              mkexpr(t2),
                                              mkexpr(t2)))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t3),
                                   IRExpr_ITE(
                                      binop(Iop_CmpNE32,
                                            mkexpr(t1), mkU32(0)),
                                      mkexpr(t4),
                                      binop(Iop_64HLtoV128,
                                            mkU64(0), mkU64(0)))));
                     break;
                  }

               case 0x03: { /* SLD.D */
                     DIP("SLD.D w%d, w%d[%d]", wd, ws, wt);
                     t1 = newTemp(Ity_I32);
                     t2 = newTemp(Ity_I64);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shl32,
                                  binop(Iop_And32,
                                        mkNarrowTo32(ty,
                                                     getIReg(wt)),
                                        mkU32(1)),
                                  mkU8(3)));
                     assign(t2,
                            binop(Iop_32HLto64,
                                  binop(Iop_Or32,
                                        mkexpr(t1),
                                        binop(Iop_Shl32,
                                              mkexpr(t1), mkU8(16))),
                                  binop(Iop_Or32,
                                        mkexpr(t1),
                                        binop(Iop_Shl32,
                                              mkexpr(t1), mkU8(16)))));
                     assign(t3,
                            binop(Iop_Shr16x8,
                                  getWReg(ws),
                                  binop(Iop_64HLtoV128,
                                        mkexpr(t2), mkexpr(t2))));
                     assign(t4,
                            binop(Iop_Shl16x8,
                               getWReg(wd),
                               binop(Iop_Sub16x8,
                                     binop(Iop_64HLtoV128,
                                           mkU64(0x10001000100010ul),
                                           mkU64(0x10001000100010ul)),
                                     binop(Iop_64HLtoV128,
                                           mkexpr(t2),
                                           mkexpr(t2)))));
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   mkexpr(t3),
                                   IRExpr_ITE(
                                      binop(Iop_CmpNE32,
                                            mkexpr(t1), mkU32(0)),
                                      mkexpr(t4),
                                      binop(Iop_64HLtoV128,
                                            mkU64(0), mkU64(0)))));
                     break;
                  }
            }

            break;
         }

      case 0x01: { /* SPLAT.df */
            switch (df) {
                  Int i;

               case 0x00: { /* SPLAT.B */
                     DIP("SPLAT.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_I32);
                     assign(t1, getWReg(ws));
                     assign(t2,
                            mkNarrowTo32(ty, getIReg(wt)));
                     IRTemp tmp[16];

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I8);
                        assign(tmp[i],
                               binop(Iop_GetElem8x16,
                                     mkexpr(t1),
                                     unop(Iop_32to8, mkexpr(t2))));
                     }

                     putWReg(wd,
                          binop(Iop_64HLtoV128,
                                binop(Iop_32HLto64,
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[15]),
                                                  mkexpr(tmp[14])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[13]),
                                                  mkexpr(tmp[12]))),
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[11]),
                                                  mkexpr(tmp[10])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[9]),
                                                  mkexpr(tmp[8])))),
                                binop(Iop_32HLto64,
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x01: { /* SPLAT.H */
                     DIP("SPLAT.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_I32);
                     assign(t1, getWReg(ws));
                     assign(t2,
                            mkNarrowTo32(ty, getIReg(wt)));
                     IRTemp tmp[8];

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               binop(Iop_GetElem16x8,
                                     mkexpr(t1),
                                     unop(Iop_32to8, mkexpr(t2))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* SPLAT.W */
                     DIP("SPLAT.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_I32);
                     assign(t1, getWReg(ws));
                     assign(t2,
                            mkNarrowTo32(ty, getIReg(wt)));
                     IRTemp tmp[4];

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               binop(Iop_GetElem32x4,
                                     mkexpr(t1),
                                     unop(Iop_32to8, mkexpr(t2))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* SPLAT.D */
                     DIP("SPLAT.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_I32);
                     assign(t1, getWReg(ws));
                     assign(t2,
                            mkNarrowTo32(ty, getIReg(wt)));
                     IRTemp tmp[2];

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               binop(Iop_GetElem64x2,
                                     mkexpr(t1),
                                     unop(Iop_32to8, mkexpr(t2))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkexpr(tmp[1]), mkexpr(tmp[0])));
                     break;
                  }
            }

            break;
         }

      case 0x02: { /* PCKEV.df */
            switch (df) {
               case 0x00: { /* PCKEV.B */
                     DIP("PCKEV.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackEvenLanes8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* PCKEV.H */
                     DIP("PCKEV.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackEvenLanes16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* PCKEV.W */
                     DIP("PCKEV.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackEvenLanes32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* PCKEV.D */
                     DIP("PCKEV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* PCKOD.df */
            switch (df) {
               case 0x00: { /* PCKOD.B */
                     DIP("PCKOD.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackOddLanes8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* PCKOD.H */
                     DIP("PCKOD.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackOddLanes16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* PCKOD.W */
                     DIP("PCKOD.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_PackOddLanes32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* PCKOD.D */
                     DIP("PCKOD.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* ILVL.df */
            switch (df) {
               case 0x00: { /* ILVL.B */
                     DIP("ILVL.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ILVL.H */
                     DIP("ILVL.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ILVL.W */
                     DIP("ILVL.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ILVL.D */
                     DIP("ILVL.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* ILVR.df */
            switch (df) {
               case 0x00: { /* ILVL.B */
                     DIP("ILVL.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ILVL.H */
                     DIP("ILVL.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ILVL.W */
                     DIP("ILVL.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ILVL.D */
                     DIP("ILVL.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }
            }

            break;
         }

      case 0x06: { /* ILVEV.df */
            switch (df) {
               case 0x00: { /* ILVEV.B */
                     DIP("ILVEV.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveEvenLanes8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ILVEV.H */
                     DIP("ILVEV.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveEvenLanes16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ILVEV.W */
                     DIP("ILVEV.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveEvenLanes32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ILVEV.D */
                     DIP("ILVEV.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveLO64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* ILVOD.df */
            switch (df) {
               case 0x00: { /* ILVOD.B */
                     DIP("ILVOD.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveOddLanes8x16,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x01: { /* ILVOD.H */
                     DIP("ILVOD.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveOddLanes16x8,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* ILVOD.W */
                     DIP("ILVOD.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveOddLanes32x4,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* ILVOD.D */
                     DIP("ILVOD.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_InterleaveHI64x2,
                                  mkexpr(t1), mkexpr(t2)));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_15(UInt cins, UChar wd, UChar ws) { /* 3R (0x15) */
   IRTemp t1, t2, t3, t4;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03800000) >> 23;
   df = (cins & 0x00600000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* VSHF.df */
            t1 = newTemp(Ity_V128);
            t2 = newTemp(Ity_V128);
            t3 = newTemp(Ity_V128);
            assign(t1, getWReg(wd));
            assign(t2, getWReg(ws));
            assign(t3, getWReg(wt));

            switch (df) {
               case 0x00: { /* VSHF.B */
                     DIP("VSHF.B w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[16];
                     Int i;

                     for (i = 0; i < 16; i++) {
                        tmp[i] = newTemp(Ity_I8);
                        assign(tmp[i],
                               IRExpr_ITE(
                                  binop(Iop_CmpEQ8,
                                        binop(Iop_And8,
                                              binop(Iop_GetElem8x16,
                                                    mkexpr(t1),
                                                    mkU8(i)),
                                              mkU8(0xC0)),
                                        mkU8(0x0)),
                                  IRExpr_ITE(
                                     binop(Iop_CmpEQ8,
                                           binop(Iop_And8,
                                                 binop(Iop_GetElem8x16,
                                                       mkexpr(t1),
                                                       mkU8(i)),
                                                 mkU8(0x10)),
                                           mkU8(0x0)),
                                     binop(Iop_GetElem8x16,
                                           mkexpr(t3),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(i))),
                                     binop(Iop_GetElem8x16,
                                           mkexpr(t2),
                                           binop(Iop_GetElem8x16,
                                                 mkexpr(t1),
                                                 mkU8(i)))),
                                  mkU8(0x0)));
                     }

                     putWReg(wd,
                          binop(Iop_64HLtoV128,
                                binop(Iop_32HLto64,
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[15]),
                                                  mkexpr(tmp[14])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[13]),
                                                  mkexpr(tmp[12]))),
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[11]),
                                                  mkexpr(tmp[10])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[9]),
                                                  mkexpr(tmp[8])))),
                                binop(Iop_32HLto64,
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[7]),
                                                  mkexpr(tmp[6])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[5]),
                                                  mkexpr(tmp[4]))),
                                      binop(Iop_16HLto32,
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[3]),
                                                  mkexpr(tmp[2])),
                                            binop(Iop_8HLto16,
                                                  mkexpr(tmp[1]),
                                                  mkexpr(tmp[0]))))));
                     break;
                  }

               case 0x01: { /* VSHF.H */
                     DIP("VSHF.H w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[8];
                     Int i;

                     for (i = 0; i < 8; i++) {
                        tmp[i] = newTemp(Ity_I16);
                        assign(tmp[i],
                               IRExpr_ITE(
                                  binop(Iop_CmpEQ16,
                                        binop(Iop_And16,
                                              binop(Iop_GetElem16x8,
                                                    mkexpr(t1),
                                                    mkU8(i)),
                                              mkU16(0xC0)),
                                        mkU16(0x0)),
                                  IRExpr_ITE(
                                     binop(Iop_CmpEQ16,
                                           binop(Iop_And16,
                                                 binop(Iop_GetElem16x8,
                                                       mkexpr(t1),
                                                       mkU8(i)),
                                                 mkU16(0x08)),
                                           mkU16(0x0)),
                                     binop(Iop_GetElem16x8,
                                           mkexpr(t3),
                                           unop(Iop_16to8,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t1),
                                                      mkU8(i)))),
                                     binop(Iop_GetElem16x8,
                                           mkexpr(t2),
                                           unop(Iop_16to8,
                                                binop(Iop_GetElem16x8,
                                                      mkexpr(t1),
                                                      mkU8(i))))),
                                  mkU16(0x0)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[7]),
                                               mkexpr(tmp[6])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[5]),
                                               mkexpr(tmp[4]))),
                                   binop(Iop_32HLto64,
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[3]),
                                               mkexpr(tmp[2])),
                                         binop(Iop_16HLto32,
                                               mkexpr(tmp[1]),
                                               mkexpr(tmp[0])))));
                     break;
                  }

               case 0x02: { /* VSHF.W */
                     DIP("VSHF.W w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_I32);
                        assign(tmp[i],
                               IRExpr_ITE(
                                  binop(Iop_CmpEQ32,
                                        binop(Iop_And32,
                                              binop(Iop_GetElem32x4,
                                                    mkexpr(t1),
                                                    mkU8(i)),
                                              mkU32(0xC0)),
                                        mkU32(0x0)),
                                  IRExpr_ITE(
                                     binop(Iop_CmpEQ32,
                                           binop(Iop_And32,
                                                 binop(Iop_GetElem32x4,
                                                       mkexpr(t1),
                                                       mkU8(i)),
                                                 mkU32(0x04)),
                                           mkU32(0x0)),
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t3),
                                           unop(Iop_32to8,
                                                binop(Iop_GetElem32x4,
                                                      mkexpr(t1),
                                                      mkU8(i)))),
                                     binop(Iop_GetElem32x4,
                                           mkexpr(t2),
                                           unop(Iop_32to8,
                                                binop(Iop_GetElem32x4,
                                                      mkexpr(t1),
                                                      mkU8(i))))),
                                  mkU32(0x0)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[3]),
                                         mkexpr(tmp[2])),
                                   binop(Iop_32HLto64,
                                         mkexpr(tmp[1]),
                                         mkexpr(tmp[0]))));
                     break;
                  }

               case 0x03: { /* VSHF.D */
                     DIP("VSHF.D w%d, w%d, w%d", wd, ws, wt);
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_I64);
                        assign(tmp[i],
                               IRExpr_ITE(
                                  binop(Iop_CmpEQ64,
                                        binop(Iop_And64,
                                              binop(Iop_GetElem64x2,
                                                    mkexpr(t1),
                                                    mkU8(i)),
                                              mkU64(0xC0)),
                                        mkU64(0x0)),
                                  IRExpr_ITE(
                                     binop(Iop_CmpEQ64,
                                           binop(Iop_And64,
                                                 binop(Iop_GetElem64x2,
                                                       mkexpr(t1),
                                                       mkU8(i)),
                                                 mkU64(0x02)),
                                           mkU64(0x0)),
                                     binop(Iop_GetElem64x2,
                                           mkexpr(t3),
                                           unop(Iop_64to8,
                                                binop(Iop_GetElem64x2,
                                                      mkexpr(t1),
                                                      mkU8(i)))),
                                     binop(Iop_GetElem64x2,
                                           mkexpr(t2),
                                           unop(Iop_64to8,
                                                binop(Iop_GetElem64x2,
                                                      mkexpr(t1),
                                                      mkU8(i))))),
                                  mkU64(0x0)));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkexpr(tmp[1]), mkexpr(tmp[0])));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* SRAR.df */
            switch (df) {
               case 0x00: { /* SRAR.B */
                     DIP("SRAR.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Sar8x16,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x808080808080808ull),
                                        mkU64(0x808080808080808ull)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ8x16,
                                       binop(Iop_ShlN8x16,
                                             getWReg(wt),
                                             mkU8(5)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN8x16,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl8x16,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(7)));
                     putWReg(wd,
                             binop(Iop_Add8x16,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* SRAR.H */
                     DIP("SRAR.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Sar16x8,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x10001000100010ul),
                                        mkU64(0x10001000100010ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ16x8,
                                       binop(Iop_ShlN16x8,
                                             getWReg(wt),
                                             mkU8(12)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN16x8,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl16x8,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(15)));
                     putWReg(wd,
                             binop(Iop_Add16x8,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* SRAR.W */
                     DIP("SRAR.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128); // shifted
                     t2 = newTemp(Ity_V128); // 32 - wt
                     t3 = newTemp(Ity_V128); // rv
                     t4 = newTemp(Ity_V128); // wt % 32 == 0
                     assign(t1,
                            binop(Iop_Sar32x4,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x2000000020ul),
                                        mkU64(0x2000000020ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ32x4,
                                       binop(Iop_ShlN32x4,
                                             getWReg(wt),
                                             mkU8(27)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN32x4,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl32x4,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(31)));
                     putWReg(wd,
                             binop(Iop_Add32x4,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* SRAR.D */
                     DIP("SRAR.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Sar64x2,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(64ul), mkU64(64ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ64x2,
                                       binop(Iop_ShlN64x2,
                                             getWReg(wt),
                                             mkU8(58)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN64x2,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl64x2,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(63)));
                     putWReg(wd,
                             binop(Iop_Add64x2,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* SRLR.df */
            switch (df) {
               case 0x00: { /* SRLR.B */
                     DIP("SRLR.B w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shr8x16,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub8x16,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x808080808080808ull),
                                        mkU64(0x808080808080808ull)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ8x16,
                                       binop(Iop_ShlN8x16,
                                             getWReg(wt),
                                             mkU8(5)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN8x16,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl8x16,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(7)));
                     putWReg(wd,
                             binop(Iop_Add8x16,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x01: { /* SRLR.H */
                     DIP("SRLR.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shr16x8,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub16x8,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x10001000100010ul),
                                        mkU64(0x10001000100010ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ16x8,
                                       binop(Iop_ShlN16x8,
                                             getWReg(wt),
                                             mkU8(12)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN16x8,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl16x8,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(15)));
                     putWReg(wd,
                             binop(Iop_Add16x8,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x02: { /* SRLR.W */
                     DIP("SRLR.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shr32x4,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub32x4,
                                  binop(Iop_64HLtoV128,
                                        mkU64(0x2000000020ul),
                                        mkU64(0x2000000020ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ32x4,
                                       binop(Iop_ShlN32x4,
                                             getWReg(wt),
                                             mkU8(27)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN32x4,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl32x4,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(31)));
                     putWReg(wd,
                             binop(Iop_Add32x4,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               case 0x03: { /* SRLR.D */
                     DIP("SRLR.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     t4 = newTemp(Ity_V128);
                     assign(t1,
                            binop(Iop_Shr64x2,
                                  getWReg(ws),
                                  getWReg(wt)));
                     assign(t2,
                            binop(Iop_Sub64x2,
                                  binop(Iop_64HLtoV128,
                                        mkU64(64ul), mkU64(64ul)),
                                  getWReg(wt)));
                     assign(t4,
                            unop(Iop_NotV128,
                                 binop(Iop_CmpEQ64x2,
                                       binop(Iop_ShlN64x2,
                                             getWReg(wt),
                                             mkU8(58)),
                                       binop(Iop_64HLtoV128,
                                             mkU64(0), mkU64(0)))));
                     assign(t3,
                            binop(Iop_ShrN64x2,
                                  binop(Iop_AndV128,
                                        binop(Iop_Shl64x2,
                                              getWReg(ws),
                                              mkexpr(t2)),
                                        mkexpr(t4)),
                                  mkU8(63)));
                     putWReg(wd,
                             binop(Iop_Add64x2,
                                   mkexpr(t1), mkexpr(t3)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* HADD_S.df */
            switch (df) {
               case 0x01: { /* HADD_S.H */
                     DIP("HADD_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add16x8,
                                  binop(Iop_SarN16x8,
                                        mkexpr(t1), mkU8(8)),
                                  binop(Iop_SarN16x8,
                                        binop(Iop_ShlN16x8,
                                              mkexpr(t2), mkU8(8)),
                                        mkU8(8))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* HADD_S.W */
                     DIP("HADD_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add32x4,
                                  binop(Iop_SarN32x4,
                                        mkexpr(t1), mkU8(16)),
                                  binop(Iop_SarN32x4,
                                        binop(Iop_ShlN32x4,
                                              mkexpr(t2), mkU8(16)),
                                        mkU8(16))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* HADD_S.D */
                     DIP("HADD_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add64x2,
                                  binop(Iop_SarN64x2,
                                        mkexpr(t1), mkU8(32)),
                                  binop(Iop_SarN64x2,
                                        binop(Iop_ShlN64x2,
                                              mkexpr(t2), mkU8(32)),
                                        mkU8(32))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* HADD_U.df */
            switch (df) {
               case 0x01: { /* HADD_U.H */
                     DIP("HADD_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add16x8,
                                  binop(Iop_ShrN16x8,
                                        mkexpr(t1), mkU8(8)),
                                  binop(Iop_ShrN16x8,
                                        binop(Iop_ShlN16x8,
                                              mkexpr(t2), mkU8(8)),
                                        mkU8(8))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* HADD_U.W */
                     DIP("HADD_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add32x4,
                                  binop(Iop_ShrN32x4,
                                        mkexpr(t1), mkU8(16)),
                                  binop(Iop_ShrN32x4,
                                        binop(Iop_ShlN32x4,
                                              mkexpr(t2), mkU8(16)),
                                        mkU8(16))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* HADD_U.D */
                     DIP("HADD_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Add64x2,
                                  binop(Iop_ShrN64x2,
                                        mkexpr(t1), mkU8(32)),
                                  binop(Iop_ShrN64x2,
                                        binop(Iop_ShlN64x2,
                                              mkexpr(t2), mkU8(32)),
                                        mkU8(32))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x06: { /* HSUB_S.df */
            switch (df) {
               case 0x01: { /* HSUB_S.H */
                     DIP("HSUB_S.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub16x8,
                                  binop(Iop_SarN16x8,
                                        mkexpr(t1), mkU8(8)),
                                  binop(Iop_SarN16x8,
                                        binop(Iop_ShlN16x8,
                                              mkexpr(t2), mkU8(8)),
                                        mkU8(8))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* HSUB_S.W */
                     DIP("HSUB_S.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub32x4,
                                  binop(Iop_SarN32x4,
                                        mkexpr(t1), mkU8(16)),
                                  binop(Iop_SarN32x4,
                                        binop(Iop_ShlN32x4,
                                              mkexpr(t2), mkU8(16)),
                                        mkU8(16))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* HSUB_S.D */
                     DIP("HSUB_S.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub64x2,
                                  binop(Iop_SarN64x2,
                                        mkexpr(t1), mkU8(32)),
                                  binop(Iop_SarN64x2,
                                        binop(Iop_ShlN64x2,
                                              mkexpr(t2), mkU8(32)),
                                        mkU8(32))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* HSUB_U.df */
            switch (df) {
               case 0x01: { /* HSUB_U.H */
                     DIP("HSUB_U.H w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub16x8,
                                  binop(Iop_ShrN16x8,
                                        mkexpr(t1), mkU8(8)),
                                  binop(Iop_ShrN16x8,
                                        binop(Iop_ShlN16x8,
                                              mkexpr(t2), mkU8(8)),
                                        mkU8(8))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x02: { /* HSUB_U.W */
                     DIP("HSUB_U.W w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub32x4,
                                  binop(Iop_ShrN32x4,
                                        mkexpr(t1), mkU8(16)),
                                  binop(Iop_ShrN32x4,
                                        binop(Iop_ShlN32x4,
                                              mkexpr(t2), mkU8(16)),
                                        mkU8(16))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               case 0x03: { /* HSUB_U.D */
                     DIP("HSUB_U.D w%d, w%d, w%d", wd, ws, wt);
                     t1 = newTemp(Ity_V128);
                     t2 = newTemp(Ity_V128);
                     t3 = newTemp(Ity_V128);
                     assign(t1, getWReg(ws));
                     assign(t2, getWReg(wt));
                     assign(t3,
                            binop(Iop_Sub64x2,
                                  binop(Iop_ShrN64x2,
                                        mkexpr(t1), mkU8(32)),
                                  binop(Iop_ShrN64x2,
                                        binop(Iop_ShlN64x2,
                                              mkexpr(t2), mkU8(32)),
                                        mkU8(32))));
                     putWReg(wd, mkexpr(t3));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_1A(UInt cins, UChar wd, UChar ws) { /* 3R (0x1A) */
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03C00000) >> 22;
   df = (cins & 0x00200000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* FCAF.df */
            switch (df) {
               case 0x00: { /* FCAF.W */
                     DIP("FCAF.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCAFW, 2);
                     putWReg(wd, binop(Iop_64HLtoV128, mkU64(0ul), mkU64(0ul)));
                     break;
                  }

               case 0x01: { /* FCAF.D */
                     DIP("FCAF.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCAFD, 2);
                     putWReg(wd, binop(Iop_64HLtoV128, mkU64(0ul), mkU64(0ul)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* FCUN.df */
            switch (df) {
               case 0x00: { /* FCUN.W */
                     DIP("FCUN.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCUNW, 2);
                     putWReg(wd, binop(Iop_CmpUN32Fx4,
                                       getWReg(ws),
                                       getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FCUN.D */
                     DIP("FCUN.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCUND, 2);
                     putWReg(wd, binop(Iop_CmpUN64Fx2,
                                       getWReg(ws),
                                       getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* FCEQ.df */
            switch (df) {
               case 0x00: { /* FCEQ.W */
                     DIP("FCEQ.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCEQW, 2);
                     putWReg(wd, binop(Iop_CmpEQ32Fx4,
                                       getWReg(ws),
                                       getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FCEQ.D */
                     DIP("FCEQ.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCEQD, 2);
                     putWReg(wd, binop(Iop_CmpEQ64Fx2,
                                       getWReg(ws),
                                       getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* FCUEQ.df */
            switch (df) {
               case 0x00: { /* FCUEQ.W */
                     DIP("FCUEQ.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCUEQW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpEQ32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FCUEQ.D */
                     DIP("FCUEQ.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCUEQD, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpEQ64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* FCLT.df */
            switch (df) {
               case 0x00: { /* FCLT.W */
                     DIP("FCLT.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCLTW, 2);
                     putWReg(wd,
                             binop(Iop_CmpLT32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FCLT.D */
                     DIP("FCLT.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCLTD, 2);
                     putWReg(wd,
                             binop(Iop_CmpLT64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* FCULT.df */
            switch (df) {
               case 0x00: { /* FCULT.W */
                     DIP("FCULT.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCULTW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLT32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FCULT.D */
                     DIP("FCULT.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCULTD, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLT64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x06: { /* FCLE.df */
            switch (df) {
               case 0x00: { /* FCLE.W */
                     DIP("FCLE.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCLEW, 2);
                     putWReg(wd,
                             binop(Iop_CmpLE32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FCLE.D */
                     DIP("FCLE.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCLED, 2);
                     putWReg(wd,
                             binop(Iop_CmpLE64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x07: { /* FCULE.df */
            switch (df) {
               case 0x00: { /* FCULE.W */
                     DIP("FCULE.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCULEW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLE32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FCULE.D */
                     DIP("FCULE.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FCULED, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLE64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x08: { /* FSAF.df */
            switch (df) {
               case 0x00: { /* FSAF.W */
                     DIP("FSAF.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSAFW, 2);
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkU64(0ul), mkU64(0ul)));
                     break;
                  }

               case 0x01: { /* FSAF.D */
                     DIP("FSAF.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSAFD, 2);
                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   mkU64(0ul), mkU64(0ul)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x09: { /* FSUN.df */
            switch (df) {
               case 0x00: { /* FSUN.W */
                     DIP("FSUN.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUNW, 2);
                     putWReg(wd,
                             binop(Iop_CmpUN32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FSUN.D */
                     DIP("FSUN.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUND, 2);
                     putWReg(wd,
                             binop(Iop_CmpUN64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0A: { /* FSEQ.df */
            switch (df) {
               case 0x00: { /* FSEQ.W */
                     DIP("FSEQ.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSEQW, 2);
                     putWReg(wd,
                             binop(Iop_CmpEQ32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FSEQ.D */
                     DIP("FSEQ.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSEQD, 2);
                     putWReg(wd,
                             binop(Iop_CmpEQ64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0B: { /* FSUEQ.df */
            switch (df) {
               case 0x00: { /* FSUEQ.W */
                     DIP("FSUEQ.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUEQW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpEQ32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FSUEQ.D */
                     DIP("FSUEQ.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUEQD, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpEQ64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0C: { /* FSLT.df */
            switch (df) {
               case 0x00: { /* FSLT.W */
                     DIP("FSLT.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSLTW, 2);
                     putWReg(wd,
                             binop(Iop_CmpLT32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FSLT.D */
                     DIP("FSLT.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSLTD, 2);
                     putWReg(wd,
                             binop(Iop_CmpLT64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0D: { /* FSULT.df */
            switch (df) {
               case 0x00: { /* FSULT.W */
                     DIP("FSULT.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSULTW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLT32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FSULT.D */
                     DIP("FSULT.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSULTD, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLT64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0E: { /* FSLE.df */
            switch (df) {
               case 0x00: { /* FSLE.W */
                     DIP("FSLE.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSLEW, 2);
                     putWReg(wd,
                             binop(Iop_CmpLE32Fx4,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FSLE.D */
                     DIP("FSLE.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSLED, 2);
                     putWReg(wd,
                             binop(Iop_CmpLE64Fx2,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x0F: { /* FSULE.df */
            switch (df) {
               case 0x00: { /* FSULE.W */
                     DIP("FSULE.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSULEW, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLE32Fx4,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN32Fx4,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               case 0x01: { /* FSULE.D */
                     DIP("FSULE.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSULED, 2);
                     putWReg(wd,
                             binop(Iop_OrV128,
                                   binop(Iop_CmpLE64Fx2,
                                         getWReg(ws),
                                         getWReg(wt)),
                                   binop(Iop_CmpUN64Fx2,
                                         getWReg(ws),
                                         getWReg(wt))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      default:
         return -1;
   }

   return 0;
}

static Int msa_3R_1B(UInt cins, UChar wd, UChar ws) { /* 3R (0x1B) */
   IRTemp t1, t2, t3, t4;
   UShort operation;
   UChar df, wt;

   operation = (cins & 0x03C00000) >> 22;
   df = (cins & 0x00200000) >> 21;
   wt = (cins & 0x001F0000) >> 16;

   switch (operation) {
      case 0x00: { /* FADD.df */
            switch (df) {
               case 0x00: { /* FADD.W */
                     DIP("FADD.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FADDW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Add32Fx4, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FADD.D */
                     DIP("FADD.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FADDD, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Add64Fx2, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x01: { /* FSUB.df */
            switch (df) {
               case 0x00: { /* FSUB.W */
                     DIP("FSUB.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUBW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Sub32Fx4, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FSUB.D */
                     DIP("FSUB.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FSUBD, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Sub64Fx2, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x02: { /* FMUL.df */
            switch (df) {
               case 0x00: { /* FMUL.W */
                     DIP("FMUL.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMULW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Mul32Fx4, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FMUL.D */
                     DIP("FMUL.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMULW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Mul64Fx2, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x03: { /* FDIV.df */
            switch (df) {
               case 0x00: { /* FDIV.W */
                     DIP("FDIV.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FDIVW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Div32Fx4, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               case 0x01: { /* FDIV.D */
                     DIP("FDIV.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FDIVD, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     putWReg(wd,
                             triop(Iop_Div64Fx2, rm,
                                   getWReg(ws),
                                   getWReg(wt)));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x04: { /* FMADD.df */
            switch (df) {
               case 0x00: { /* FMADD.W */
                     DIP("FMADD.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMADDW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_F32);
                        assign(tmp[i],
                               qop(Iop_MAddF32, rm,
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(ws),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(wt),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(wd),
                                              mkU8(i)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[3])),
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[2]))),
                                   binop(Iop_32HLto64,
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[1])),
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[0])))));
                     break;
                  }

               case 0x01: { /* FMADD.D */
                     DIP("FMADD.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMADDW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_F64);
                        assign(tmp[i],
                               qop(Iop_MAddF64, rm,
                                   unop(Iop_ReinterpI64asF64,
                                        binop(Iop_GetElem64x2,
                                              getWReg(ws),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI64asF64,
                                        binop(Iop_GetElem64x2,
                                              getWReg(wt),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI64asF64,
                                        binop(Iop_GetElem64x2,
                                              getWReg(wd),
                                              mkU8(i)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   unop(Iop_ReinterpF64asI64,
                                        mkexpr(tmp[1])),
                                   unop(Iop_ReinterpF64asI64,
                                        mkexpr(tmp[0]))));
                     break;
                  }

               default:
                  return -1;
            }

            break;
         }

      case 0x05: { /* FMSUB.df */
            switch (df) {
               case 0x00: { /* FMSUB.W */
                     DIP("FMSUB.W w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMADDW, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     IRTemp tmp[4];
                     Int i;

                     for (i = 0; i < 4; i++) {
                        tmp[i] = newTemp(Ity_F32);
                        assign(tmp[i],
                               qop(Iop_MSubF32, rm,
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(ws),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(wt),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI32asF32,
                                        binop(Iop_GetElem32x4,
                                              getWReg(wd),
                                              mkU8(i)))));
                     }

                     putWReg(wd,
                             binop(Iop_64HLtoV128,
                                   binop(Iop_32HLto64,
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[3])),
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[2]))),
                                   binop(Iop_32HLto64,
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[1])),
                                         unop(Iop_ReinterpF32asI32,
                                              mkexpr(tmp[0])))));
                     break;
                  }

               case 0x01: { /* FMSUB.D */
                     DIP("FMSUB.D w%d, w%d, w%d", wd, ws, wt);
                     calculateMSACSR(ws, wt, FMADDD, 2);
                     IRExpr *rm = get_IR_roundingmode_MSA();
                     IRTemp tmp[2];
                     Int i;

                     for (i = 0; i < 2; i++) {
                        tmp[i] = newTemp(Ity_F64);
                        assign(tmp[i],
                               qop(Iop_MSubF64, rm,
                                   unop(Iop_ReinterpI64asF64,
                                        binop(Iop_GetElem64x2,
                                              getWReg(ws),
                                              mkU8(i))),
                                   unop(Iop_ReinterpI64asF64,
                                        binop(Iop_GetElem64x2,
                                              getWReg(wt),
                    