// <bits/locfacets.h>

// Locale support -*- C++ -*-

// Copyright (C) 1998 Cygnus Solutions
//
// This file is part of the libstdc++ version 3 distribution.
//
// This software is a copyrighted work licensed under the terms of the
// Cygnus libstdc++ license. Please consult the file LICENSE.STD for
// details.

//
// ISO C++ working draft paper: 22.1  Locales
//

/* Warning: this file is not meant for user inclusion.  Use <locale>. */

#ifndef _CPP_BITS_LOCFACETS_H
#define _CPP_BITS_LOCFACETS_H	1

namespace std
{

  // 22.2.1  The ctype category
  struct ctype_base
  {
    enum
    {
      space = _ISspace,
      print = _ISprint,
      cntrl = _IScntrl,
      upper = _ISupper,
      lower = _ISlower,
      alpha = _ISalpha,
      digit = _ISdigit,
      punct = _ISpunct,
      xdigit = _ISxdigit,
      alnum = _ISalnum,
      graph = _ISgraph
    };
    typedef unsigned short mask;
  };


  // 22.2.1.1  Template class ctype


  // _Ctype_nois is the common base for ctype<char> and ctype<T>.
  //    It lacks "do_is" and related virtuals.  These are filled
  //    in by _Ctype, below.

  template<typename _CharT>
    class _Ctype_nois : public locale::facet, public ctype_base
  {
  public:
    typedef _CharT char_type;

  protected:
    explicit _Ctype_nois (size_t __refs = 0) : locale::facet (__refs) {}

  public:
    inline char_type
      toupper (char_type __c) const;
    inline const char_type*
      toupper (char_type *__low, const char_type* __high) const;
    inline char_type
      tolower (char_type __c) const;
    inline const char_type*
      tolower (char_type* __low, const char_type* __high) const;
    inline char_type
      widen (char __c) const;
    inline const char*
      widen (const char* __low, const char* __high, char_type* __to) const;
    inline char
      narrow (char_type __c, char __dfault) const;
    inline const char_type*
      narrow (const char_type* __low, const char_type* __high,
	      char __dfault, char *__to) const;

  protected:
    virtual ~_Ctype_nois ();

    virtual char_type
      do_toupper (char_type) const =0;
    virtual const char_type*
      do_toupper (char_type* __low, const char_type* __high) const =0;
    virtual char_type
      do_tolower (char_type) const =0;
    virtual const char_type*
      do_tolower (char_type* __low, const char_type* __high) const =0;

    virtual char_type
      do_widen (char) const =0;
    virtual const char*
      do_widen (const char* __low, const char* __high,
		char_type* __dest) const =0;
    virtual char
      do_narrow (char_type, char __dfault) const =0;
    virtual const char_type*
      do_narrow (const char_type* __low, const char_type* __high,
		 char __dfault, char* __dest) const =0;
  };

  template<typename _CharT>
    inline _CharT
    _Ctype_nois<_CharT>::toupper (_CharT __c) const
      { return do_toupper (__c); }

  template<typename _CharT>
    const _CharT*
    _Ctype_nois<_CharT>::toupper (_CharT* __low, const _CharT* __high) const
      { return do_toupper (__low, __high); }

  template<typename _CharT>
    inline _CharT
    _Ctype_nois<_CharT>::tolower (_CharT __c) const
      { return do_tolower (__c); }

  template<typename _CharT>
    inline const _CharT*
    _Ctype_nois<_CharT>::tolower (_CharT* __low, const _CharT* __high) const
      { return do_tolower (__low, __high); }

  template<typename _CharT>
    inline _CharT
    _Ctype_nois<_CharT>::widen (char __c) const
      { return do_widen (__c); }

  template<typename _CharT>
    inline const char*
    _Ctype_nois<_CharT>::widen (const char* __low,
			    const char* __high, _CharT* __to) const
      { return do_widen (__low, __high, __to); }

  template<typename _CharT>
    inline char
    _Ctype_nois<_CharT>::narrow (_CharT __c, char __dfault) const
      { return do_narrow (__c, __dfault); }

  template<typename _CharT>
    inline const _CharT*
    _Ctype_nois<_CharT>::narrow (const _CharT* __low, const _CharT* __high,
	      char __dfault, char *__to) const
      { return do_narrow (__low, __high, __dfault, __to); }



  template<typename _CharT>
    class _Ctype : public _Ctype_nois<_CharT>
  {
  public:
    typedef _CharT char_type;

  protected:
    explicit _Ctype (size_t __refs = 0) : _Ctype_nois<_CharT> (__refs) {}

  public:
    inline bool
      is (mask __m, char_type __c) const;
    inline const char_type*
      is (const char_type *__low,
	  const char_type *__high, mask *__vec) const;
    inline const char_type*
      scan_is (mask __m, const char_type* __low,
	       const char_type* __high) const;
    inline const char_type*
      scan_not (mask __m, const char_type* __low,
		const char_type* __high) const;

  protected:
    virtual ~_Ctype ();

    virtual bool
      do_is (mask __m, char_type __c) const =0;
    virtual const char_type*
      do_is (const char_type* __low,
	     const char_type* __high, mask* __vec) const =0;

    virtual const char_type*
      do_scan_is (mask __m, const char_type* __low,
		  const char_type* __high) const =0;
    virtual const char_type*
      do_scan_not (mask __m, const char_type* __low,
		   const char_type* __high) const =0;

  };

  template<typename _CharT>
    inline bool
    _Ctype<_CharT>::is (mask __m, _CharT __c) const
      { return do_is (__m, __c); }

  template<typename _CharT>
    inline const _CharT*
    _Ctype<_CharT>::is (const _CharT* __low,
		       const _CharT* __high, mask* __vec) const
      { return do_is (__low, __high, __vec); }

  template<typename _CharT>
    inline const _CharT*
    _Ctype<_CharT>::scan_is (mask __m, const _CharT* __low,
			      const _CharT* __high) const
      { return do_scan_is (__m, __low, __high); }

  template<typename _CharT>
    inline const _CharT*
    _Ctype<_CharT>::scan_not (mask __m, const _CharT* __low,
			       const _CharT* __high) const
      { return do_scan_not (__m, __low, __high); }



  template<typename _CharT>
    class ctype : public _Ctype<_CharT>
  {
  public:
    typedef _CharT char_type;
    explicit ctype (size_t __refs = 0);
    static locale::id id;
  protected:
    virtual ~ctype ();
    // other members inherited.
  };



  // 22.2.1.3  ctype specializations
  template<>
    class ctype<char> : public _Ctype_nois<char>
  {
  private:
    const mask* _M_table;
    bool _M_del;
    static const int* const& _S_toupper;
    static const int* const& _S_tolower;
    static const mask* const& _S_table;

  public:
    typedef char char_type;

    explicit inline
      ctype (const mask* __tab = 0, bool __del = false,
	     size_t __refs = 0) throw ();

    inline bool
      is (mask __m, char __c) const throw ();
    inline const char*
      is (const char* __low, const char* __high, mask* __vec) const throw ();
    inline const char*
      scan_is (mask __m, const char* __low, const char* __high) const throw ();
    inline const char*
      scan_not (mask __m, const char* __low,
		const char* __high) const throw ();
    inline char_type
      toupper (char_type __c) const;
    inline const char_type*
      toupper (char_type *__low, const char_type* __high) const;
    inline char_type
      tolower (char_type c) const;
    inline const char_type*
      tolower (char_type* __low, const char_type* __high) const;
    inline char_type
      widen (char __c) const;
    inline const char*
      widen (const char* __low, const char* __high, char_type* __to) const;
    inline char
      narrow (char_type __c, char __dfault) const;
    inline const char_type*
      narrow (const char_type* __low, const char_type* __high,
	      char __dfault, char *__to) const;

    static locale::id id;
    static const size_t table_size = 1u + ~(unsigned char)0;

  protected:
    inline const mask* table () const throw ();
    inline static const mask* classic_table () throw ();

    virtual ~ctype ();

    virtual char        do_toupper (char) const;
    virtual const char* do_toupper (char* __low, const char* __high) const;

    virtual char        do_tolower (char) const;
    virtual const char* do_tolower (char* __low, const char* __high) const;

    virtual char        do_widen (char) const;
    virtual const char* do_widen (const char* __low, const char* __high,
				  char* __dest) const;

    virtual char        do_narrow (char, char __dfault) const;
    virtual const char* do_narrow (const char* __low, const char* __high,
				   char __dfault, char* __dest) const;
  };

  // ctype<char> inlines

  inline
  ctype<char>::ctype (const mask* __table, bool __del, size_t __refs) throw ()
      : _Ctype_nois<char> (__refs)
      , _M_table (__table == 0 ? _S_table : __table)
      , _M_del (__table != 0 && __del)
      { }

  inline bool
  ctype<char>::is (mask __m, char __c) const throw ()
      { return _M_table[(unsigned char)(__c)] & __m; }

  inline const char*
  ctype<char>::is (const char* __low, const char* __high,
		  mask* __vec) const throw ()
      {
	while (__low < __high)
	  *__vec++ = _M_table[(unsigned char)(*__low++)];
	return __high;
      }

  inline const char*
  ctype<char>::scan_is (mask __m, const char* __low,
			const char* __high) const throw ()
      {
	while (__low < __high && !(_M_table[(unsigned char)(*__low)] & __m))
	  ++__low;
	return __low;
      }

  inline const char*
  ctype<char>::scan_not (mask __m, const char* __low,
			 const char* __high) const throw ()
      {
	while (__low < __high &&
	       (_M_table[(unsigned char)(*__low)] & __m) != 0)
	  ++__low;
	return __low;
      }

  inline const ctype_base::mask*
  ctype<char>::table () const throw ()
    { return _M_table; }

  inline const ctype_base::mask*
  ctype<char>::classic_table () throw ()
    { return _S_table; }

  inline char
  ctype<char>::toupper (char __c) const
      { return do_toupper (__c); }

  inline const char*
  ctype<char>::toupper (char* __low, const char* __high) const
      { return do_toupper (__low, __high); }

  inline char
  ctype<char>::tolower (char __c) const { return do_tolower (__c); }

  inline const char*
  ctype<char>::tolower (char* __low, const char* __high) const
      { return do_tolower (__low, __high); }

  inline char
  ctype<char>::widen (char __c) const
      { return do_widen (__c); }

  inline const char*
  ctype<char>::widen (const char* __low,
			 const char* __high, char* __to) const
      { return do_widen (__low, __high, __to); }

  inline char
  ctype<char>::narrow (char __c, char __dfault) const
      { return do_narrow (__c, __dfault); }

  inline const char*
  ctype<char>::narrow (const char* __low, const char* __high,
	      char __dfault, char *__to) const
      { return do_narrow (__low, __high, __dfault, __to); }

  // ctype<wchar_t> specialization

  template<>
    class ctype<wchar_t> : public _Ctype<wchar_t>
  {
  public:
    typedef wchar_t char_type;
    explicit ctype (size_t __refs = 0);
    static locale::id id;

  private:
    static const int*& _S_toupper;
    static const int*& _S_tolower;
    static const mask*& _S_table;
    static const int _S_table_size = ctype<char>::table_size;

  protected:
    virtual ~ctype ();

    virtual bool
      do_is (mask __m, char_type __c) const;
    virtual const char_type*
      do_is (const char_type* __low,
	     const char_type* __high, mask* __vec) const;

    virtual const char_type*
      do_scan_is (mask __m, const char_type* __low,
		  const char_type* __high) const;
    virtual const char_type*
      do_scan_not (mask __m, const char_type* __low,
		   const char_type* __high) const;

    virtual char_type
      do_toupper (char_type) const;
    virtual const char_type*
      do_toupper (char_type* __low, const char_type* __high) const;
    virtual char_type
      do_tolower (char_type) const;
    virtual const char_type*
      do_tolower (char_type* __low, const char_type* __high) const;

    virtual char_type
      do_widen (char) const;
    virtual const char*
      do_widen (const char* __low, const char* __high,
		char_type* __dest) const;
    virtual char
      do_narrow (char_type, char __dfault) const;
    virtual const char_type*
      do_narrow (const char_type* __low, const char_type* __high,
		 char __dfault, char* __dest) const;
  };


  template<>
    inline const ctype<char>&
    use_facet< const ctype<char> > (const locale& __loc)
  {
    return static_cast<const ctype<char>&>(
      *__loc._M_impl->_M_facets[ctype<char>::id._M_index]);
  }

  template<>
    inline const ctype<wchar_t>&
    use_facet< const ctype<wchar_t> > (const locale& __loc)
  {
    return static_cast<const ctype<wchar_t>&>(
      *__loc._M_impl->_M_facets[ctype<wchar_t>::id._M_index]);
  }

  // 22.2.1.2  Template class ctype_byname
  ////////////////////////////////////////

  template<typename _CharT>
    class ctype_byname : public ctype<_CharT>
  {
  public:
    typedef _CharT char_type;
    explicit ctype_byname (const char*, size_t __refs = 0);

  protected:
    virtual ~ctype_byname ();
  };


  //  22.2.1.4  Class ctype_byname<char>

  template<>
    ctype_byname<char>::ctype_byname (const char*, size_t __refs);

  template<>
    ctype_byname<wchar_t>::ctype_byname (const char*, size_t __refs);

  //  22.2.1.5  Template class codecvt
  class codecvt_base
  {
  public:
    enum result
      {
	ok,
	partial,
	error,
	noconv
      };
  };

  template<typename _InternT, typename _ExternT, typename _StateT>
    class _Codecvt : public locale::facet, public codecvt_base
  {
  public:
    typedef _InternT intern_type;
    typedef _ExternT extern_type;
    typedef _StateT  state_type;

  protected:
    explicit _Codecvt (size_t __refs = 0);

  public:
    inline result
      out (state_type& __state,
	   const intern_type* __from, const intern_type* __from_end,
	   const intern_type* &__from_next,
	   extern_type* __to, extern_type* __to_limit,
	   extern_type*& __to_next) const;

    inline result
      unshift (state_type& __state,
	       extern_type* __to, extern_type* __to_limit,
	       extern_type*& __to_next) const;

    inline result
      in (state_type& __state,
	  const extern_type* __from, const extern_type* __from_end,
	  const extern_type*& __from_next,
	  intern_type* __to, intern_type* __to_limit,
	  intern_type*& __to_next) const;

    inline int encoding () const throw ();
    inline bool always_noconv () const throw ();
    inline int
      length (const state_type&, const extern_type* __from,
	      const extern_type* __end, size_t __max) const;
    inline int max_length () const throw ();

  protected:
    virtual ~_Codecvt ();
    virtual result
      do_out (state_type& __state,
	      const intern_type* __from, const intern_type* __from_end,
	      const intern_type*& __from_next,
	      extern_type* __to, extern_type* __to_limit,
	      extern_type*& __to_next) const =0;

    virtual result
      do_unshift (state_type& __state,
		  extern_type* __to, extern_type* __to_limit,
		  extern_type*& __to_next) const =0;

    virtual result
      do_in (state_type& __state,
	     const extern_type* __from, const extern_type* __from_end,
	     const extern_type*& __from_next,
	     intern_type* __to, intern_type* __to_limit,
	     intern_type*& __to_next) const =0;

    virtual int do_encoding () const throw () =0;
    virtual bool do_always_noconv () const throw () =0;
    virtual int do_length (const state_type&, const extern_type* __from,
			   const extern_type* __end, size_t __max) const =0;
    virtual int do_max_length () const throw () =0;
  };

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline codecvt_base::result
    _Codecvt<_InternT,_ExternT,_StateT>::out (state_type& __state,
	const intern_type* __from, const intern_type* __from_end,
	const intern_type* &__from_next,
	extern_type* __to, extern_type* __to_limit,
	extern_type*& __to_next) const
      { return do_out (__state,__from,__from_end,__from_next,
       		       __to,__to_limit,__to_next); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline codecvt_base::result
    _Codecvt<_InternT,_ExternT,_StateT>::unshift (state_type& __state,
	  extern_type* __to, extern_type* __to_limit,
	  extern_type*& __to_next) const
      { return do_unshift (__state, __to,__to_limit,__to_next); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline codecvt_base::result
    _Codecvt<_InternT,_ExternT,_StateT>::in (state_type& __state,
	  const extern_type* __from, const extern_type* __from_end,
	  const extern_type*& __from_next,
	  intern_type* __to, intern_type* __to_limit,
	  intern_type*& __to_next) const
      { return do_in (__state,__from,__from_end,__from_next,
       		      __to,__to_limit,__to_next); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline int
    _Codecvt<_InternT,_ExternT,_StateT>::encoding () const throw ()
      { return do_encoding (); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline bool
    _Codecvt<_InternT,_ExternT,_StateT>::always_noconv () const throw ()
      { return do_always_noconv (); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline int
    _Codecvt<_InternT,_ExternT,_StateT>::length (const state_type& __state,
						 const extern_type* __from,
						 const extern_type* __end,
						 size_t __max) const
      { return do_length (__state,__from,__end,__max); }

  template<typename _InternT, typename _ExternT, typename _StateT>
    inline int
    _Codecvt<_InternT,_ExternT,_StateT>::max_length () const throw ()
      { return do_max_length (); }

  ///////////
  template<typename _InternT, typename _ExternT, typename _StateT>
    class codecvt : public _Codecvt<_InternT,_ExternT,_StateT>
  {
  public:
    typedef _InternT intern_type;
    typedef _ExternT extern_type;
    typedef _StateT state_type;

    explicit codecvt (size_t __refs = 0);
    static locale::id id;
  protected:
    virtual ~codecvt ();
  };

  // codecvt<char,char,mbstate_t> specialization

  template<>
    class codecvt<char,char,mbstate_t> : public _Codecvt<char,char,mbstate_t>
  {
  public:
    typedef char intern_type;
    typedef char extern_type;
    typedef mbstate_t state_type;

    explicit codecvt (size_t __refs = 0);
    static locale::id id;

  protected:
    virtual ~codecvt ();
    virtual result
      do_out (state_type& __state,
	      const intern_type* __from, const intern_type* __from_end,
	      const intern_type*& __from_next,
	      extern_type* __to, extern_type* __to_limit,
	      extern_type*& __to_next) const;

    virtual result
      do_unshift (state_type& __state,
		  extern_type* __to, extern_type* __to_limit,
		  extern_type*& __to_next) const;

    virtual result
      do_in (state_type& __state,
	     const extern_type* __from, const extern_type* __from_end,
	     const extern_type*& __from_next,
	     intern_type* __to, intern_type* __to_limit,
	     intern_type*& __to_next) const;

    virtual int do_encoding () const throw ();
    virtual bool do_always_noconv () const throw ();
    virtual int do_length (const state_type&, const extern_type* __from,
			   const extern_type* __end, size_t __max) const;
    virtual int do_max_length () const throw ();
  };

  // codecvt<wchar_t,char,mbstate_t> specialization

  template<>
    class codecvt<wchar_t,char,mbstate_t>
      : public _Codecvt<wchar_t,char,mbstate_t>
  {
  public:
    typedef wchar_t intern_type;
    typedef char extern_type;
    typedef mbstate_t state_type;

    explicit codecvt (size_t __refs = 0);
    static locale::id id;

  protected:
    virtual ~codecvt ();
    virtual result
      do_out (state_type& __state,
	      const intern_type* __from, const intern_type* __from_end,
	      const intern_type*& __from_next,
	      extern_type* __to, extern_type* __to_limit,
	      extern_type*& __to_next) const;

    virtual result
      do_unshift (state_type& __state,
		  extern_type* __to, extern_type* __to_limit,
		  extern_type*& __to_next) const;

    virtual result
      do_in (state_type& __state,
	     const extern_type* __from, const extern_type* __from_end,
	     const extern_type*& __from_next,
	     intern_type* __to, intern_type* __to_limit,
	     intern_type*& __to_next) const;

    virtual int do_encoding () const throw ();
    virtual bool do_always_noconv () const throw ();
    virtual int do_length (const state_type&, const extern_type* __from,
			   const extern_type* __end, size_t __max) const;
    virtual int do_max_length () const throw ();
  };

  // 22.2.1.6  Template class codecvt_byname

  template<typename _InternT, typename _ExternT, typename _StateT>
    class codecvt_byname : public codecvt<_InternT,_ExternT,_StateT>
  {
  public:
    explicit codecvt_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~codecvt_byname ();
  };

  template<>
    class codecvt_byname<char,char,mbstate_t>
      : public codecvt<char,char,mbstate_t>
  {
  public:
    explicit codecvt_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~codecvt_byname ();
  };

  template<>
    class codecvt_byname<wchar_t,char,mbstate_t>
      : public codecvt<wchar_t,char,mbstate_t>
  {
  public:
    explicit codecvt_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~codecvt_byname ();
  };

  template <typename _CharT, typename _InIter>
    class _Numeric_get;  // forward

  // _Format_cache holds the information extracted from the
  //   numpunct<> and moneypunct<> facets in a form optimized
  //   for parsing and formatting.  It is stored via a void*
  //   pointer in the pword() array of an iosbase object passed
  //   to the _get and _put facets.

  template <typename _CharT>
    class _Format_cache
  {
   // warning: contains no user-serviceable parts.
  public: // XXX
    _Format_cache ();
    ~_Format_cache () throw();
    static _Format_cache<_CharT>* _S_get(ios_base& __ios);
    static int _S_pword_ix;  // ios_base::pword() reserved cell

    bool _M_valid;

#   define _FORMAT_CACHE_NUMERIC_LITERALS \
       "-+xX0123456789abcdef0123456789ABCDEF"
    static const int _S_literal_count =
      -1 + sizeof(_FORMAT_CACHE_NUMERIC_LITERALS);
    _CharT _M_literals[_S_literal_count];

    // note: code depends on the order of definitions of the names
    enum {  // these are indices into _FORMAT_CACHE_NUMERIC_LITERALS above
      _S_minus, _S_plus, _S_ecks, _S_Ecks, _S_digits,
      _S_digits_end = _S_digits+16,
      _S_udigits = _S_digits_end,  _S_udigits_end = _S_udigits + 16,
      _S_ee = _S_digits + 14, _S_Ee = _S_udigits + 14
    };

    _CharT _M_decsep;
    _CharT _M_grsep;

    basic_string<_CharT> _M_boolnames[2];

    bool _M_use_grouping;
    string _M_grouping;

    void _M_populate (ios_base&);
    static void _S_callback (ios_base::event __event,
			     ios_base& __ios, int __ix) throw();

#if 0  /* XXX compiler bug? */
    template <typename _CharT, typename _InIter>
      friend class _Numeric_get;
    friend class locale;
    template <class _Ch,class _InIter> friend class num_get<_Ch,_InIter>;
    template <class _Ch,class _InIter> friend class time_get<_Ch,_InIter>;
    template <class _Ch,class _InIter> friend class money_get<_Ch,_InIter>;
    template <class _Ch,class _OutIter> friend class num_put<_Ch,_OutIter>;
    template <class _Ch,class _OutIter> friend class time_put<_Ch,_OutIter>;
    template <class _Ch,class _OutIter> friend class money_put<_Ch,_OutIter>;
#endif
  };

  // specialize default ctors, for efficiency.  Defs are in src/ios.cc
  template <> _Format_cache<char>::_Format_cache();
  template <> _Format_cache<wchar_t>::_Format_cache();

  // _Numeric_get is used by num_get, money_get, and time_get
  //   to help in parsing out numbers.

  template <typename _CharT, typename _InIter>
    class _Numeric_get
  {
  public:
    typedef _CharT     char_type;
    typedef _InIter    iter_type;
  private:
    explicit _Numeric_get ();
    virtual ~_Numeric_get ();

    iter_type _M_get_digits(iter_type __in, iter_type __end) const;

    template <typename _CharTparm, typename _InIterparm>
      friend class num_get;
    template <typename _CharTparm, typename _InIterparm>
      friend class time_get;
    template <typename _CharTparm, typename _InIterparm>
      friend class money_get;
    template <typename _CharTparm, typename _InIterparm>
      friend class num_put;
    template <typename _CharTparm, typename _InIterparm>
      friend class time_put;
    template <typename _CharTparm, typename _InIterparm>
      friend class money_put;
  };

  // num_get

  template <typename _CharT, typename _InIter>
    class num_get : public locale::facet
  {
  public:
    typedef _CharT   char_type;
    typedef _InIter  iter_type;
    explicit num_get (size_t __refs = 0);
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, bool& __v)           const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, long& __v)           const;
#if 0 // XXX need appropriate macro guard for this.
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, long long& __v)      const;
#endif
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, unsigned short& __v) const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, unsigned int& __v)   const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, unsigned long& __v)  const;
#if 0 // XXX need appropriate macro guard for this.
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
		 _Iostate& __err, unsigned long long& __v)  const;
#endif
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, float& __v)          const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, double& __v)         const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, long double& __v)    const;
    inline iter_type get (iter_type __in, iter_type __end, ios_base& __io,
			 _Iostate& __err, void*& __v)          const;

    static locale::id id;

  protected:
    virtual ~num_get ();
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, bool&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, long&) const;
#if 0 // XXX
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, long long&) const;
#endif
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, unsigned short&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, unsigned int&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, unsigned long&) const;
#if 0 // XXX
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
	     _Iostate& __err, unsigned long long&) const;
#endif
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, float&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, double&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, long double&) const;
    virtual iter_type do_get (iter_type, iter_type, ios_base&,
			     _Iostate& __err, void*&) const;
  };

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  bool& __v)           const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  long& __v)           const
     { return do_get (__in,__end,__io,__err,__v); }

#if 0
  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  long long& __v)      const
     { return do_get (__in,__end,__io,__err,__v); }
#endif

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  unsigned short& __v) const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  unsigned int& __v)   const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  unsigned long& __v)  const
     { return do_get (__in,__end,__io,__err,__v); }

#if 0 // XXX
  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  unsigned long long& __v)  const
     { return do_get (__in,__end,__io,__err,__v); }
#endif

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  float& __v)          const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  double& __v)         const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  long double& __v)    const
     { return do_get (__in,__end,__io,__err,__v); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    num_get<_CharT,_InIter>::get (iter_type __in, iter_type __end,
				  ios_base& __io, _Iostate& __err,
				  void*& __v)          const
     { return do_get (__in,__end,__io,__err,__v); }


  // _Numeric_put is used by num_put, money_put, and time_put
  //   to help in formatting out numbers.

  template <typename _CharT, typename _OutIter>
    class _Numeric_put
  {
  public:
    typedef _CharT      char_type;
    typedef _OutIter    iter_type;
  protected:
    explicit _Numeric_put ();
  protected:
    virtual ~_Numeric_put ();
  };

  ////////////
  template <typename _CharT, typename _OutIter>
    class num_put : public locale::facet
  {
  public:
    typedef _CharT       char_type;
    typedef _OutIter     iter_type;
    explicit num_put (size_t __refs = 0);
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 bool __v) const;
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 long __v) const;
#if 0 // XXX
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 long long __v) const;
#endif
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 unsigned long __v) const;
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 double __v) const;
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 long double __v) const;
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
			 const void* __v) const;
    static locale::id id;
  protected:
    virtual ~num_put ();

    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     bool __v) const;
    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     long __v) const;
    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     unsigned long) const;
    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     double __v) const;
    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     long double __v) const;
    virtual iter_type do_put (iter_type, ios_base&, char_type __fill,
			     const void* __v) const;
  };

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, bool __v) const
      { return do_put (__s,__f,__fill,__v); }

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, long __v) const
      { return do_put (__s,__f,__fill,__v); }

#if 0
  // XXX - long long versions of do_put aren't declared or defined.
  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, long long __v) const
      { return do_put (__s,__f,__fill,__v); }
#endif

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, unsigned long __v) const
      { return do_put (__s,__f,__fill,__v); }

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, double __v) const
      { return do_put (__s,__f,__fill,__v); }

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				  char_type __fill, long double __v) const
      { return do_put (__s,__f,__fill,__v); }

  template <typename _CharT, typename _OutIter>
    inline _OutIter
      num_put<_CharT,_OutIter>::put (_OutIter __s, ios_base& __f,
				    char_type __fill, const void* __v) const
      { return do_put (__s,__f,__fill,__v); }



  template <typename _CharT>
    class _Punct : public locale::facet
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;
  protected:
    explicit _Punct (size_t __refs = 0);
  public:
    inline char_type    decimal_point ()   const;
    inline char_type    thousands_sep ()   const;
    inline string       grouping ()        const;

  protected:
    virtual ~_Punct ();
    virtual char_type    do_decimal_point () const;
    virtual char_type    do_thousands_sep () const;
    virtual string       do_grouping ()      const;

  private:
    char_type _M_decimal_point;
    char_type _M_thousands_sep;
    string    _M_grouping;

  protected:
    // for use at construction time only:
    void _M_init (char_type, char_type, const string&);
  };

  template <typename _CharT>
    inline _CharT
    _Punct<_CharT>::decimal_point () const { return do_decimal_point (); }

  template <typename _CharT>
    inline _CharT
    _Punct<_CharT>::thousands_sep () const { return do_thousands_sep (); }

  template <typename _CharT>
    inline string
    _Punct<_CharT>::grouping ()      const { return do_grouping (); }


  template <typename _CharT>
    class _Numpunct : public _Punct<_CharT>
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;
  protected:
    explicit _Numpunct (size_t __refs = 0);
  public:
    inline string_type  truename ()        const;
    inline string_type  falsename ()       const;

  protected:
    virtual ~_Numpunct ();
    virtual string_type  do_truename ()      const;      // for bool
    virtual string_type  do_falsename ()     const;      // for bool

  private:
    string_type _M_truename;
    string_type _M_falsename;

  protected:
    // for use only during construction
    void _M_init_boolnames (const string_type& __t, const string_type& __f);
  };

  template <typename _CharT>
    inline basic_string<_CharT>
    _Numpunct<_CharT>::truename ()      const { return do_truename (); }

  template <typename _CharT>
    inline basic_string<_CharT>
    _Numpunct<_CharT>::falsename ()     const { return do_falsename (); }

  template <typename _CharT>
    class numpunct : public _Numpunct<_CharT>
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;
    explicit numpunct (size_t __refs = 0);

    static locale::id id;
  protected:
    virtual ~numpunct ();
  };

  template <> numpunct<char>::numpunct (size_t __refs);
  template <> numpunct<wchar_t>::numpunct (size_t __refs);

  template <typename _CharT>
    class numpunct_byname : public numpunct<_CharT>
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;
    explicit numpunct_byname (const char*, size_t __refs = 0);

  protected:
    virtual ~numpunct_byname ();
  };

  template <>
    numpunct_byname<char>::numpunct_byname (const char*, size_t __refs);
  template <>
    numpunct_byname<wchar_t>::numpunct_byname (const char*, size_t __refs);


  /////////

  template <typename _CharT>
    class _Collate : public locale::facet
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;
  protected:
    explicit _Collate (size_t __refs = 0);
  public:
    inline int compare (const _CharT* __lo1, const _CharT* __hi1,
	 	       const _CharT* __lo2, const _CharT* __hi2) const;
    inline string_type transform (const _CharT* __lo,
				  const _CharT* __hi) const;
    inline long hash (const _CharT* __lo, const _CharT* __hi) const;
  protected:
   ~_Collate ();                  // virtual
    virtual int  do_compare (const _CharT* __lo1, const _CharT* __hi1,
			    const _CharT* __lo2, const _CharT* __hi2) const =0;
    virtual string_type do_transform
			     (const _CharT* __lo, const _CharT* __hi) const =0;
    virtual long   do_hash   (const _CharT* __lo, const _CharT* __hi) const =0;
  };

  template <typename _CharT>
    inline int
    _Collate<_CharT>::compare (const _CharT* __lo1, const _CharT* __hi1,
	 		      const _CharT* __lo2, const _CharT* __hi2) const
      { return do_compare(__lo1, __hi1, __lo2, __hi2); }

  template <typename _CharT>
    inline basic_string<_CharT>
    _Collate<_CharT>::transform (const _CharT* __lo, const _CharT* __hi) const
      { return do_transform(__lo, __hi); }

  template <typename _CharT>
    inline long
    _Collate<_CharT>::hash (const _CharT* __lo, const _CharT* __hi) const
      { return do_hash(__lo, __hi); }


  template <typename _CharT>
    class collate : public _Collate<_CharT>
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;

    explicit collate (size_t __refs = 0);
    static locale::id id;

  protected:
    virtual ~collate ();
  };

  template <>
    class collate<char> : public _Collate<char>
  {
  public:
    typedef char               char_type;
    typedef basic_string<char> string_type;

    explicit collate (size_t __refs = 0);
    static locale::id id;

  protected:
    virtual ~collate ();
    virtual int    do_compare (const char* __lo1, const char* __hi1,
			      const char* __lo2, const char* __hi2) const;
    virtual string_type do_transform
			     (const char* __lo, const char* __hi) const;
    virtual long   do_hash   (const char* __lo, const char* __hi) const;
  };

  template <>
    class collate<wchar_t> : public _Collate<wchar_t>
  {
  public:
    typedef wchar_t               char_type;
    typedef basic_string<wchar_t> string_type;

    explicit collate (size_t __refs = 0);
    static locale::id id;

  protected:
    virtual ~collate ();
    virtual int   do_compare (const wchar_t* __lo1, const wchar_t* __hi1,
			     const wchar_t* __lo2, const wchar_t* __hi2) const;
    virtual string_type do_transform
			     (const wchar_t* __lo, const wchar_t* __hi) const;
    virtual long   do_hash   (const wchar_t* __lo, const wchar_t* __hi) const;
  };


  template <typename _CharT>
    class collate_byname : public collate<_CharT>
  {
  public:
    typedef _CharT               char_type;
    typedef basic_string<_CharT> string_type;

    explicit collate_byname (const char*, size_t __refs = 0);

  protected:
    virtual ~collate_byname ();
  };

  template <>
    collate_byname<char>::collate_byname (const char*, size_t __refs);
  template <>
    collate_byname<wchar_t>::collate_byname (const char*, size_t __refs);

  /////////////////
  class time_base
  {
    public:
      enum dateorder { no_order, dmy, mdy, ymd, ydm };
  };

  template <typename _CharT, typename _InIter>
    class time_get : public locale::facet, public time_base
  {
  public:
    typedef _CharT     char_type;
    typedef _InIter    iter_type;
    explicit time_get (size_t __refs = 0);

    inline dateorder date_order ()  const;
    inline iter_type get_time (iter_type __s, iter_type __end, ios_base& __f,
		       _Iostate& __err, tm* __t)  const;
    inline iter_type get_date (iter_type __s, iter_type __end, ios_base& __f,
		       _Iostate& __err, tm* __t)  const;
    inline iter_type get_weekday (iter_type __s, iter_type __end, ios_base& __f,
		       _Iostate& __err, tm* __t) const;
    inline iter_type get_monthname (iter_type __s, iter_type __end,
				   ios_base& __f, _Iostate& __err,
				   tm* __t) const;
    inline iter_type get_year (iter_type __s, iter_type __end, ios_base& __f,
		       _Iostate& __err, tm* __t) const;
    static locale::id id;
  protected:
    virtual ~time_get ();
    virtual dateorder do_date_order ()  const;
    virtual iter_type do_get_time (iter_type __s, iter_type __end, ios_base&,
				  _Iostate& __err, tm* __t) const;
    virtual iter_type do_get_date (iter_type __s, iter_type __end, ios_base&,
				  _Iostate& __err, tm* __t) const;
    virtual iter_type do_get_weekday (iter_type __s, iter_type __end, ios_base&,
				      _Iostate& __err, tm* __t) const;
    virtual iter_type do_get_monthname (iter_type __s, iter_type __end,
		                        ios_base&, _Iostate& __err,
					tm* __t) const;
    virtual iter_type do_get_year (iter_type __s, iter_type __end, ios_base&,
				  _Iostate& __err, tm* __t) const;

    mutable basic_string<_CharT>* _M_daynames;
    mutable basic_string<_CharT>* _M_monthnames;
  };

  template <typename _CharT, typename _InIter>
    inline time_base::dateorder
    time_get<_CharT,_InIter>::date_order ()  const
      { return do_date_order (); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    time_get<_CharT,_InIter>::get_time (iter_type __s, iter_type __end,
				       ios_base& __f, _Iostate& __err,
				       tm* __t)  const
      { return do_get_time (__s,__end,__f,__err,__t); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    time_get<_CharT,_InIter>::get_date (iter_type __s, iter_type __end,
				       ios_base& __f, _Iostate& __err,
				       tm* __t)  const
      { return do_get_date (__s,__end,__f,__err,__t); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    time_get<_CharT,_InIter>::get_weekday (iter_type __s, iter_type __end,
				          ios_base& __f,
					  _Iostate& __err,
					  tm* __t) const
      { return do_get_weekday (__s,__end,__f,__err,__t); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    time_get<_CharT,_InIter>::get_monthname (iter_type __s, iter_type __end,
					    ios_base& __f,
					    _Iostate& __err,
					    tm* __t) const
      { return do_get_monthname (__s,__end,__f,__err,__t); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    time_get<_CharT,_InIter>::get_year (iter_type __s, iter_type __end,
 				       ios_base& __f, _Iostate& __err,
				       tm* __t) const
      { return do_get_year (__s,__end,__f,__err,__t); }


  template <typename _CharT, typename _InIter>
    class time_get_byname : public time_get<_CharT,_InIter>
  {
  public:
    typedef _CharT     char_type;
    typedef _InIter    iter_type;
    explicit time_get_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~time_get_byname ();
  };

  //////////
  template <typename _CharT, typename _OutIter>
    class time_put : public locale::facet, public time_base
  {
  public:
    typedef _CharT     char_type;
    typedef _OutIter   iter_type;
    explicit time_put (size_t __refs = 0);
    iter_type put (iter_type __s, ios_base& __f,
		  char_type __fill, const tm* __tmb,
		  const _CharT* __pattern, const _CharT* __pat_end) const;
    inline iter_type put (iter_type __s, ios_base& __f, char_type __fill,
                  const tm* __tmb, char __format, char __modifier = 0) const;

    static locale::id id;
  protected:
    virtual ~time_put ();
    virtual iter_type do_put (iter_type __s, ios_base&, char_type,
			      const tm* __t, char __format,
			      char __modifier) const;
  };

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    time_put<_CharT,_OutIter>::put (iter_type __s, ios_base& __f,
				   char_type __fill, const tm* __tmb,
				   char __format, char __modifier) const
      { return do_put (__s,__f,__fill,__tmb,__format,__modifier); }


  template <typename _CharT, typename _OutIter>
    class time_put_byname : time_put<_CharT,_OutIter>
  {
  public:
    typedef _CharT     char_type;
    typedef _OutIter   iter_type;
    explicit time_put_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~time_put_byname ();
  };


  template <typename _CharT, typename _InIter>
    class money_get : public locale::facet
  {
  public:
    typedef _CharT        char_type;
    typedef _InIter       iter_type;
    typedef basic_string<_CharT> string_type;
    explicit money_get (size_t __refs = 0);
    inline iter_type get (iter_type __s, iter_type __end, bool __intl,
		  ios_base& __f, _Iostate& __err,
		  long double& __units) const;
    inline iter_type get (iter_type __s, iter_type __end, bool __intl,
		  ios_base& __f, _Iostate& __err,
		  string_type& __digits) const;
    static locale::id id;
  protected:
    virtual ~money_get ();
    virtual iter_type do_get (iter_type, iter_type, bool, ios_base&,
			     _Iostate&, long double& __units) const;
    virtual iter_type do_get (iter_type, iter_type, bool, ios_base&,
			     _Iostate&, string_type& __digits) const;
  };

  template <typename _CharT, typename _InIter>
    inline _InIter
    money_get<_CharT,_InIter>::get (iter_type __s, iter_type __end,
				    bool __intl, ios_base& __f,
				    _Iostate& __err,
				    long double& __units) const
      { return do_get (__s,__end,__intl,__f,__err,__units); }

  template <typename _CharT, typename _InIter>
    inline _InIter
    money_get<_CharT,_InIter>::get (iter_type __s, iter_type __end,
				    bool __intl, ios_base& __f,
				    _Iostate& __err,
				    string_type& __digits) const
      { return do_get (__s,__end,__intl,__f,__err,__digits); }

  /////////////
  template <typename _CharT, typename _OutIter>
    class money_put : public locale::facet
  {
  public:
    typedef _CharT              char_type;
    typedef _OutIter            iter_type;
    typedef basic_string<_CharT> string_type;
    explicit money_put (size_t __refs = 0);
    inline iter_type put (iter_type __s, bool __intl, ios_base& __f,
			  char_type __fill, long double __units) const;
    inline iter_type put (iter_type __s, bool __intl, ios_base& __f,
			  char_type __fill, const string_type& __digits) const;
    static locale::id id;

  protected:
    virtual ~money_put ();
    virtual iter_type
      do_put (iter_type, bool, ios_base&, char_type __fill,
	      long double __units) const;
    virtual iter_type
      do_put (iter_type, bool, ios_base&, char_type __fill,
	      const string_type& __digits) const;
  };

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    money_put<_CharT,_OutIter>::put (iter_type __s, bool __intl,
				     ios_base& __f, char_type __fill,
				     long double __units) const
      { return do_put (__s,__intl,__f,__fill,__units); }

  template <typename _CharT, typename _OutIter>
    inline _OutIter
    money_put<_CharT,_OutIter>::put (iter_type __s, bool __intl,
				     ios_base& __f, char_type __fill,
				     const string_type& __digits) const
      { return do_put (__s,__intl,__f,__fill,__digits); }

  ////////////////
  struct money_base
  {
      enum part { none, space, symbol, sign, value };
      struct pattern { char field[4]; };
  };

  template<typename _CharT>
    class _Moneypunct : public _Punct<_CharT>, public money_base
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
  protected:
    explicit _Moneypunct (size_t __refs = 0);
  public:
    inline string_type  curr_symbol ()   const;
    inline string_type  positive_sign () const;
    inline string_type  negative_sign () const;
    inline int          frac_digits ()   const;
    inline pattern      pos_format ()    const;
    inline pattern      neg_format ()    const;
  protected:
    virtual ~_Moneypunct ();
    virtual string_type  do_curr_symbol ()   const;
    virtual string_type  do_positive_sign () const;
    virtual string_type  do_negative_sign () const;
    virtual int          do_frac_digits ()   const;
    virtual pattern      do_pos_format ()    const;
    virtual pattern      do_neg_format ()    const;
  };

  template <typename _CharT>
    inline basic_string<_CharT>
    _Moneypunct<_CharT>::curr_symbol ()   const
      { return do_curr_symbol (); }
  template <typename _CharT>
    inline basic_string<_CharT>
    _Moneypunct<_CharT>::positive_sign () const
      { return do_positive_sign (); }
  template <typename _CharT>
    inline basic_string<_CharT>
    _Moneypunct<_CharT>::negative_sign () const
      { return do_negative_sign (); }
  template <typename _CharT>
    inline int
    _Moneypunct<_CharT>::frac_digits ()   const
      { return do_frac_digits (); }
  template <typename _CharT>
    inline money_base::pattern
    _Moneypunct<_CharT>::pos_format ()    const
      { return do_pos_format (); }
  template <typename _CharT>
    inline money_base::pattern
    _Moneypunct<_CharT>::neg_format ()    const
      { return do_neg_format (); }


  template <typename _CharT, bool _Intl>
    class moneypunct : public _Moneypunct<_CharT>
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
    static const bool intl = _Intl;
    explicit moneypunct (size_t __refs = 0);
    static locale::id id;
  protected:
    virtual ~moneypunct ();
  };

  template <> moneypunct<char,false>::moneypunct (size_t __refs);
  template <> moneypunct<char,true>::moneypunct (size_t __refs);
  template <> moneypunct<wchar_t,false>::moneypunct (size_t __refs);
  template <> moneypunct<wchar_t,true>::moneypunct (size_t __refs);

  template <typename _CharT, bool _Intl>
    class moneypunct_byname : public moneypunct<_CharT,_Intl>
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
    static const bool intl = _Intl;
    explicit moneypunct_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~moneypunct_byname ();
  };

  template <>
    moneypunct_byname<char,false>::moneypunct_byname(const char*,
						     size_t __refs);
  template <>
    moneypunct_byname<char,true>::moneypunct_byname (const char*,
						     size_t __refs);
  template <>
    moneypunct_byname<wchar_t,false>::moneypunct_byname (const char*,
							 size_t __refs);
  template <>
    moneypunct_byname<wchar_t,true>::moneypunct_byname (const char*,
							size_t __refs);

  ////////////
  struct messages_base
  {
    typedef int catalog;
  };

  template <typename _CharT>
    class _Messages : public locale::facet, public messages_base
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
  protected:
    explicit _Messages (size_t __refs = 0);
  public:
    inline catalog open (const basic_string<char>& __fn, const locale&) const;
    inline string_type  get (catalog __c, int __set, int __msgid,
			     const string_type& __dfault) const;
    inline void close (catalog __c) const;
  protected:
    virtual ~_Messages ();
    // probably these should be pure, and implemented only in
    //  specializations of messages<>.  But for now...
    virtual catalog do_open (const basic_string<char>&, const locale&) const;
    virtual string_type  do_get (catalog, int __set, int __msgid,
				 const string_type& __dfault) const;
    virtual void    do_close (catalog) const;
  };

  template <typename _CharT>
    inline messages_base::catalog
    _Messages<_CharT>::open (const basic_string<char>& __fn,
			   const locale& __loc) const
      { return do_open (__fn,__loc); }

  template <typename _CharT>
    inline basic_string<_CharT>
    _Messages<_CharT>::get (catalog __c, int __set, int __msgid,
			  const string_type& __dfault) const
      { return do_get (__c,__set,__msgid,__dfault); }

  template <typename _CharT>
    inline void
    _Messages<_CharT>::close (catalog __c) const
      { return do_close (__c); }

  template <typename _CharT>
    class messages : public _Messages<_CharT>
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
    explicit messages (size_t __refs = 0);
    static locale::id id;
  protected:
    virtual ~messages ();
  };

  template <> messages<char>::messages (size_t __refs);
  template <> messages<wchar_t>::messages (size_t __refs);

  template <typename _CharT>
    class messages_byname : public messages<_CharT>
  {
  public:
    typedef _CharT char_type;
    typedef basic_string<_CharT> string_type;
    explicit messages_byname (const char*, size_t __refs = 0);
  protected:
    virtual ~messages_byname ();
  };

  template <>
    messages_byname<char>::messages_byname (const char*, size_t __refs);
  template <>
    messages_byname<wchar_t>::messages_byname (const char*, size_t __refs);

    // subclause convenience interfaces, inlines

    // note: these are inline because, when used in a loop,
    // some compilers can hoist the body out of the loop; then
    // it's just as fast as the C is*() function.

  template<typename _CharT>
    inline bool isspace (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::space, __c); }
  template<typename _CharT>
    inline bool isprint (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::print, __c); }
  template<typename _CharT>
    inline bool iscntrl (_CharT __c, const locale& __loc)
    { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::cntrl, __c); }
  template<typename _CharT>
    inline bool isupper (_CharT __c, const locale& __loc)
    { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::upper, __c); }
  template<typename _CharT>
    inline bool islower (_CharT __c, const locale& __loc)
    { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::lower, __c); }
  template<typename _CharT>
    inline bool isalpha (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::alpha, __c); }
  template<typename _CharT>
    inline bool isdigit (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::digit, __c); }
  template<typename _CharT>
    inline bool ispunct (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::punct, __c); }
  template<typename _CharT>
    inline bool isxdigit (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).
          is (ctype_base::xdigit, __c); }
  template<typename _CharT>
    inline bool isalnum (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::alnum, __c); }
  template<typename _CharT>
    inline bool isgraph (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).is (ctype_base::graph, __c); }

  template<typename _CharT>
    inline _CharT toupper (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).toupper (__c); }
  template<typename _CharT>
    inline _CharT tolower (_CharT __c, const locale& __loc)
      { return use_facet<ctype<_CharT> > (__loc).tolower (__c); }


} // namespace std

#endif	/* _CPP_BITS_LOCFACETS_H */

// Local Variables:
// mode:c++
// End:
