// Iostreams base classes -*- C++ -*-

// Copyright (C) 1997-1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

//
// ISO C++ 14882: 27.8  File-based streams
//

#ifndef _CPP_BITS_IOSBASE_H
#define _CPP_BITS_IOSBASE_H 1

namespace std {

  // The following definitions of bitmask types are enums, not ints,
  // as permitted (but not required) in the standard, in order to provide
  // better type safety in iostream calls.  A side effect is that
  // expressions involving them are no longer compile-time constants.
  enum _Ios_Fmtflags { _S_ios_fmtflags_end = 1<<16 };

  inline _Ios_Fmtflags 
  operator&(_Ios_Fmtflags __a, _Ios_Fmtflags __b)
  { return _Ios_Fmtflags(int (__a) & int (__b)); }

  inline _Ios_Fmtflags 
  operator|(_Ios_Fmtflags __a, _Ios_Fmtflags __b)
  { return _Ios_Fmtflags(int (__a) | int (__b)); }

  inline _Ios_Fmtflags 
  operator^(_Ios_Fmtflags __a, _Ios_Fmtflags __b)
  { return _Ios_Fmtflags(int (__a) ^ int (__b)); }

  inline _Ios_Fmtflags 
  operator|=(_Ios_Fmtflags& __a, _Ios_Fmtflags __b)
  { return __a = __a | __b; }

  inline _Ios_Fmtflags 
  operator&=(_Ios_Fmtflags& __a, _Ios_Fmtflags __b)
  { return __a = __a & __b; }

  inline _Ios_Fmtflags 
  operator^=(_Ios_Fmtflags& __a, _Ios_Fmtflags __b)
  { return __a = __a ^ __b; }

  inline _Ios_Fmtflags 
  operator~(_Ios_Fmtflags __a)
  { return _Ios_Fmtflags (~int (__a)); }


  enum _Ios_Openmode { _S_ios_openmode_end = 1<<16 };

  inline _Ios_Openmode 
  operator&(_Ios_Openmode __a, _Ios_Openmode __b)
  { return _Ios_Openmode(int (__a) & int (__b)); }

  inline _Ios_Openmode 
  operator|(_Ios_Openmode __a, _Ios_Openmode __b)
  { return _Ios_Openmode(int (__a) | int (__b)); }

  inline _Ios_Openmode 
  operator^(_Ios_Openmode __a, _Ios_Openmode __b)
  { return _Ios_Openmode(int (__a) ^ int (__b)); }

  inline _Ios_Openmode 
  operator|=(_Ios_Openmode& __a, _Ios_Openmode __b)
  { return __a = __a | __b; }

  inline _Ios_Openmode 
  operator&=(_Ios_Openmode& __a, _Ios_Openmode __b)
  { return __a = __a & __b; }

  inline _Ios_Openmode 
  operator^=(_Ios_Openmode& __a, _Ios_Openmode __b)
  { return __a = __a ^ __b; }

  inline _Ios_Openmode 
  operator~(_Ios_Openmode __a)
  { return _Ios_Openmode (~int (__a)); }


  enum _Ios_Iostate { _S_ios_iostate_end = 1<<16 };

  inline _Ios_Iostate 
  operator&(_Ios_Iostate __a, _Ios_Iostate __b)
  { return _Ios_Iostate(int (__a) & int (__b)); }

  inline _Ios_Iostate 
  operator|(_Ios_Iostate __a, _Ios_Iostate __b)
  { return _Ios_Iostate(int (__a) | int (__b)); }

  inline _Ios_Iostate 
  operator^(_Ios_Iostate __a, _Ios_Iostate __b)
  { return _Ios_Iostate(int (__a) ^ int (__b)); }

  inline _Ios_Iostate 
  operator|=(_Ios_Iostate& __a, _Ios_Iostate __b)
  { return __a = __a | __b; }

  inline _Ios_Iostate 
  operator&=(_Ios_Iostate& __a, _Ios_Iostate __b)
  { return __a = __a & __b; }

  inline _Ios_Iostate 
  operator^=(_Ios_Iostate& __a, _Ios_Iostate __b)
  { return __a = __a ^ __b; }

  inline _Ios_Iostate 
  operator~(_Ios_Iostate __a)
  { return _Ios_Iostate(~int (__a)); }

  enum _Ios_Seekdir { _S_ios_Seekdir_end = 1<<16 };

  // 27.4.2  Class ios_base
  class ios_base
  {
  public:

    // 27.4.2.1.1  Class ios_base::failure
    class failure : public exception
    {
    public:
#ifdef _GLIBCPP_RESOLVE_LIB_DEFECTS
      // Can't do exception(_msg) as defined in 27.4.2.1.1
      explicit 
      failure(const string& __str);

      virtual 
      ~failure() { };

      virtual const 
      char* what() const throw() { return _M_name; }
      
    private:
      enum { _M_bufsize = 256 };
      char _M_name[_M_bufsize];
#endif
    };

    // 27.4.2.1.2  Type ios_base::fmtflags
    typedef _Ios_Fmtflags fmtflags;
    // 27.4.2.1.2  Type fmtflags
    // We use the bit values from libio.h if possible.
    static const fmtflags boolalpha =   fmtflags(_IO_BOOLALPHA);
    static const fmtflags dec =         fmtflags(_IO_DEC);
    static const fmtflags fixed =       fmtflags(_IO_FIXED);
    static const fmtflags hex =         fmtflags(_IO_HEX);
    static const fmtflags internal =    fmtflags(_IO_INTERNAL);
    static const fmtflags left =        fmtflags(_IO_LEFT);
    static const fmtflags oct =         fmtflags(_IO_OCT);
    static const fmtflags right =       fmtflags(_IO_RIGHT);
    static const fmtflags scientific =  fmtflags(_IO_SCIENTIFIC);
    static const fmtflags showbase =    fmtflags(_IO_SHOWBASE);
    static const fmtflags showpoint =   fmtflags(_IO_SHOWPOINT);
    static const fmtflags showpos =     fmtflags(_IO_SHOWPOS);
    static const fmtflags skipws =      fmtflags(_IO_SKIPWS);
    static const fmtflags unitbuf =     fmtflags(_IO_UNITBUF);
    static const fmtflags uppercase =   fmtflags(_IO_UPPERCASE);
    static const fmtflags adjustfield = fmtflags(_IO_LEFT | _IO_RIGHT |
                                                 _IO_INTERNAL);
    static const fmtflags basefield =   fmtflags(_IO_DEC | _IO_OCT | _IO_HEX);
    static const fmtflags floatfield =  fmtflags(_IO_SCIENTIFIC | _IO_FIXED);

    // 27.4.2.1.3  Type ios_base::iostate
    typedef _Ios_Iostate iostate;
    static const iostate badbit =  iostate(_IO_BAD_SEEN);
    static const iostate eofbit =  iostate(_IO_EOF_SEEN);
    static const iostate failbit = iostate(_IO_ERR_SEEN);
    static const iostate goodbit = iostate(0);

    // 27.4.2.1.4  Type openmode
    typedef _Ios_Openmode openmode;
    static const openmode app =    openmode(_IOS_APPEND);
    static const openmode ate =    openmode(_IOS_ATEND);
    static const openmode binary = openmode(_IOS_BIN);
    static const openmode in =     openmode(_IOS_INPUT);
    static const openmode out =    openmode(_IOS_OUTPUT);
    static const openmode trunc =  openmode(_IOS_TRUNC);

    // 27.4.2.1.5  Type seekdir
    typedef _Ios_Seekdir seekdir;
    static const seekdir beg = seekdir(0);
    static const seekdir cur = seekdir(SEEK_CUR);
    static const seekdir end = seekdir(SEEK_END);

#ifdef _GLIBCPP_DEPRICATED
    typedef int io_state;
    typedef int open_mode;
    typedef int seek_dir;
#endif

    // Data members:
  private:
    struct _Callback_list;

    struct _Words 
    { 
      void* 	_M_pword; 
      long 	_M_iword; 
    };

    streamsize 		_M_precision;
    streamsize 		_M_width;
    fmtflags 		_M_flags;

    // 27.4.2.5  Members for iword/pword storage
    static const int 	_S_local_words = 8;
    _Words  		_M_word_array[_S_local_words];  // Guaranteed storage
    _Words  		_M_dummy;    // Only for failed iword/pword calls.
    _Words* 		_M_words;
    int     		_M_word_limit;
 
    // 27.4.2.6  Members for callbacks
    _Callback_list*  	_M_callbacks;

    // Members for locale and locale caching.
    locale 		_M_locale_ios;

    _Words& 
    _M_grow_words(int __index);

  protected:
    void 
    _M_dispose_callbacks();

    void 
    _M_init();

    void 
    _M_copy_base(ios_base& __rhs);

  public:
    // 27.4.2.1.6  Class ios_base::Init
    // Used to initialize standard streams. In theory, g++ could use
    // -finit-priority to order this stuff correctly without going
    // through these machinations.

    class Init 
    {
    public:
      Init();
      ~Init();
    private:
      static int 	_M_ios_base_init;
      filebuf* 		_M_cout;
      filebuf* 		_M_cin;
      filebuf* 		_M_cerr;
#ifdef _GLIBCPP_USE_WCHAR_T
      wfilebuf* 	_M_wcout;
      wfilebuf*        	_M_wcin;
      wfilebuf* 	_M_wcerr;
#endif
    };

    // Fmtflags state:
    inline fmtflags 
    flags() const { return _M_flags; }

    inline fmtflags 
    flags(fmtflags __fmtfl)
    { 
      fmtflags __old = _M_flags; 
      _M_flags = __fmtfl; 
      return __old; 
    }

    inline fmtflags 
    setf(fmtflags __fmtfl)
    { 
      fmtflags __old = _M_flags; 
      _M_flags |= __fmtfl; 
      return __old; 
    }

    inline fmtflags 
    setf(fmtflags __fmtfl, fmtflags __mask)
    {
      fmtflags __old = _M_flags;
      _M_flags = (_M_flags & ~__mask) | (__fmtfl & __mask);
      return __old;
    }

    inline void 
    unsetf(fmtflags __mask) { _M_flags &= ~__mask; }

    inline streamsize 
    precision() const { return _M_precision; }

    inline streamsize 
    precision(streamsize __prec)
    { 
      streamsize __old = _M_precision; 
      _M_precision = __prec; 
      return __old; 
    }

    inline streamsize 
    width() const { return _M_width; }

    inline streamsize 
    width(streamsize __wide)
    { 
      streamsize __old = _M_width; 
      _M_width = __wide; 
      return __old; 
    }

    // Locales:
    locale 
    imbue(const locale& __loc);

    inline locale 
    getloc() const { return _M_locale_ios; }

    // Storage:
    static int 
    xalloc() throw();

    inline long& 
    iword(int __ix)
    {
      _Words& __word = (__ix < _M_word_limit) 
			? _M_words[__ix] : _M_grow_words(__ix);
      return __word._M_iword;
    }

    inline void*& 
    pword(int __ix)
    {
      _Words& __word = (__ix < _M_word_limit) 
			? _M_words[__ix] : _M_grow_words(__ix);
      return __word._M_pword;
    }

    // Destructor
    ~ios_base();

    // Callbacks;
    enum event
    {
      erase_event,
      imbue_event,
      copyfmt_event
    };

    typedef void (*event_callback) (event, ios_base&, int __index);

    void 
    register_callback(event_callback __fn, int __index);

    static bool 
    sync_with_stdio(bool /*__sync*/ = true)
    { 
      // XXX Not done: 
      // Otherwise, called with __sync == false, allows the standard
      // streams to operate independently of the standard C streams.
      return true; 
    }

  protected:
    void 
    _M_call_callbacks(event __ev) throw();

    ios_base();

#ifdef _GLIBCPP_RESOLVE_LIB_DEFECTS
  private:
    ios_base(const ios_base&);

    ios_base& 
    operator=(const ios_base&);
#endif
  };
 
  // 27.4.5.1 fmtflags manipulators:
  inline ios_base& 
  boolalpha(ios_base& __str)
  {
    __str.setf(ios_base::boolalpha);
    return __str;
  }

  inline ios_base& 
  noboolalpha(ios_base& __str)
  {
    __str.unsetf(ios_base::boolalpha);
    return __str;
  }

  inline ios_base& 
  showbase(ios_base& __str)
  {
    __str.setf(ios_base::showbase);
    return __str;
  }

  inline ios_base& 
  noshowbase(ios_base& __str)
  {
    __str.unsetf(ios_base::showbase);
    return __str;
  }

  inline ios_base& 
  showpoint(ios_base& __str)
  {
    __str.setf(ios_base::showpoint);
    return __str;
  }

  inline ios_base& 
  noshowpoint(ios_base& __str)
  {
    __str.unsetf(ios_base::showpoint);
    return __str;
  }

  inline ios_base& 
  showpos(ios_base& __str)
  {
    __str.setf(ios_base::showpos);
    return __str;
  }

  inline ios_base& 
  noshowpos(ios_base& __str)
  {
    __str.unsetf(ios_base::showpos);
    return __str;
  }

  inline ios_base& 
  skipws(ios_base& __str)
  {
    __str.setf(ios_base::skipws);
    return __str;
  }
  
  inline ios_base& 
  noskipws(ios_base& __str)
  {
    __str.unsetf(ios_base::skipws);
    return __str;
  }

  inline ios_base& 
  uppercase(ios_base& __str)
  {
    __str.setf(ios_base::uppercase);
    return __str;
  }

  inline ios_base& 
  nouppercase(ios_base& __str)
  {
    __str.unsetf(ios_base::uppercase);
    return __str;
  }

  inline ios_base& 
  unitbuf(ios_base& __str)
  {
     __str.setf(ios_base::unitbuf);      
     return __str;
  }

  inline ios_base& 
  nounitbuf(ios_base& __str)
  {
     __str.unsetf(ios_base::unitbuf);
     return __str;    
  }

  // 27.4.5.2 adjustfield anipulators:
  inline ios_base& 
  internal(ios_base& __str)
  {
     __str.setf(ios_base::internal, ios_base::adjustfield);
     return __str;    
  }

  inline ios_base& 
  left(ios_base& __str)
  {
    __str.setf(ios_base::left, ios_base::adjustfield);
    return __str;
  }
  
  inline ios_base& 
  right(ios_base& __str)
  {
    __str.setf(ios_base::right, ios_base::adjustfield);
    return __str;
  }
  
  // 27.4.5.3 basefield anipulators:
  inline ios_base& 
  dec(ios_base& __str)
  {
    __str.setf(ios_base::dec, ios_base::basefield);
    return __str;
  }
  
  inline ios_base& 
  hex(ios_base& __str)
  {
    __str.setf(ios_base::hex, ios_base::basefield);
    return __str;
  }

  inline ios_base& 
  oct(ios_base& __str)
  {
    __str.setf(ios_base::oct, ios_base::basefield);
    return __str;
  }
  
  // 27.4.5.4 floatfield anipulators:
  inline ios_base& 
  fixed(ios_base& __str)
  {
    __str.setf(ios_base::fixed, ios_base::floatfield);
    return __str;
  }

  inline ios_base& 
  scientific(ios_base& __str)
  {
    __str.setf(ios_base::scientific, ios_base::floatfield);
    return __str;
  }

} // namespace std

#endif /* _CPP_BITS_IOSBASE_H */









