// This may look like C code, but it is really -*- C++ -*-

/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _Rational_h
#pragma once
#define _Rational_h 1

#include <Integer.h>
#include <math.h>

class RatTmp;

class Rational
{
  friend class     RatTmp;
protected:
  Integer          num;
  Integer          den;

  void             normalize();

public:
                   Rational();
                   Rational(double);
                   Rational(long n, long d = 1);
                   Rational(Integer& n);
                   Rational(Integer& n, Integer& d);
                   Rational(Rational&);

                   ~Rational();

  void             operator =  (Rational& y);

  friend int       operator == (Rational& x, Rational& y);
  friend int       operator != (Rational& x, Rational& y);
  friend int       operator <  (Rational& x, Rational& y);
  friend int       operator <= (Rational& x, Rational& y);
  friend int       operator >  (Rational& x, Rational& y);
  friend int       operator >= (Rational& x, Rational& y);

  RatTmp           operator +  (Rational& x);
  void             operator += (Rational& y);
  RatTmp           operator -  (Rational& y);
  void             operator -= (Rational& y);
  RatTmp           operator *  (Rational& y);
  void             operator *= (Rational& y);
  RatTmp           operator /  (Rational& x);
  void             operator /= (Rational& y);

  friend Rational& operator <? (Rational& x, Rational& y); // min
  friend Rational& operator >? (Rational& x, Rational& y); // max

  RatTmp           operator -  ();


// builtin Rational functions


  void             negate();                      // x = -x
  void             invert();                      // x = 1/x

  friend int       sign(Rational& x);             // -1, 0, or +1
  friend RatTmp    abs(Rational& x);              // absolute value
  friend RatTmp    sqr(Rational& x);              // square
  friend RatTmp    pow(Rational& x, long y);
  friend RatTmp    pow(Rational& x, Integer& y);
  IntTmp           numerator();
  IntTmp           denominator();

// coercion & conversion

  double           operator double();
  friend IntTmp    floor(Rational& x);
  friend IntTmp    ceil(Rational& x);
  friend IntTmp    trunc(Rational& x);
  friend IntTmp    round(Rational& x);

  friend istream&  operator >> (istream& s, Rational& y);
  friend ostream&  operator << (ostream& s, Rational& y);

// miscellany

  friend int       compare(Rational& x, Rational& y);
  void             error(char* msg);
  int              OK();

};

class RatTmp : public Rational
{
public:
                   RatTmp(Integer& n, Integer& d);
                   RatTmp(Rational&);
                   RatTmp(RatTmp&);

                   ~RatTmp();

  RatTmp           operator +  (Rational& x);
  RatTmp           operator -  (Rational& y);
  RatTmp           operator *  (Rational& y);
  RatTmp           operator /  (Rational& x);

  RatTmp           operator -  ();

  friend RatTmp    abs(RatTmp& x);
  friend RatTmp    sqr(RatTmp& x);
};


//#ifdef __OPTIMIZE__

inline Rational::Rational()  {}
inline Rational::~Rational() {}
inline RatTmp::~RatTmp() {}

inline Rational::Rational(Rational& y)
{
  num = y.num; den = y.den;
}

inline RatTmp::RatTmp(Rational& y)
{
  num = IntTmp(y.num); den = IntTmp(y.den);
}

inline RatTmp::RatTmp(RatTmp& y)
{
  num = IntTmp(y.num); den = IntTmp(y.den);
}

inline Rational::Rational(Integer& n)
{
  num = n; den = 1;
}

inline Rational::Rational(Integer& n, Integer& d)
{
  num = n; den = d; normalize();
}

inline RatTmp::RatTmp(Integer& n, Integer& d)
{
  num = IntTmp(n); den = IntTmp(d); normalize();
}

inline Rational::Rational(long n, long d = 1)
{
  num = n; den = d; normalize();
}

inline  void Rational::operator =  (Rational& y)
{
  num = y.num;  den = y.den;
}

inline int operator == (Rational& x, Rational& y)
{
  return compare(x.num, y.num) == 0 && compare(x.den, y.den) == 0;
}

inline int operator != (Rational& x, Rational& y)
{
  return compare(x.num, y.num) != 0 || compare(x.den, y.den) != 0;
}

inline int operator <  (Rational& x, Rational& y)
{
  return compare(x, y) <  0; 
}

inline int operator <= (Rational& x, Rational& y)
{
  return compare(x, y) <= 0; 
}

inline int operator >  (Rational& x, Rational& y)
{
  return compare(x, y) >  0; 
}

inline int operator >= (Rational& x, Rational& y)
{
  return compare(x, y) >= 0; 
}

inline int sign(Rational& x)
{
  return sign(x.num);
}

inline void Rational::negate()
{
  num.negate();
}

inline Rational& operator <? (Rational& x, Rational& y)
{
  if (compare(x, y) <= 0) return x; else return y;
}

inline Rational& operator >? (Rational& x, Rational& y)
{
  if (compare(x, y) >= 0) return x; else return y;
}


//#endif

#endif
