// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _Complex_h
#pragma once
#define _Complex_h 1


#include <stream.h>
#include <math.h>

class Complex
{
protected:
  double           re;
  double           im;

public:
                   Complex();
                   Complex(Complex& c);
                   Complex(double r, double i = 0.0);

                   ~Complex();

  Complex&         operator =  (Complex& y);

  friend int       operator == (Complex& x, Complex& y);
  friend int       operator == (Complex& x, double y);

  friend int       operator != (Complex& x, Complex& y);
  friend int       operator != (Complex& x, double y);

  Complex          operator -  ();

  friend Complex   operator +  (Complex& x, Complex& y);
  friend Complex   operator +  (Complex& x, double y);
  friend Complex   operator +  (double   x, Complex& y);
  friend Complex   operator -  (Complex& x, Complex& y);
  friend Complex   operator -  (Complex& x, double y);
  friend Complex   operator -  (double   x, Complex& y);
  friend Complex   operator *  (Complex& x, Complex& y);
  friend Complex   operator *  (Complex& x, double y);
  friend Complex   operator *  (double   x, Complex& y);
  friend Complex   operator /  (Complex& x, Complex& y);
  friend Complex   operator /  (Complex& x, double y);
  friend Complex   operator /  (double   x, Complex& y);

  Complex&         operator += (Complex& y); 
  Complex&         operator += (double y); 
  Complex&         operator -= (Complex& y); 
  Complex&         operator -= (double y); 
  Complex&         operator *= (Complex& y); 
  Complex&         operator *= (double y); 
  Complex&         operator /= (Complex& y); 
  Complex&         operator /= (double y); 

  friend double    real(Complex& x);
  friend double    imag(Complex& x);
  friend double    abs(Complex& x);
  friend double    norm(Complex& x);
  friend double    arg(Complex& x);

  friend Complex   polar(double r, double t = 0.0);
  friend Complex   conj(Complex& x);

  friend Complex   cos(Complex& x);
  friend Complex   sin(Complex& x);

  friend Complex   cosh(Complex& x);
  friend Complex   sinh(Complex& x);

  friend Complex   exp(Complex& x);
  friend Complex   log(Complex& x);

  friend Complex   pow(Complex& x, long p);
  friend Complex   pow(Complex& x, Complex& p);
  friend Complex   sqrt(Complex& x);
   
  friend istream&  operator >> (istream& s, Complex& x);
  friend ostream&  operator << (ostream& s, Complex& x);

  void             error(char* msg);
};


// error handlers

extern  void default_Complex_error_handler(char*);
extern  one_arg_error_handler_t Complex_error_handler;

extern  one_arg_error_handler_t 
        set_Complex_error_handler(one_arg_error_handler_t f);


//#ifdef __OPTIMIZE__

inline Complex:: Complex() {}
inline Complex::~Complex() {}

inline Complex::Complex(double r, double i = 0.0)
{
  re = r; im = i;
}

inline Complex::Complex(Complex& x)
{
  re = x.re; im = x.im;
}

inline Complex& Complex::operator = (Complex& x)
{
  re = x.re; im = x.im; return *this;
}

inline int operator == (Complex& x, Complex& y)
{
  return x.re == y.re && x.im == y.im;
}

inline int operator == (Complex& x, double y)
{
  return x.im == 0.0 && x.re == y;
}

inline int operator != (Complex& x, Complex& y)
{
  return x.re != y.re || x.im != y.im;
}

inline int operator != (Complex& x, double y)
{
  return x.im != 0.0 || x.re != y;
}

inline Complex Complex::operator - ()
{
  return Complex(-re, -im);
}

inline Complex conj(Complex& x)
{
  return Complex(x.re, -x.im);
}

inline Complex operator + (Complex& x, Complex& y)
{
  return Complex(x.re + y.re, x.im + y.im);
}

inline Complex operator + (Complex& x, double y)
{
  return Complex(x.re + y, x.im);
}

inline Complex operator + (double x, Complex& y)
{
  return Complex(x + y.re, y.im);
}

inline Complex operator - (Complex& x, Complex& y)
{
  return Complex(x.re - y.re, x.im - y.im);
}

inline Complex operator - (Complex& x, double y)
{
  return Complex(x.re - y, x.im);
}

inline Complex operator - (double x, Complex& y)
{
  return Complex(x - y.re, -y.im);
}

inline Complex operator * (Complex& x, Complex& y)
{
  return Complex(x.re * y.re - x.im * y.im, x.re * y.im + x.im * y.re);
}

inline Complex operator * (Complex& x, double y)
{
  return Complex(x.re * y, x.im * y);
}

inline Complex operator * (double x, Complex& y)
{
  return Complex(x * y.re, x * y.im);
}

inline Complex& Complex::operator += (Complex& y)
{
  re += y.re;  im += y.im;  return *this;
}

inline Complex& Complex::operator += (double y)
{
  re += y;  return *this;
}

inline Complex& Complex::operator -= (Complex& y)
{
  re -= y.re; im -= y.im; return *this;
}

inline Complex& Complex::operator -= (double y)
{
  re -= y; return *this;
}

inline Complex& Complex::operator *= (Complex& y)
{
  double r = re * y.re - im * y.im;
  im = re * y.im + im * y.re;
  re = r;
  return *this;
}

inline Complex& Complex::operator *= (double y)
{
  re *=  y; im *=  y; return *this;
}

inline double real(Complex& x)
{
  return x.re;
}

inline double imag(Complex& x)
{
  return x.im;
}

inline double abs(Complex& x)
{
  return hypot(x.re, x.im);
}

inline double norm(Complex& x)
{
  return (x.re * x.re + x.im * x.im);
}

inline double arg(Complex& x)
{
  return atan2(x.im, x.re);
}

//#endif

#endif
