// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _stream_h
#define _stream_h 1

#include <File.h>

class whitespace                // a class used only to input and
{                               // discard white space characters
  char filler;                     
};

class ostream: File
{
public:
           File::open;      File::close;
           File::remove;    File::filedesc;  File::is_open;
           File::raw;       File::put;       
           File::iocount;   File::error;     File::name;
           File::setname;   File::rdstate;   File::flush;
           File::eof;       File::fail;      File::bad;
           File::good;      File::clear;     File::failif;
           File::setbuf;    File::writable;  File::readable;
           

           ostream();
           ostream(const char* filename, io_mode m, access_mode a);
           ostream(const char* filename, const char* m);
           ostream(int filedesc, io_mode m = io_writeonly);
           ostream(FILE* fileptr);

           ~ostream();

  void*    operator void*();

  ostream& form(const char* fmt, ...);           

  ostream& operator << (char   c);
  ostream& operator << (short  n);
  ostream& operator << (unsigned short n);
  ostream& operator << (int    n);
  ostream& operator << (unsigned int n);
  ostream& operator << (long   n);
  ostream& operator << (unsigned long n);
  ostream& operator << (float  n);
  ostream& operator << (double n);
  ostream& operator << (const char* s);
};


class istream: File
{
  ostream* tied_to;        // unused, but maintained for AT&T compatibility
  void     _flush();

public:
           File::open;      File::close;     File::get;
           File::remove;    File::filedesc;  File::is_open;
           File::raw;       File::unget;     File::getline;
           File::iocount;   File::error;     File::name;
           File::setname;   File::rdstate;   File::putback;
           File::eof;       File::fail;      File::bad;
           File::good;      File::clear;     File::failif;
           File::setbuf;    File::writable;  File::readable;


           istream();
           istream(const char* filename, io_mode m, access_mode a);
           istream(const char* filename, const char* m);
           istream(int filedesc, io_mode m = io_readonly);
           istream(FILE* fileptr);

           ~istream();

  void*    operator void*();

  ostream* tie(ostream* s);

  istream& scan(const char* fmt, ...);

  istream& operator >> (char&   c);
  istream& operator >> (short&  n);
  istream& operator >> (unsigned short& n);
  istream& operator >> (int&    n);
  istream& operator >> (unsigned int& n);
  istream& operator >> (long&   n);
  istream& operator >> (unsigned long& n);
  istream& operator >> (float&  n);
  istream& operator >> (double& n);
  istream& operator >> (char*   s);
  istream& operator >> (whitespace& w);
};

void eatwhite(istream& s);


// pre-declared streams

extern istream  cin;             // stdin
extern ostream  cout;            // stdout
extern ostream  cerr;            // stderr

extern whitespace WS;            // for convenience

//#ifdef __OPTIMIZE__


inline void* ostream::operator void*()
{ 
  return (state & (_bad|_fail))? 0 : this ; 
}

inline ostream& ostream::operator<<(char   c)
{ 
  put(c);
  return *this;
}

inline ostream& ostream::operator<<(short  n)
{ 
  return form("%d",(int)n);
}

inline ostream& ostream::operator<<(unsigned short n)
{ 
  return form("%u",(unsigned)n);
}

inline ostream& ostream::operator<<(int    n)
{ 
  return form("%d",n);
}

inline ostream& ostream::operator<<(unsigned int n)
{ 
  return form("%u",n);
}

inline ostream& ostream::operator<<(long   n)
{ 
  return form("%ld",n);
}

inline ostream& ostream::operator<<(unsigned long n)
{ 
  return form("%lu",n);
}

inline ostream& ostream::operator<<(float  n)
{ 
  return form("%g",(double)n);
}

inline ostream& ostream::operator<<(double n)
{ 
  return form("%g",n);
}

inline ostream& ostream::operator<<(const char* s)
{ 
  put(s);
  return *this;
}

inline void*    istream::operator void*()
{ 
  return (state & (_bad|_fail))? 0 : this ; 
}

inline istream& istream::operator>>(char&   c)
{ 
  _flush();
  get(c);
  return *this;
}

inline istream& istream::operator>>(short&  n)
{ 
  return scan("%hd", &n); 
}

inline istream& istream::operator>>(unsigned short& n)
{ 
  return scan("%hd", &n); 
}

inline istream& istream::operator>>(int&    n)
{ 
  return scan("%d",  &n); 
}

inline istream& istream::operator>>(unsigned int& n)
{ 
  return scan("%d",  &n); 
}

inline istream& istream::operator>>(long&   n)
{ 
  return scan("%ld", &n); 
}

inline istream& istream::operator>>(unsigned long& n)
{ 
  return scan("%ld", &n); 
}

inline istream& istream::operator>>(float&  n)
{ 
  return scan("%f",  &n); 
}

inline istream& istream::operator>>(double& n)
{ 
  return scan("%lf", &n); 
}

inline istream& istream::operator>>(char*   s)
{ 
  return scan("%s",   s); 
}

//#endif

#endif
