// This may look like C code, but it is really -*- C++ -*-

/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _Integer_h
#define _Integer_h 1

#include <stream.h>


struct _Irep                    // internal Integer representations
{
  unsigned short  sgn;          // 1 means >= 0; 0 means < 0 
                                //   (space wasted for portable alignment)
  unsigned short  len;          // current length
  unsigned short  sz;           // allocated space
  short           ref;          // reference count
  unsigned short  s[1];         // represented as ushort array starting here
};

class Rational;

class Integer
{
  friend class    Rational;

  _Irep*          rep;

  void            copy(const unsigned short*, int, int);
  void            checklength();
  friend void     mk_tmpInteger(long y);

public:
                  Integer();
                  Integer(long);
                  Integer(Integer&);

                  ~Integer();

  Integer&        operator =  (Integer& y);
  Integer&        operator =  (long y);

  friend int      operator == (Integer& x, Integer& y);
  friend int      operator == (Integer& x, long y);

  friend int      operator != (Integer& x, Integer& y);
  friend int      operator != (Integer& x, long y);

  friend int      operator <  (Integer& x, Integer& y);
  friend int      operator <  (Integer& x, long y);

  friend int      operator <= (Integer& x, Integer& y);
  friend int      operator <= (Integer& x, long y);

  friend int      operator >  (Integer& x, Integer& y);
  friend int      operator >  (Integer& x, long y);

  friend int      operator >= (Integer& x, Integer& y);
  friend int      operator >= (Integer& x, long y);

  friend int      ucompare(Integer& x, Integer& y); // unsigned comparison
  friend int      ucompare(Integer& x, long y);

  Integer         operator -  ();
  Integer&        negate();                     // negate in-place

  Integer         operator ~  ();
  void            operator ++ ();
  void            operator -- ();

  friend Integer  operator +  (Integer& x, Integer& y);
  friend Integer  operator +  (Integer& x, long y);
  friend Integer  operator +  (long x, Integer& y);
  friend Integer  operator -  (Integer& x, Integer& y);
  friend Integer  operator -  (Integer& x, long y);
  friend Integer  operator -  (long x, Integer& y);
  friend Integer  operator *  (Integer& x, Integer& y);
  friend Integer  operator *  (Integer& x, long y);
  friend Integer  operator /  (Integer& x, Integer& y);
  friend Integer  operator /  (Integer& x, long y);
  friend Integer  operator %  (Integer& x, Integer& y);
  friend Integer  operator %  (Integer& x, long y);
  friend Integer  operator << (Integer& x, Integer& y);
  friend Integer  operator << (Integer& x, long y);
  friend Integer  operator >> (Integer& x, Integer& y);
  friend Integer  operator >> (Integer& x, long y);
  friend Integer  operator &  (Integer& x, Integer& y);
  friend Integer  operator &  (Integer& x, long y);
  friend Integer  operator |  (Integer& x, Integer& y);
  friend Integer  operator |  (Integer& x, long y);
  friend Integer  operator ^  (Integer& x, Integer& y);
  friend Integer  operator ^  (Integer& x, long y);
  friend Integer& operator <? (Integer& x, Integer& y); // min
  friend Integer& operator >? (Integer& x, Integer& y); // max

  Integer&        operator += (Integer& y);
  Integer&        operator += (long y);
  Integer&        operator -= (Integer& y);
  Integer&        operator -= (long y);
  Integer&        operator *= (Integer& y);
  Integer&        operator *= (long y);
  Integer&        operator /= (Integer& y);
  Integer&        operator /= (long y);
  Integer&        operator %= (Integer& y);
  Integer&        operator %= (long y);
  Integer&        operator <<=(Integer& y);
  Integer&        operator <<=(long y);
  Integer&        operator >>=(Integer& y);
  Integer&        operator >>=(long y);
  Integer&        operator &= (Integer& y);
  Integer&        operator &= (long y);
  Integer&        operator |= (Integer& y);
  Integer&        operator |= (long y);
  Integer&        operator ^= (Integer& y);
  Integer&        operator ^= (long y);

// builtin Integer functions

  friend Integer  abs(Integer& x);              // absolute value
  friend long     lg (Integer& x);              // floor log base 2 of abs(x)
  friend Integer  sqr(Integer& x);              // square
  friend Integer  sqrt(Integer& x);             // floor of square root
  friend Integer  gcd(Integer& x, Integer& y);  // greatest common divisor
  friend Integer  lcm(Integer& x, Integer& y);  // least common multiple
  friend int      even(Integer& y);             // true if x is even
  friend int      odd(Integer& y);              // true if x is odd
  friend int      sign(Integer& y);             // returns -1, 0, +1
  friend Integer  pow(Integer& x, Integer& y);  // x to the y power
  friend Integer  pow(Integer& x, long y);
  friend Integer  Ipow(long x, long y);         // x to the y as Integer 
  friend void     setbit(Integer& x, long b);   // set b'th bit
  friend void     clearbit(Integer& x, long b); // clear b'th bit
  friend int      testbit(Integer& x, long b);  // return b'th bit

// coercion & conversion

  int             fits_in_long();
  int             fits_in_double();
  long            operator long();
  double          operator double();

  friend char*    Itoa(Integer& x, int base = 10, int width = 0);
  friend Integer  atoI(const char* s);
  
  friend istream& operator >> (istream& s, Integer& y);
  friend ostream& operator << (ostream& s, Integer& y);

// miscellany

  void            make_unique();
  void            setlength(int, int=0);
  void            error(char* msg);

// non-operator versions

  friend void     add(Integer& x, int negx, Integer& y, int negy, Integer& r);
  friend void     add(Integer& x, int negx, long y, Integer& r);
  friend void     multiply(Integer& x, Integer& y, Integer& r);
  friend void     multiply(Integer& x, long y, Integer& r);
  friend void     divide(Integer& x, Integer& y, Integer& q, Integer& r,
                         char code=0);
  friend void     divide(Integer& x, long y, Integer& q, Integer& r,
                         char code=0);
  friend void     divide(Integer& x, unsigned short y, Integer& q, int& r,
                         char code=0);
  friend void     bitop(Integer& x, Integer& y, Integer& r, char op);
  friend void     bitop(Integer& x, long y, Integer& r, char op);
  friend void     lshift(Integer& x, long y, Integer& r);
  friend int      compare(Integer& x, Integer& y);
  friend int      compare(Integer& x, long y);
  void            copy(Integer& y);
  void            copy(long y);
};


// Integer functions that need not be declared friends

char*             dec(Integer& x, int width = 0);
char*             oct(Integer& x, int width = 0);
char*             hex(Integer& x, int width = 0);

// error handlers

extern  void default_Integer_error_handler(char*);
extern  one_arg_error_handler_t Integer_error_handler;

extern  one_arg_error_handler_t 
        set_Integer_error_handler(one_arg_error_handler_t f);

//#ifdef __OPTIMIZE__


extern  _Irep   _nil_Irep;

inline Integer::Integer()
{
  rep = &_nil_Irep;
}

inline Integer::Integer(long y)
{
  rep = &_nil_Irep;  copy(y);
}

inline Integer::Integer(Integer&  y)
{
  rep = y.rep; if (rep->ref > 0) rep->ref++;
}

inline Integer::~Integer()
{
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
}

inline Integer&  Integer::operator = (Integer&  y)
{
  y.rep->ref++;
  if (rep->ref > 0 && --rep->ref == 0) delete rep;
  rep = y.rep;
  return *this;
}

inline Integer&  Integer::operator = (long y)
{
  copy(y);  return *this;
}

inline int operator == (Integer&  x, Integer&  y)
{
  return compare(x, y) == 0; 
}

inline int operator == (Integer&  x, long y)
{
  return compare(x, y) == 0; 
}

inline int operator != (Integer&  x, Integer&  y)
{
  return compare(x, y) != 0; 
}

inline int operator != (Integer&  x, long y)
{
  return compare(x, y) != 0; 
}

inline int operator <  (Integer&  x, Integer&  y)
{
  return compare(x, y) <  0; 
}

inline int operator <  (Integer&  x, long y)
{
  return compare(x, y) <  0; 
}

inline int operator <= (Integer&  x, Integer&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator <= (Integer&  x, long y)
{
  return compare(x, y) <= 0; 
}

inline int operator >  (Integer&  x, Integer&  y)
{
  return compare(x, y) >  0; 
}

inline int operator >  (Integer&  x, long y)
{
  return compare(x, y) >  0; 
}

inline int operator >= (Integer&  x, Integer&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator >= (Integer&  x, long y)
{
  return compare(x, y) >= 0; 
}

inline int sign(Integer& x)
{
  return (x.rep->len == 0) ? 0 : ( (x.rep->sgn == 1) ? 1 : -1 );
}

inline Integer  operator +  (Integer&  x, Integer& y)
{
  Integer r; add(x, 0, y, 0, r); return r;
}

inline Integer  operator -  (Integer&  x, Integer& y)
{
  Integer r; add(x, 0, y, 1, r); return r;
}

inline Integer&  Integer::operator += (Integer& y)
{
  add(*this, 0, y, 0, *this); return *this;
}

inline Integer&  Integer::operator -= (Integer& y)
{
  add(*this, 0, y, 1, *this); return *this;
}

inline Integer  operator *  (Integer&  x, Integer& y)
{
  Integer r; multiply(x, y, r); return r;
}

inline Integer&  Integer::operator *= (Integer& y)
{
  multiply(*this, y, *this); return *this;
}

inline Integer operator / (Integer& x, Integer& y)
{
  Integer q;  divide(x, y, q, q, 'q');  return q;
}

inline Integer& Integer::operator /= (Integer& y)
{
  divide(*this, y, *this, *this, 'q');  return *this;
}

inline Integer operator % (Integer& x, Integer& y)
{
  Integer r;  divide(x, y, r, r, 'r'); return r;
}

inline Integer& Integer::operator %= (Integer& y)
{
  divide(*this, y, *this, *this, 'r');   return *this;
}

inline void Integer::operator ++ ()
{
  add(*this, 0, 1, *this);
}

inline void Integer::operator -- ()
{
  add(*this, 0, -1, *this);
}

inline Integer  operator << (Integer&  x, Integer&  y)
{     // correct since max Integer bits < max long
  Integer r; lshift(x, long(y), r); return r;
}

inline Integer  operator >> (Integer&  x, Integer&  y)
{  
  Integer r; lshift(x, -(long(y)), r); return r;
}

inline Integer& Integer::operator <<= (Integer&  y)
{
  lshift(*this, long(y), *this); return *this;
}

inline Integer& Integer::operator >>= (Integer&  y)
{
  lshift(*this, -(long(y)), *this); return *this;
}

inline Integer  operator &  (Integer&  x, Integer& y)
{
  Integer r; bitop(x, y, r, '&'); return r;
}

inline Integer  operator |  (Integer&  x, Integer& y)
{
  Integer r; bitop(x, y, r, '|'); return r;
}

inline Integer  operator ^  (Integer&  x, Integer& y)
{
  Integer r; bitop(x, y, r, '^'); return r;
}

inline Integer&  Integer::operator &= (Integer& y)
{
  bitop(*this, y, *this, '&'); return *this;
}

inline Integer&  Integer::operator |= (Integer& y)
{
  bitop(*this, y, *this, '|'); return *this;
}

inline Integer&  Integer::operator ^= (Integer& y)
{
  bitop(*this, y, *this, '^'); return *this;
}

inline Integer  operator +  (Integer&  x, long y)
{
  Integer r; add(x, 0, y, r); return r;
}

inline Integer  operator -  (Integer&  x, long y)
{
  Integer r; add(x, 0, -y, r); return r;
}

inline Integer  operator +  (long  x, Integer& y)
{
  Integer r; add(y, 0, x, r); return r;
}

inline Integer  operator -  (long  x, Integer& y)
{
  Integer r; add(y, 1, x, r); return r;
}

inline Integer&  Integer::operator += (long y)
{
  add(*this, 0, y, *this); return *this;
}

inline Integer&  Integer::operator -= (long y)
{
  add(*this, 0, -y, *this); return *this;
}

inline Integer  operator *  (Integer&  x, long y)
{
  Integer r; multiply(x, y, r); return r;
}

inline Integer& Integer::operator *= (long y)
{
  multiply(*this, y, *this); return *this;
}

inline Integer operator / (Integer& x, long y)
{
  Integer q;  divide(x, y, q, q, 'q');  return q;
}

inline Integer& Integer::operator /= (long y)
{
  divide(*this, y, *this, *this, 'q');  return *this;
}

inline Integer operator % (Integer& x, long y)
{
  Integer r;  divide(x, y, r, r, 'r'); return r;
}

inline Integer& Integer::operator %= (long y)
{
  divide(*this, y, *this, *this, 'r');   return *this;
}

inline Integer  operator << (Integer&  x, long y)
{
  Integer r; lshift(x, y, r); return r;
}

inline Integer  operator >> (Integer&  x, long y)
{  
  Integer r; lshift(x, -y, r); return r;
}

inline Integer& Integer::operator <<= (long  y)
{
  lshift(*this, y, *this); return *this;
}

inline Integer&  Integer::operator >>= (long y)
{
  lshift(*this, -y, *this); return *this;
}

inline Integer  operator &  (Integer&  x, long y)
{
  Integer r; bitop(x, y, r, '&'); return r;
}

inline Integer  operator |  (Integer&  x, long y)
{
  Integer r; bitop(x, y, r, '|'); return r;
}

inline Integer  operator ^  (Integer&  x, long y)
{
  Integer r; bitop(x, y, r, '^'); return r;
}

inline Integer&  Integer::operator &= (long y)
{
  bitop(*this, y, *this, '&'); return *this;
}

inline Integer&  Integer::operator |= (long y)
{
  bitop(*this, y, *this, '|'); return *this;
}

inline Integer&  Integer::operator ^= (long y)
{
  bitop(*this, y, *this, '^'); return *this;
}

inline Integer& operator <? (Integer& x, Integer& y)
{
  if (compare(x, y) <= 0) return x; else return y;
}

inline Integer& operator >? (Integer& x, Integer& y)
{
  if (compare(x, y) >= 0) return x; else return y;
}

inline Integer  pow(Integer&  x, Integer&  y)
{
  return pow(x, long(y)); // correct since max bits < max long
}

inline Integer  Ipow(long  x, long  y)
{
  return pow(Integer(x), y);
}

inline Integer sqr(Integer& x)
{
  Integer r; multiply(x, x, r); return r;
}


inline Integer abs(Integer&  y)
{
  Integer r;  r.copy(y.rep->s, y.rep->len, 1);   return r;
}

inline int even(Integer& y)
{
  return y.rep->len == 0 || !(y.rep->s[0] & 1);
}

inline int odd(Integer& y)
{
  return y.rep->len > 0 && (y.rep->s[0] & 1);
}

inline ostream& operator << (ostream& s, Integer& y)
{
  return s << Itoa(y);
}

inline void Integer::make_unique()
{
  if (rep->ref != 1) setlength(rep->len);
}

inline Integer& Integer::negate()
{
  make_unique(); if (rep->len != 0) rep->sgn = !rep->sgn; return *this;
}

inline void Integer::copy(Integer& y)
{
  copy(y.rep->s, y.rep->len, y.rep->sgn);
}

//#endif

#endif
