/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#include <Rational.h>
#include <std.h>
#include <math.h>
#include "libconfig.h"


// error handling

void default_Rational_error_handler(char* msg)
{
  cerr << "Fatal Rational error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t Rational_error_handler= default_Rational_error_handler;

one_arg_error_handler_t set_Rational_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = Rational_error_handler;
  Rational_error_handler = f;
  return old;
}

void Rational::error(char* msg)
{
  (*Rational_error_handler)(msg);
}

static Integer _Int_One(1);

void Rational::normalize()
{
  int s = sign(den);
  if (s == 0)
    error("Zero denominator.");
  else if (s < 0)
  {
    den.negate();
    num.negate();
  }

  Integer g = gcd(num, den);
  if (ucompare(g, _Int_One) != 0)
  {
    num /= g;
    den /= g;
  }
}

Rational operator + (Rational& x, Rational& y)
{
  return Rational(x.num * y.den + y.num * x.den, x.den * y.den);
}

Rational operator - (Rational& x, Rational& y)
{
  return Rational(x.num * y.den - y.num * x.den, x.den * y.den);
}

Rational operator * (Rational& x, Rational& y)
{
  return Rational(x.num * y.num, x.den * y.den);
}

Rational operator / (Rational& x, Rational& y)
{
  return Rational(x.den * y.den, x.num * y.num);
}

Rational Rational::operator - ()
{
  return Rational(-num, den);
}

Rational abs(Rational& x)
{
  return Rational(abs(x.num), x.den);
}

Rational& Rational::operator +=(Rational& y)
{
  num = num * y.den + y.num * den;
  den *= y.den;
  normalize();
  return *this;
}

Rational& Rational::operator -=(Rational& y)
{
  num = num * y.den - y.num * den;
  den *= y.den;
  normalize();
  return *this;
}

Rational& Rational::operator *=(Rational& y)
{
  num *= y.num;
  den *= y.den;
  normalize();
  return *this;
}

Rational& Rational::operator /=(Rational& y)
{
  Integer n;
  multiply(num, y.den, n);
  den *= y.num;
  num = n;
  normalize();
  return *this;
}

void Rational::invert()
{
  Integer tmp = num;  
  num = den;  
  den = tmp;  
  int s = sign(den);
  if (s == 0)
    error("Zero denominator.");
  else if (s < 0)
  {
    den.negate();
    num.negate();
  }
}

Rational sqr(Rational& x)
{
  Rational r;
  multiply(x.num, x.num, r.num);
  multiply(x.den, x.den, r.den);
  return r;
}

int compare(Rational& x, Rational& y)
{
  int xsgn = sign(x.num);
  int ysgn = sign(y.num);
  int d = xsgn - ysgn;
  if (d == 0 && xsgn != 0)
    d = compare(x.num * y.den, x.den * y.num);
  return d;
}

Rational::Rational(double x)
{
  num = 0;
  den = 1;
  if (x != 0.0)
  {
    int neg = x < 0;
    if (neg)
      x = -x;

    const long shift = 15;       // a safe shift per step
    const long width = 32768.0;  // = 2^shift
    const int maxiter = 20;      // ought not be necessary, but just in case,
                                 // max 300 bits of precision
    int expt;
    double mantissa = frexp(x, &expt);
    long exponent = expt;
    double intpart;
    int k = 0;
    while (mantissa != 0.0 && k++ < maxiter)
    {
      mantissa *= width;
      mantissa = modf(mantissa, &intpart);
      num <<= shift;
      num += (long)intpart;
      exponent -= shift;
    }
    if (exponent > 0)
      num <<= exponent;
    else if (exponent < 0)
      den <<= -exponent;
    if (neg)
      num.negate();
  }
  normalize();
}


double Rational::operator double()
{
  Integer q, r;
  divide(num, den, q, r);
  double d1 = double(q);
 
  if (d1 == HUGE || d1 == -HUGE || sign(r) == 0)
    return d1;
  else      // use as much precision as available for fractional part
  {
    double  d2 = 0.0;
    double  d3 = 0.0; 
    double bound = HUGE / 2.0;
    int cont = 1;
    for (int i = den.rep->len - 1; i >= 0 && cont; --i)
    {
      unsigned short a = I_RADIX >> 1;
      while (a != 0)
      {
        if (d2 >= bound)
        {
          cont = 0;
          break;
        }
        d2 *= 2.0;
        if (den.rep->s[i] & a)
          d2 += 1.0;

        if (i < r.rep->len)
        {
          d3 *= 2.0;
          if (r.rep->s[i] & a)
            d3 += 1.0;
        }
        a >>= 1;
      }
    }

    if (sign(r) < 0)
      d3 = -d3;
    return d1 + d3 / d2;
  }
}

Integer floor(Rational& x)
{
  Integer q, r;
  divide(x.num, x.den, q, r);
  if (sign(x.num) < 0 && sign(r) != 0)
    q--;
  return q;
}

Integer ceil(Rational& x)
{
  Integer q, r;
  divide(x.num, x.den, q, r);
  if (sign(x.num) >= 0 && sign(r) != 0)
    q++;
  return q;
}

Integer round(Rational& x)
{
  Integer q, r;
  divide(x.num, x.den, q, r);
  r <<= 1;
  if (ucompare(r, x.den) >= 0)
  {
    if (sign(x.num) >= 0)
      q++;
    else
      q--;
  }
  return q;
}


// power: no need to normalize since num & den already relatively prime

Rational pow(Rational& x, long y)
{
  Rational r;
  if (y >= 0)
  {
    r.num = pow(x.num, y);
    r.den = pow(x.den, y);
  }
  else
  {
    y = -y;
    r.den = pow(x.num, y);
    r.num = pow(x.den, y);
    if (sign(r.den) < 0)
    {
      r.num.negate();
      r.den.negate();
    }
  }
  return r;
}


ostream& operator << (ostream& s, Rational& y)
{
  if (y.den == 1)
    return s << Itoa(y.num);
  else
    return s << Itoa(y.num) << "/" << Itoa(y.den);
}

istream& operator >> (istream& s, Rational& y)
{
  s >> y.num;
  char ch;
  s >> ch;
  if (ch == '/')
    s >> y.den;
  else
  {
    s.unget(ch);
    y.den = 1;
  }
  return s;
}
