/*
 * Copyright © 2016 Red Hat
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/* Lowers the additional tex_src_plane src, generated by nir_lower_tex
 * for planar YUV textures, into separate samplers, matching the logic
 * that mesa/st uses to insert additional sampler view/state (since both
 * sides need to agree).
 *
 * This should run after nir_lower_samplers.
 */

#include "compiler/nir/nir.h"
#include "st_nir.h"

typedef struct {
   unsigned lower_2plane;
   unsigned lower_3plane;

   /* Maps a primary sampler (used for Y) to the U or UV sampler.  In
    * case of 3-plane YUV format, the V plane is next sampler after U.
    */
   unsigned char sampler_map[PIPE_MAX_SAMPLERS][2];
} lower_tex_src_state;

static void
assign_extra_samplers(lower_tex_src_state *state, unsigned free_slots)
{
   unsigned mask = state->lower_2plane | state->lower_3plane;

   while (mask) {
      unsigned extra, y_samp = u_bit_scan(&mask);

      extra = u_bit_scan(&free_slots);
      state->sampler_map[y_samp][0] = extra;

      if (state->lower_3plane & (1 << y_samp)) {
         extra = u_bit_scan(&free_slots);
         state->sampler_map[y_samp][1] = extra;
      }
   }
}

static void
lower_tex_src_plane_block(lower_tex_src_state *state, nir_block *block)
{
   nir_foreach_instr(instr, block) {
      if (instr->type != nir_instr_type_tex)
         continue;

      nir_tex_instr *tex = nir_instr_as_tex(instr);
      int plane_index = nir_tex_instr_src_index(tex, nir_tex_src_plane);

      if (plane_index < 0)
         continue;

      nir_const_value *plane = nir_src_as_const_value(tex->src[plane_index].src);
      assume(plane);

      if (plane[0].i32 > 0) {
         unsigned y_samp = tex->texture_index;

         assume(tex->texture_index == tex->sampler_index);
         assume(((state->lower_3plane & (1 << y_samp)) && plane[0].i32 < 3) ||
               (plane[0].i32 < 2));

         tex->texture_index = tex->sampler_index =
               state->sampler_map[y_samp][plane[0].i32 - 1];
      }

      nir_tex_instr_remove_src(tex, plane_index);
   }
}

static void
lower_tex_src_plane_impl(lower_tex_src_state *state, nir_function_impl *impl)
{
   nir_foreach_block(block, impl) {
      lower_tex_src_plane_block(state, block);
   }

   nir_metadata_preserve(impl, nir_metadata_block_index |
                               nir_metadata_dominance);
}

void
st_nir_lower_tex_src_plane(struct nir_shader *shader, unsigned free_slots,
                           unsigned lower_2plane, unsigned lower_3plane)
{
   lower_tex_src_state state = {0};

   state.lower_2plane = lower_2plane;
   state.lower_3plane = lower_3plane;

   assign_extra_samplers(&state, free_slots);

   nir_foreach_function(function, shader) {
      if (function->impl)
         lower_tex_src_plane_impl(&state, function->impl);
   }
}
