/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgservices.h"
#include <utility>

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    struct TestData {
        bool sourceOperationIsTemplate;
        bool recurrentOperationIsTemplate;
    };

    std::vector<TestData> testDataSet {
        { false, false },
        { false, true },
        { true, true },
    };

    // ============================================================================
    for (const auto& testData : testDataSet) {
        const auto sourceOperationIsTemplate = testData.sourceOperationIsTemplate;
        const auto recurrentOperationIsTemplate = testData.recurrentOperationIsTemplate;
        const bool createTemplateForRecurrent = !sourceOperationIsTemplate && recurrentOperationIsTemplate;
        SKGTRACE << "####### Test data set:" << Qt::endl;
        SKGTRACE << "####### sourceOperationIsTemplate=" << (sourceOperationIsTemplate ? "TRUE" : "FALSE") << Qt::endl;
        SKGTRACE << "####### recurrentOperationIsTemplate=" << (recurrentOperationIsTemplate ? "TRUE" : "FALSE") << Qt::endl;
        SKGTRACE << "####### createTemplateForRecurrent=" << (createTemplateForRecurrent ? "TRUE" : "FALSE") << Qt::endl;

        // Test bank document
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGUnitValueObject unit_euro_val1;
        QDate d1 = QDate::currentDate().addMonths(-6);
        QDate d2 = QDate::currentDate().addMonths(-4);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("BANK_T1"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)

            const auto createOperation = [&](SKGOperationObject & op, const QDate & operationDate, const QString & comment, bool isTemplate) {
                SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account.addOperation(op), true)

                SKGTESTERROR(QLatin1String("OP:setTemplate"), op.setTemplate(isTemplate), true)
                SKGTESTERROR(QLatin1String("OP:setMode"), op.setMode(QLatin1String("cheque")), true)
                SKGTESTERROR(QLatin1String("OP:setComment"), op.setComment(comment), true)
                SKGTESTERROR(QLatin1String("OP:setDate"), op.setDate(operationDate), true)
                SKGTESTERROR(QLatin1String("OP:setUnit"), op.setUnit(unit_euro), true)
                SKGTESTERROR(QLatin1String("OP:save"), op.save(), true)
            };

            const auto addSuboperations = [&](SKGOperationObject & op) {
                {
                    SKGSubOperationObject subop;
                    SKGTESTERROR(QLatin1String("OP:addSubOperation"), op.addSubOperation(subop), true)
                    SKGTESTERROR(QLatin1String("SUBOP:setQuantity"), subop.setQuantity(8.5), true)
                    SKGTESTERROR(QLatin1String("SUBOP:save"), subop.save(), true)
                }
                {
                    SKGSubOperationObject subop;
                    SKGTESTERROR(QLatin1String("OP:addSubOperation"), op.addSubOperation(subop), true)
                    SKGTESTERROR(QLatin1String("SUBOP:setQuantity"), subop.setQuantity(10), true)
                    SKGTESTERROR(QLatin1String("SUBOP:save"), subop.save(), true)
                }
            };

            SKGOperationObject op1;
            SKGOperationObject templateOp1;

            SKGTRACE << "####### Test recurrent with an unsaved operation" << Qt::endl;
            SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account.addOperation(op1), true)

            SKGObjectBase::SKGListSKGObjectBase recups;
            SKGTESTERROR(QLatin1String("OP:getRecurrentOperations"), op1.getRecurrentOperations(recups), false)
            SKGTEST(QLatin1String("RECOP:recops.count"), recups.count(), 0)
            SKGTEST(QLatin1String("OP:getRecurrentOperation"), op1.getRecurrentOperation(), 0)

            SKGRecurrentOperationObject recuope1;
            SKGTESTERROR(QLatin1String("OP:addRecurrentOperation"), op1.addRecurrentOperation(recuope1), false)
            SKGTEST(QLatin1String("OP:getRecurrentOperation"), op1.getRecurrentOperation(), 0)

            const auto addGroupedOperation = [&](SKGOperationObject & op, const QDate & operationDate, const QString & comment) {
                SKGTESTERROR(QLatin1String("OP:load"), op.load(), true) {
                    SKGOperationObject op2;
                    SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account.addOperation(op2), true)
                    SKGTESTERROR(QLatin1String("OP:setMode"), op2.setMode(QLatin1String("cheque")), true)
                    SKGTESTERROR(QLatin1String("OP:setComment"), op2.setComment(comment), true)
                    SKGTESTERROR(QLatin1String("OP:setDate"), op2.setDate(operationDate), true)
                    SKGTESTERROR(QLatin1String("OP:setUnit"), op2.setUnit(unit_euro), true)
                    SKGTESTERROR(QLatin1String("OP:save"), op2.save(), true)

                    SKGSubOperationObject subop;
                    SKGTESTERROR(QLatin1String("OP:addSubOperation"), op2.addSubOperation(subop), true)
                    SKGTESTERROR(QLatin1String("SUBOP:setQuantity"), subop.setQuantity(8.5), true)
                    SKGTESTERROR(QLatin1String("SUBOP:save"), subop.save(), true)

                    SKGTESTERROR(QLatin1String("OP:save"), op.setGroupOperation(op2), true)
                    SKGTESTERROR(QLatin1String("OP:save"), op.save(), true)
                }
            };

            const auto createRecurrentOperation = [&](SKGRecurrentOperationObject & recuope, SKGOperationObject & sourceOp, SKGOperationObject & templateOp) {
                SKGOperationObject op = sourceOp;
                if (createTemplateForRecurrent) {
                    IFOKDO(err, op.duplicate(templateOp, op.getDate(), true))
                    op = templateOp;
                }

                SKGTESTERROR(QLatin1String("OP:addRecurrentOperation"), op.addRecurrentOperation(recuope), true)
                SKGTESTERROR(QLatin1String("RECOP:setPeriodIncrement"), recuope.setPeriodIncrement(2), true)
                SKGTESTERROR(QLatin1String("RECOP:setPeriodUnit"), recuope.setPeriodUnit(SKGRecurrentOperationObject::MONTH), true)
                SKGTESTERROR(QLatin1String("RECOP:setAutoWriteDays"), recuope.setAutoWriteDays(6), true)
                SKGTESTERROR(QLatin1String("RECOP:autoWriteEnabled"), recuope.autoWriteEnabled(true), true)
                SKGTESTERROR(QLatin1String("RECOP:setWarnDays"), recuope.setWarnDays(10), true)
                SKGTESTERROR(QLatin1String("RECOP:warnEnabled"), recuope.warnEnabled(true), true)
                SKGTESTERROR(QLatin1String("RECOP:setTimeLimit"), recuope.setTimeLimit(d2), true)
                SKGTESTERROR(QLatin1String("RECOP:timeLimit"), recuope.timeLimit(true), true)
                SKGTEST(QLatin1String("OP:getRecurrentOperation"), op.getRecurrentOperation(), 0)
                SKGTESTERROR(QLatin1String("RECOP:save"), recuope.save(), true)

                if (createTemplateForRecurrent) {
                    SKGTESTERROR(QLatin1String("RECOP:save"), sourceOp.setAttribute(QLatin1String("r_recurrentoperation_id"), SKGServices::intToString(recuope.getID())), true)
                    SKGTESTERROR(QLatin1String("OP:save"), sourceOp.save(), true)
                }

                SKGTESTBOOL(QLatin1String("RECOP:isTemplate"), recuope.isTemplate(), recurrentOperationIsTemplate)
            };

            SKGTRACE << "####### Test recurrent with a saved operation" << Qt::endl;
            createOperation(op1, d1, QLatin1String("10 tickets"), sourceOperationIsTemplate);
            addSuboperations(op1);
            addGroupedOperation(op1, d1, QLatin1String("10 tickets"));

            createRecurrentOperation(recuope1, op1, templateOp1);
            SKGTEST(QLatin1String("OP:getRecurrentOperation"), op1.getRecurrentOperation(), recuope1.getID())
            SKGTESTBOOL(QLatin1String("OP:exist"), templateOp1.exist(), createTemplateForRecurrent)

            SKGRecurrentOperationObject recuope2(recuope1);
            SKGRecurrentOperationObject recuope3(SKGObjectBase(recuope1.getDocument(), QLatin1String("xxx"), recuope1.getID()));
            SKGRecurrentOperationObject recuope4(SKGNamedObject(recuope1.getDocument(), QLatin1String("xxx"), recuope1.getID()));

            SKGObjectBase::SKGListSKGObjectBase recops;
            SKGTESTERROR(QLatin1String("OP:getRecurrentOperations"), op1.getRecurrentOperations(recops), true)
            SKGTEST(QLatin1String("RECOP:recops.count"), recops.count(), createTemplateForRecurrent ? 0 : 1)
            if (createTemplateForRecurrent) {
                SKGTESTERROR(QLatin1String("OP:getRecurrentOperations"), templateOp1.getRecurrentOperations(recops), true)
                SKGTEST(QLatin1String("RECOP:recops.count"), recops.count(), 1)
            }

            SKGTEST(QLatin1String("RECOP:getPeriodIncrement"), recuope1.getPeriodIncrement(), 2)
            SKGTEST(QLatin1String("RECOP:getPeriodUnit"), static_cast<unsigned int>(recuope1.getPeriodUnit()), static_cast<unsigned int>(SKGRecurrentOperationObject::MONTH))
            SKGTEST(QLatin1String("RECOP:getAutoWriteDays"), recuope1.getAutoWriteDays(), 6)
            SKGTESTBOOL("RECOP:isAutoWriteEnabled", recuope1.isAutoWriteEnabled(), true)
            SKGTEST(QLatin1String("RECOP:getWarnDays"), recuope1.getWarnDays(), 10)
            SKGTESTBOOL("RECOP:isWarnEnabled", recuope1.isWarnEnabled(), true)
            SKGTEST(QLatin1String("RECOP:getTimeLimit"), recuope1.getTimeLimit(), 3)
            SKGTESTBOOL("RECOP:hasTimeLimit", recuope1.hasTimeLimit(), true)
            SKGTEST(QLatin1String("RECOP:getDate"), recuope1.getDate().toString(), d1.toString())

            SKGOperationObject ope2;
            SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope1.getParentOperation(ope2), true)
            SKGTESTBOOL("RECOP:op1==ope2", (op1 == ope2), !createTemplateForRecurrent)
            SKGTESTBOOL("RECOP:templateOp1==ope2", (templateOp1 == ope2), createTemplateForRecurrent)
            SKGTEST(QLatin1String("OP:getRecurrentOperation"), ope2.getRecurrentOperation(), recuope1.getID())

            document1.dump(DUMPOPERATION);

            const auto testProcess = [&](SKGRecurrentOperationObject & recuope, int resultingOperationsCount) {
                int nbi = 0;
                SKGTESTERROR(QLatin1String("RECOP:process"), recuope.process(nbi), true)
                SKGTEST(QLatin1String("RECOP:nbi"), nbi, resultingOperationsCount)
                SKGOperationObject opeP;
                SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(opeP), true)
                SKGTESTBOOL("RECOP:op1!=opeP", (op1 != opeP), !sourceOperationIsTemplate)
                SKGTEST(QLatin1String("OP:getRecurrentOperation"), opeP.getRecurrentOperation(), recuope.getID())

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recops.count"), recopsP.count(), resultingOperationsCount + (createTemplateForRecurrent ? 1 : 0))
                for (const auto& opr : std::as_const(recopsP)) {
                    SKGTEST(QLatin1String("OP:getAttribute"), opr.getAttribute(QLatin1String("r_recurrentoperation_id")), SKGServices::intToString(recuope.getID()))
                }

                SKGTESTERROR(QLatin1String("RECOP:SKGRecurrentOperationObject::process"), SKGRecurrentOperationObject::process(&document1, nbi), true)
            };

            SKGTRACE << "####### Process a schedule" << Qt::endl;
            testProcess(recuope1, 3);
            document1.dump(DUMPOPERATION);

            SKGTESTERROR(QLatin1String("RECOP:setPeriodIncrement"), recuope1.setPeriodIncrement(2), true)
            SKGTESTERROR(QLatin1String("RECOP:setPeriodUnit"), recuope1.setPeriodUnit(SKGRecurrentOperationObject::DAY), true)
            recuope1.getNextDate();
            SKGTESTERROR(QLatin1String("RECOP:setPeriodUnit"), recuope1.setPeriodUnit(SKGRecurrentOperationObject::WEEK), true)
            recuope1.getNextDate();
            SKGTESTERROR(QLatin1String("RECOP:setPeriodUnit"), recuope1.setPeriodUnit(SKGRecurrentOperationObject::MONTH), true)
            recuope1.getNextDate();
            SKGTESTERROR(QLatin1String("RECOP:setPeriodUnit"), recuope1.setPeriodUnit(SKGRecurrentOperationObject::YEAR), true)
            recuope1.getNextDate();

            const auto testAssign = [&](SKGRecurrentOperationObject & recuope, const QDate & operationDate, const QString & comment, int resultingOperationsCount) {
                SKGOperationObject opA;
                SKGTESTERROR(QLatin1String("OP:duplicate"), op1.duplicate(opA, operationDate, false), true)
                SKGTESTERROR(QLatin1String("OP:setComment"), opA.setComment(comment), true)
                SKGTESTERROR(QLatin1String("OP:duplicate"), opA.setAttribute(QLatin1String("r_recurrentoperation_id"), QString("0")), true)
                SKGTESTERROR(QLatin1String("OP:save"), opA.save(), true)

                // Update comment for all grouped transactions
                SKGObjectBase::SKGListSKGObjectBase goupops;
                SKGTESTERROR(QLatin1String("OP:getGroupedOperations"), opA.getGroupedOperations(goupops), true)
                for (int i = 0; !err && i < goupops.count(); ++i) {
                    SKGOperationObject groupop(goupops.at(i));
                    SKGTESTERROR(QLatin1String("OP:setComment"), groupop.setComment(comment), true)
                    SKGTESTERROR(QLatin1String("OP:save"), groupop.save(), true)
                }

                auto expectedRecurrentId = recuope.getID();
                if (!recurrentOperationIsTemplate) {
                    SKGOperationObject parentOp;
                    SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                    if (operationDate >= parentOp.getDate()) {
                        // This transaction should become a parent for a non-template schedule
                        expectedRecurrentId = 0;
                    }
                }
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), opA.setRecurrentOperation(recuope.getID()), true)
                SKGTEST(QLatin1String("OP:getAttribute"), opA.getAttribute(QLatin1String("r_recurrentoperation_id")), SKGServices::intToString(expectedRecurrentId))

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recops.count"), recopsP.count(), resultingOperationsCount + (createTemplateForRecurrent ? 1 : 0))
                for (const auto& opr : std::as_const(recopsP)) {
                    SKGTEST(QLatin1String("OP:getAttribute"), opr.getAttribute(QLatin1String("r_recurrentoperation_id")), SKGServices::intToString(recuope.getID()))
                }
            };

            const auto removeOperation = [&](SKGOperationObject & op) {
                const auto removeSubOp = [&](SKGSubOperationObject & subop) {
                    SKGTESTERROR(QLatin1String("SUBOP:remove"), subop.remove(false), true)
                };
                const auto removeOpAndSubops = [&](SKGOperationObject & op) {
                    SKGObjectBase::SKGListSKGObjectBase subops;
                    SKGTESTERROR(QLatin1String("OP:getSubOperations"), op.getSubOperations(subops), true)
                    for (int i = 0; !err && i < subops.count(); ++i) {
                        SKGSubOperationObject subop(subops.at(i));
                        removeSubOp(subop);
                    }
                    SKGTESTERROR(QLatin1String("OP:remove"), op.remove(false), true)
                };
                const auto removeOp = [&](SKGOperationObject & op) {
                    if (op.isInGroup()) {
                        SKGObjectBase::SKGListSKGObjectBase groupops;
                        SKGTESTERROR(QLatin1String("OP:getGroupedOperations"), op.getGroupedOperations(groupops), true)
                        for (int i = 0; !err && i < groupops.count(); ++i) {
                            SKGOperationObject groupop(groupops.at(i));
                            removeOpAndSubops(groupop);
                        }
                    } else {
                        removeOpAndSubops(op);
                    }
                };

                removeOp(op);
            };

            const auto removeAllScheduled = [&](SKGRecurrentOperationObject & recuope) {
                SKGObjectBase::SKGListSKGObjectBase recops;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recops), true)
                for (int i = 0; !err && i < recops.count(); ++i) {
                    SKGOperationObject op(recops.at(i));
                    removeOperation(op);
                }

                recuope.load();
                SKGOperationObject parentOp;
                SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)

                SKGTESTERROR(QLatin1String("RECOP:remove"), recuope.remove(false), true)

                removeOperation(parentOp);
            };

            const auto testDocumentIsEmpty = [&]() {
                bool exists = false;
                SKGTESTERROR(QLatin1String("DOC:existObjects"), document1.existObjects("operation", "", exists), true)
                SKGTESTBOOL(QLatin1String("DOC:operation.exists"), exists, false)
                SKGTESTERROR(QLatin1String("DOC:existObjects"), document1.existObjects("recurrentoperation", "", exists), true)
                SKGTESTBOOL(QLatin1String("DOC:recurrentoperation.exists"), exists, false)
                SKGTESTERROR(QLatin1String("DOC:existObjects"), document1.existObjects("suboperation", "", exists), true)
                SKGTESTBOOL(QLatin1String("DOC:recurrentoperation.suboperation"), exists, false)
            };

            QDate start_date, end_date;

            SKGTRACE << "####### Assign a schedule to a preceeding operation" << Qt::endl;
            start_date = d1.addMonths(-3);
            testAssign(recuope1, start_date, "10 tickets (old)", 4);
            document1.dump(DUMPOPERATION);

            SKGTRACE << "####### Assign a schedule to a succeeding operation" << Qt::endl;
            end_date = d2.addMonths(2);
            testAssign(recuope1, end_date, "10 tickets (new)", 5);
            document1.dump(DUMPOPERATION);

            SKGTRACE << "####### Remove all scheduled transactions" << Qt::endl;
            removeAllScheduled(recuope1);
            testDocumentIsEmpty();
            document1.dump(DUMPOPERATION);

            SKGTRACE << "####### Unassign last transaction from a schedule" << Qt::endl;
            {
                SKGOperationObject op;
                createOperation(op, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op, templateOp);

                const bool shouldFail = sourceOperationIsTemplate && recurrentOperationIsTemplate;
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), op.setRecurrentOperation(0), !shouldFail)
                if (!shouldFail) {
                    SKGTEST(QLatin1String("OP:getAttribute"), op.getAttribute(QLatin1String("r_recurrentoperation_id")), QString("0"))
                }

                document1.dump(DUMPOPERATION);

                recuope.load();
                SKGTESTBOOL(QLatin1String("RECOP:exist"), recuope.exist(), recurrentOperationIsTemplate)

                if (createTemplateForRecurrent) {
                    SKGOperationObject parentOp;
                    SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                    SKGTESTBOOL(QLatin1String("RECOP:exist"), parentOp.exist(), true)
                    SKGTESTBOOL("RECOP:templateOp==parentOp", (templateOp == parentOp), true)
                }

                removeAllScheduled(recuope);
                removeOperation(op);

                testDocumentIsEmpty();
            }

            SKGTRACE << "####### Remove last transaction from a schedule" << Qt::endl;
            {
                SKGOperationObject op;
                createOperation(op, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op, templateOp);

                removeOperation(op);

                document1.dump(DUMPOPERATION);

                recuope.load();
                SKGTESTBOOL(QLatin1String("RECOP:exist"), recuope.exist(), createTemplateForRecurrent)

                if (createTemplateForRecurrent) {
                    SKGOperationObject parentOp;
                    SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                    SKGTESTBOOL(QLatin1String("RECOP:exist"), parentOp.exist(), createTemplateForRecurrent)
                    SKGTESTBOOL("RECOP:templateOp==parentOp", (templateOp == parentOp), true)
                    removeAllScheduled(recuope);
                }

                testDocumentIsEmpty();
            }

            SKGTRACE << "####### Unassign non-last transaction from a schedule" << Qt::endl;
            {
                SKGOperationObject op1;
                createOperation(op1, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op1, templateOp);

                SKGOperationObject op2;
                createOperation(op2, d2, QLatin1String("15 tickets"), false);
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), op2.setRecurrentOperation(recuope.getID()), true)

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recopsP.count"), recopsP.count(), createTemplateForRecurrent ? 2 : 1)
                SKGOperationObject firstOp(recopsP.first());
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), firstOp.setRecurrentOperation(0), true)
                SKGTEST(QLatin1String("OP:getAttribute"), firstOp.getAttribute(QLatin1String("r_recurrentoperation_id")), QString("0"))

                document1.dump(DUMPOPERATION);

                recuope.load();
                SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(recuope.exist()), static_cast<unsigned int>(true))

                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recops.count"), recopsP.count(), createTemplateForRecurrent ? 1 : 0)
                for (const auto& opr : std::as_const(recopsP)) {
                    SKGTEST(QLatin1String("OP:getAttribute"), opr.getAttribute(QLatin1String("r_recurrentoperation_id")), SKGServices::intToString(recuope.getID()))
                }

                SKGOperationObject parentOp;
                SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(parentOp.exist()), static_cast<unsigned int>(true))
                SKGTESTBOOL("RECOP:templateOp==parentOp", (templateOp == parentOp), createTemplateForRecurrent)
                SKGTEST(QLatin1String("OP:getRecurrentOperation"), parentOp.getRecurrentOperation(), recuope.getID())

                removeAllScheduled(recuope);
                removeOperation(firstOp);
                testDocumentIsEmpty();
            }

            SKGTRACE << "####### Remove non-last transaction from a schedule" << Qt::endl;
            {
                SKGOperationObject op1;
                createOperation(op1, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op1, templateOp);

                SKGOperationObject op2;
                createOperation(op2, d2, QLatin1String("15 tickets"), false);
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), op2.setRecurrentOperation(recuope.getID()), true)

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recopsP.count"), recopsP.count(), createTemplateForRecurrent ? 2 : 1)
                SKGOperationObject firstOp(recopsP.first());
                removeOperation(firstOp);

                document1.dump(DUMPOPERATION);

                recuope.load();
                SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(recuope.exist()), static_cast<unsigned int>(true))

                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recops.count"), recopsP.count(), createTemplateForRecurrent ? 1 : 0)
                for (const auto& opr : std::as_const(recopsP)) {
                    SKGTEST(QLatin1String("OP:getAttribute"), opr.getAttribute(QLatin1String("r_recurrentoperation_id")), SKGServices::intToString(recuope.getID()))
                }

                SKGOperationObject parentOp;
                SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(parentOp.exist()), static_cast<unsigned int>(true))
                SKGTESTBOOL("RECOP:templateOp==parentOp", (templateOp == parentOp), createTemplateForRecurrent)
                SKGTEST(QLatin1String("OP:getRecurrentOperation"), parentOp.getRecurrentOperation(), recuope.getID())

                removeAllScheduled(recuope);
                testDocumentIsEmpty();
            }

            SKGTRACE << "####### Toggle template mode for a schedule with a single operation" << Qt::endl;
            {
                SKGOperationObject op;
                createOperation(op, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op, templateOp);

                const bool newTemplateState = !recurrentOperationIsTemplate;
                const bool shouldFail = sourceOperationIsTemplate && recurrentOperationIsTemplate;
                SKGTESTERROR(QLatin1String("RECOP:setTemplate"), recuope.setTemplate(newTemplateState), !shouldFail)
                if (!shouldFail) {
                    SKGOperationObject parentOp;
                    SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                    SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(parentOp.exist()), static_cast<unsigned int>(true))
                    SKGTESTBOOL("RECOP:parentOp.idTemplate", parentOp.isTemplate(), newTemplateState)
                }

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recopsP.count"), recopsP.count(), !!newTemplateState && !sourceOperationIsTemplate ? 1 : 0)

                document1.dump(DUMPOPERATION);

                removeAllScheduled(recuope);
                testDocumentIsEmpty();
            }

            SKGTRACE << "####### Toggle template mode for a schedule with several transactions" << Qt::endl;
            {
                SKGOperationObject op1;
                createOperation(op1, d1, QLatin1String("20 tickets"), sourceOperationIsTemplate);

                SKGRecurrentOperationObject recuope;
                SKGOperationObject templateOp;
                createRecurrentOperation(recuope, op1, templateOp);

                SKGOperationObject op2;
                createOperation(op2, d2, QLatin1String("15 tickets"), false);
                SKGTESTERROR(QLatin1String("OP:setRecurrentOperation"), op2.setRecurrentOperation(recuope.getID()), true)

                const bool newTemplateState = !recurrentOperationIsTemplate;
                recuope.load();
                SKGTESTERROR(QLatin1String("RECOP:setTemplate"), recuope.setTemplate(newTemplateState), true)
                SKGOperationObject parentOp;
                SKGTESTERROR(QLatin1String("RECOP:getParentOperation"), recuope.getParentOperation(parentOp), true)
                SKGTEST(QLatin1String("RECOP:exist"), static_cast<unsigned int>(parentOp.exist()), static_cast<unsigned int>(true))
                SKGTESTBOOL("RECOP:parentOp.idTemplate", parentOp.isTemplate(), newTemplateState)

                SKGObjectBase::SKGListSKGObjectBase recopsP;
                SKGTESTERROR(QLatin1String("RECOP:getRecurredOperations"), recuope.getRecurredOperations(recopsP), true)
                SKGTEST(QLatin1String("RECOP:recopsP.count"), recopsP.count(), !sourceOperationIsTemplate ? (!!newTemplateState ? 2 : 1) : 0)

                document1.dump(DUMPOPERATION);

                removeAllScheduled(recuope);
                testDocumentIsEmpty();
            }
        }

        //492287
        {
            SKGError err;
            SKGDocumentBank document1;
            SKGTESTERROR(QLatin1String("document1.load()"), document1.load(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestreccurent/492287.skg"), true) {}
        }
    }

    // End test
    SKGENDTEST()
}
