/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file defines classes SKGInterestObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skginterestobject.h"
#include "skgdocument.h"

SKGInterestObject::SKGInterestObject(): SKGInterestObject(nullptr)
{}

SKGInterestObject::SKGInterestObject(SKGDocument* iDocument, int iID): SKGObjectBase(iDocument, QLatin1String("v_interest"), iID)
{}

SKGInterestObject::SKGInterestObject(const SKGInterestObject& iObject)
    = default;

SKGInterestObject::SKGInterestObject(const SKGObjectBase& iObject)
{
    if (iObject.getRealTable() == QLatin1String("interest")) {
        copyFrom(iObject);
    } else {
        *this = SKGObjectBase(iObject.getDocument(), QLatin1String("v_interest"), iObject.getID());
    }
}

SKGInterestObject& SKGInterestObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGInterestObject& SKGInterestObject::operator= (const SKGInterestObject& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGInterestObject::~SKGInterestObject()
    = default;

SKGError SKGInterestObject::setRate(double iValue)
{
    return setAttribute(QLatin1String("f_rate"), SKGServices::doubleToString(iValue));
}

double SKGInterestObject::getRate() const
{
    return SKGServices::stringToDouble(getAttribute(QLatin1String("f_rate")));
}

SKGError SKGInterestObject::setDate(QDate iDate)
{
    return setAttribute(QLatin1String("d_date"), SKGServices::dateToSqlString(iDate));
}

QDate SKGInterestObject::getDate() const
{
    return SKGServices::stringToDate(getAttribute(QLatin1String("d_date")));
}

SKGError SKGInterestObject::setIncomeValueDateMode(SKGInterestObject::ValueDateMode iMode)
{
    return setAttribute(QLatin1String("t_income_value_date_mode"), (iMode == FIFTEEN ? QLatin1String("F") : SKGServices::intToString(static_cast<int>(iMode) - 1)));
}

SKGInterestObject::ValueDateMode SKGInterestObject::getIncomeValueDateMode() const
{
    QString mode = getAttribute(QLatin1String("t_income_value_date_mode"));
    return (mode == QLatin1String("F") ? FIFTEEN : static_cast<SKGInterestObject::ValueDateMode>(SKGServices::stringToInt(mode) + 1));
}

SKGError SKGInterestObject::setExpenditueValueDateMode(SKGInterestObject::ValueDateMode iMode)
{
    return setAttribute(QLatin1String("t_expenditure_value_date_mode"), (iMode == FIFTEEN ? QLatin1String("F") : SKGServices::intToString(static_cast<int>(iMode) - 1)));
}

SKGInterestObject::ValueDateMode SKGInterestObject::getExpenditueValueDateMode() const
{
    QString mode = getAttribute(QLatin1String("t_expenditure_value_date_mode"));
    return (mode == QLatin1String("F") ? FIFTEEN : static_cast<SKGInterestObject::ValueDateMode>(SKGServices::stringToInt(mode) + 1));
}

SKGError SKGInterestObject::setInterestComputationMode(SKGInterestObject::InterestMode iMode)
{
    return setAttribute(QLatin1String("t_base"),
                        (iMode == FIFTEEN24 ? QLatin1String("24") :
                         (iMode == DAYS360 ? QLatin1String("360") :
                          QLatin1String("365"))));
}

SKGInterestObject::InterestMode SKGInterestObject::getInterestComputationMode() const
{
    QString mode = getAttribute(QLatin1String("t_base"));
    return (mode == QLatin1String("24") ? FIFTEEN24 : (mode == QLatin1String("360") ? DAYS360 : DAYS365));
}

QString SKGInterestObject::getWhereclauseId() const
{
    // Could we use the id
    QString output = SKGObjectBase::getWhereclauseId();
    if (output.isEmpty()) {
        // No, so we use the date and parent
        if (!(getAttribute(QLatin1String("d_date")).isEmpty()) && !(getAttribute(QLatin1String("rd_account_id")).isEmpty())) {
            output = "d_date='" % getAttribute(QLatin1String("d_date")) % "' AND rd_account_id=" % getAttribute(QLatin1String("rd_account_id"));
        }
    }
    return output;
}

SKGError SKGInterestObject::setAccount(const SKGAccountObject& iAccount)
{
    return setAttribute(QLatin1String("rd_account_id"), SKGServices::intToString(iAccount.getID()));
}

SKGError SKGInterestObject::getAccount(SKGAccountObject& oAccount) const
{
    SKGError err = getDocument()->getObject(QLatin1String("v_account"), "id=" % getAttribute(QLatin1String("rd_account_id")), oAccount);
    return err;
}


