/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOPERATIONPLUGIN_H
#define SKGOPERATIONPLUGIN_H
/** @file
* This file is Skrooge plugin for operation management.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skginterfaceplugin.h"
#include "ui_skgoperationpluginwidget_pref.h"

class QMenu;
class QAction;
class SKGDocumentBank;
class KToolBarPopupAction;

/**
 * This file is Skrooge plugin for operation management
 */
class SKGOperationPlugin : public SKGInterfacePlugin
{
    Q_OBJECT
    Q_INTERFACES(SKGInterfacePlugin)

public:
    /**
     * Default Constructor
     */
    explicit SKGOperationPlugin(QWidget* iWidget, QObject* iParent, const QVariantList& iArg);

    /**
     * Default Destructor
     */
    virtual ~SKGOperationPlugin();

    /**
     * Called to initialise the plugin
     * @param iDocument the main document
     * @param iArgument the arguments
     * @return true if the plugin is compatible with the document
     */
    virtual bool setupActions(SKGDocument* iDocument, const QStringList& iArgument) Q_DECL_OVERRIDE;

    /**
     * Must be modified to refresh widgets after a modification.
     */
    virtual void refresh() Q_DECL_OVERRIDE;

    /**
     * The number of dashboard widgets of the plugin.
     * @return The number of dashboard widgets of the plugin
     */
    virtual int getNbDashboardWidgets() Q_DECL_OVERRIDE;

    /**
     * Get a dashboard widget title of the plugin.
     * @param iIndex the index of the widget
     * @return The title
     */
    virtual QString getDashboardWidgetTitle(int iIndex) Q_DECL_OVERRIDE;

    /**
     * Get a dashboard widget of the plugin.
     * @param iIndex the index of the widget
     * @return The dashboard widget of the plugin
     */
    virtual SKGBoardWidget* getDashboardWidget(int iIndex) Q_DECL_OVERRIDE;

    /**
     * The preference widget of the plugin.
     * @return The preference widget of the plugin
     */
    virtual QWidget* getPreferenceWidget() Q_DECL_OVERRIDE;

    /**
     * The preference skeleton of the plugin.
     * @return The preference skeleton of the plugin
     */
    virtual KConfigSkeleton* getPreferenceSkeleton() Q_DECL_OVERRIDE;

    /**
     * This function is called when preferences have been modified. Must be used to save some parameters into the document.
     * A transaction is already opened
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError savePreferences() const Q_DECL_OVERRIDE;

    /**
     * The page widget of the plugin.
     * @return The page widget of the plugin
     */
    virtual SKGTabPage* getWidget() Q_DECL_OVERRIDE;

    /**
     * The title of the plugin.
     * @return The title of the plugin
     */
    virtual QString title() const Q_DECL_OVERRIDE;

    /**
     * The icon of the plugin.
     * @return The icon of the plugin
     */
    virtual QString icon() const Q_DECL_OVERRIDE;

    /**
     * The toolTip of the plugin.
     * @return The toolTip of the plugin
     */
    virtual QString toolTip() const Q_DECL_OVERRIDE;

    /**
     * The tips list of the plugin.
     * @return The tips list of the plugin
     */
    virtual QStringList tips() const Q_DECL_OVERRIDE;

    /**
     * Must be implemented to set the position of the plugin.
     * @return integer value between 0 and 999 (default = 999)
     */
    virtual int getOrder() const Q_DECL_OVERRIDE;

    /**
     * Must be implemented to know if a plugin must be display in pages chooser.
     * @return true of false (default = false)
     */
    virtual bool isInPagesChooser() const Q_DECL_OVERRIDE;

    /**
     * The advice list of the plugin.
     * @return The advice list of the plugin
     */
    virtual SKGAdviceList advice(const QStringList& iIgnoredAdvice) Q_DECL_OVERRIDE;

    /**
     * Must be implemented to execute the automatic correction for the advice.
     * @param iAdviceIdentifier the identifier of the advice
     * @param iSolution the identifier of the possible solution
     * @return an object managing the error. MUST return ERR_NOTIMPL if iAdviceIdentifier is not known
     *   @see SKGError
     */
    virtual SKGError executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) Q_DECL_OVERRIDE;

private Q_SLOTS:
    void onDuplicate();
    void onCreateTemplate();
    void onSwitchToPointed();
    void onOpenOperations();
    void onGroupOperation();
    void onUngroupOperation();
    void onMergeSubOperations();
    void onAlignComment();
    void onAlignDate();
    void onRemoveGroupWithOneOperation();
    void onShowApplyTemplateMenu();
    void onApplyTemplate();

private:
    Q_DISABLE_COPY(SKGOperationPlugin)

    QAction* m_duplicateAction;
    QAction* m_createTemplateAction;
    QAction* m_switchToPointedAction;
    QAction* m_openHighLights;
    QAction* m_openLastModified;
    QAction* m_openSuboperations;
    QAction* m_openDuplicate;
    QAction* m_groupOperation;
    QAction* m_ungroupOperation;
    QAction* m_mergeOperationAction;
    QAction* m_open;
    QAction* m_cleanAlignDate;
    QAction* m_cleanAlignComment;
    QAction* m_cleanRemoveGroupWithOneOperation;
    KToolBarPopupAction* m_applyTemplateAction;
    QMenu* m_applyTemplateMenu;

    SKGDocumentBank* m_currentBankDocument;

    Ui::skgoperationplugin_pref ui;
};

#endif  // SKGOPERATIONPLUGIN_H
