/*
  This file is part of the Groupware/KOrganizer integration.

  Requires the Qt and KDE widget libraries, available at no cost at
  http://www.trolltech.com and http://www.kde.org respectively

  Copyright (c) 2002 Klarlvdalens Datakonsult AB

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "kogroupware.h"
#include "koprefs.h"
#include "calendarview.h"
#include "mailscheduler.h"
#include "kogroupwareincomingdialogimpl.h"
#include "koviewmanager.h"
#include "koeventeditor.h"
#include "kotodoeditor.h"
#include "kotodoview.h"

#include <libkcal/incidencebase.h>
#include <libkcal/attendee.h>
#include <libkcal/freebusy.h>
#include <libkcal/journal.h>
#include <libkcal/calendarimap.h>
#include <libkcal/calformat.h>

#include <kdebug.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kio/netaccess.h>
#include <kapplication.h>
#include <kconfig.h>

#include <qfile.h>
#include <qregexp.h>
#include <qmessagebox.h>

KOGroupware* KOGroupware::mInstance = 0;

KOGroupware* KOGroupware::create( QObject* kmailTarget, CalendarView* view,
                                  KCal::Calendar* calendar )
{
  if( !mInstance )
    mInstance = new KOGroupware( kmailTarget, view, calendar );
  return mInstance;
}

KOGroupware* KOGroupware::instance()
{
  // Doesn't create, that is the task of create()
  Q_ASSERT( mInstance );
  return mInstance;
}


KOGroupware::KOGroupware( QObject* kmailTarget, CalendarView* view,
                          KCal::Calendar* calendar )
  : QObject( 0, "kmgroupware_instance" ), mTimerID( 0 ),
    mUploadingFreeBusy( false ), mCanEditEvents( true ), mCanEditTodos( true ),
    mEventDialogCount( 0 ), mTodoDialogCount( 0 )
{
  // Tell korgac we're running in groupware mode
  KConfig config( "korganizerrc" );
  config.setGroup("Groupware");
  config.writeEntry( "Running in groupware mode", true );

  mView = view;
  mCalendar = calendar;
  CalendarIMAP* mCalendarIMAP = dynamic_cast<CalendarIMAP*>( mCalendar );
  if( mCalendarIMAP ) {
    connect( mCalendarIMAP, SIGNAL( eventsChanged() ),
	     this, SLOT( slotPerhapsUploadFB() ) );
    connect( kmailTarget, SIGNAL( syncRunning( const QString&, bool ) ),
	     mCalendarIMAP, SLOT( slotSyncRunning( const QString&, bool ) ) );
  }

  connect( this, SIGNAL( conflict( const QString& ) ),
	   kmailTarget, SLOT( slotConflictResolved( const QString& ) ) );
  connect( this, SIGNAL( newOrUpdatedNote( const QString&, const QString&, const QColor&, const QString&, bool ) ),
	   kmailTarget, SLOT( slotNewOrUpdatedNote( const QString&, const QString&, const QColor&, const QString&, bool ) ) );
  connect( this, SIGNAL( deletedNote( const QString& ) ),
	   kmailTarget, SLOT( slotDeleteNote( const QString& ) ) );
  connect( kmailTarget, SIGNAL( signalNoteAdded( const QString&, bool& ) ),
	   this, SLOT( slotNoteAdded( const QString&, bool& ) ) );
  connect( kmailTarget, SIGNAL( signalNoteDeleted( const QString& ) ),
	   this, SLOT( slotNoteRemoved( const QString& ) ) );
  connect( this, SIGNAL( doSendICalMessage( const QString&, const QString&,
					    const QStringList&, const QString& ) ),
	   kmailTarget, SLOT( slotSendICalMessage( const QString&, const QString&,
						   const QStringList&, const QString& ) ) );
  connect( kmailTarget, SIGNAL( syncRunning( const QString&, bool ) ),
	   this, SLOT( slotSyncRunning( const QString&, bool ) ) );
  connect( this, SIGNAL( dialogsUp( const QString&, bool ) ),
	   kmailTarget, SLOT( slotHoldSyncing( const QString&, bool ) ) );
  connect( this, SIGNAL( getStripAlarmsForSending( bool& ) ),
	   kmailTarget, SLOT( slotGetStripAlarmsForSending( bool& ) ) );
}


/*!
  This method processes incoming event requests. The requests can
  already be tentatively or unconditionally accepted, in which case
  all this method does is registering them in the calendar. If the
  event request is a query, a dialog will be shown that lets the
  user select whether to accept, tentatively accept or decline the
  invitation.

  \param state whether the request is preaccepted, tentatively
  preaccepted or not yet decided on. The state Declined should not be
  used for incoming state values, it is only used for internal
  handling.
  \param vCalIn a string containing the vCal representation of the
  event request
  \param vCalInOK is true after the call if vCalIn was well-formed
  \param vCalOut a string containing the vCal representation of the
  answer. This string is suitable for sending back to the event
  originator. If vCalInOK is false on return, the value of this
  parameter is undefined.
  \param vCalOutOK is true if the user has made a decision, or if the
  event was preaccepted. If this parameter is false, the user has
  declined to decide on whether to accept or decline the event. The
  value of vCalOut is undefined in this case.
*/

KOGroupware::EventState KOGroupware::incomingEventRequest( EventState inState,
							   const QCString& receiver,
							   const QString& vCalIn,
							   bool& vCalInOK,
							   QString& vCalOut,
							   bool& vCalOutOK )
{
  EventState outState = inState;

  // Parse the event request into a ScheduleMessage; this needs to
  // be done in any case.
  KCal::ScheduleMessage *message = mFormat.parseScheduleMessage( mCalendar,
								 vCalIn );
  if( message ) {
    // kdDebug(5850) << "KOGroupware::incomingEventRequest: got message '"
    // << vCalIn << "'" << endl;
    vCalInOK = true;
  } else {
    QString errorMessage;
    if( mFormat.exception() ) {
      errorMessage = mFormat.exception()->message();
    }
    kdDebug(5850) << "KOGroupware::incomingEventRequest() Error parsing "
      "message: " << errorMessage << endl;
    vCalInOK = false;
    // If the message was broken, there's nothing we can do.
    return outState;
  }

  KCal::Incidence* event = dynamic_cast<KCal::Incidence*>( message->event() );
  Q_ASSERT( event );
  if( !event ) { // something bad happened, just to be safe
    vCalOutOK = false;
    return outState;
  }

  // Now check if the event needs to be accepted or if this is
  // already done.
  if( inState == Request ) {
    // Need to accept, present it to the user
    KOGroupwareIncomingDialogImpl dlg( event );
    int ret = dlg.exec();
    if( ret == QDialog::Rejected ) {
      // User declined to make a choice, we can't send a vCal back
      vCalOutOK = false;
      return outState;
    }

    if( dlg.isDeclined() )
      outState = Declined;
    else if( dlg.isConditionallyAccepted() )
      outState = ConditionallyAccepted;
    else if( dlg.isAccepted() )
      outState = Accepted;
    else
      kdDebug(5850) << "KOGroupware::incomingEventRequest(): unknown event request state" << endl;
  }

  // If the event has an alarm, make sure it doesn't have a negative time.
  // This is yet another OL workaround
  QPtrList<Alarm> alarms = event->alarms();
  Alarm* alarm;
  QPtrListIterator<Alarm> it( alarms );
  for (; (alarm = it.current()) != 0; ++it) {
    if ( alarm->hasTime() ) {
      QDateTime t = alarm->time();
      int offset = event->dtStart().secsTo( t );
      if( offset > 0 )
	// PENDING(Bo): Not implemented yet
	kdDebug(5850) << "Warning: Alarm fires after the event\n";
    } else {
      int offset = alarm->offset().asSeconds();
      if( offset > 0 ) {
	// This number should be negative so the alarm fires before the event
	Duration d( -offset );
	alarm->setOffset( d );
      }
    }
  }

  // Enter the event into the calendar. We just create a
  // Scheduler, because all the code we need is already there. We
  // take an MailScheduler, because we need a concrete one, but we
  // really only want code from Scheduler.
  // PENDING(kalle) Handle tentative acceptance differently.
  if( outState == Accepted || outState == ConditionallyAccepted ) {
    KCal::MailScheduler scheduler( mCalendar );
    scheduler.acceptTransaction( event,
				 (KCal::Scheduler::Method)message->method(),
				 message->status() );
    mView->updateView();
  }

  QPtrList<KCal::Attendee> attendees = event->attendees();
  QPtrListIterator<KCal::Attendee> attendeeIt( attendees );
  KCal::Attendee* myself = 0, *current = 0;;
  // Find myself, there will always be all attendees listed, even if
  // only I need to answer it.
  while( ( current = attendeeIt.current() ) != 0 ) {
    ++attendeeIt;
    if( current->email().utf8() == receiver ) {
      // We are the current one, and even the receiver, note
      // this and quit searching.
      myself = current;
      break;
    }

    if( current->email() == KOPrefs::instance()->email() ) {
      // If we are the current one, note that. Still continue to
      // search in case we find the receiver himself.
      myself = current;
    }
  }

  Q_ASSERT( myself );

  KCal::Attendee* newMyself = 0;
  if( myself ) {
    switch( outState ) {
    case Accepted:
      myself->setStatus( KCal::Attendee::Accepted );
      break;
    case ConditionallyAccepted:
      myself->setStatus( KCal::Attendee::Tentative );
      break;
    case Declined:
      myself->setStatus( KCal::Attendee::Declined );
      break;
    default:
      ;
    };

    // No more request response
    myself->setRSVP(false);

    event->updated();

    newMyself = new KCal::Attendee( myself->name(),
				    receiver.isEmpty() ?
				    myself->email() :
				    receiver,
				    myself->RSVP(),
				    myself->status(),
				    myself->role(),
				    myself->uid() );
  }

  event->updated();

  // Send back the answer; construct it on the base of state. We
  // make a clone of the event since we need to manipulate it here.
  // NOTE: This contains a workaround around a libkcal bug: REPLY
  // vCals may not have more than one ATTENDEE (as libical correctly
  // specifies), but libkcal always writes out all the ATTENDEEs,
  // thus producing invalid vCals. We make a clone of the vEvent
  // here and remove all attendees except ourselves.
  Incidence* newIncidence = event->clone();
  Event* newEvent = static_cast<KCal::Event*>( newIncidence );

  newEvent->clearAttendees();

  // OL compatibility setting
  bool stripAlarms = false;
  emit getStripAlarmsForSending( stripAlarms );
  if( stripAlarms )
    // Strip alarms from the send
    newEvent->clearAlarms();

  if( newMyself )
    newEvent->addAttendee( newMyself );

  // Create the outgoing vCal
  QString messageText = mFormat.createScheduleMessage( newEvent,
						       KCal::Scheduler::Reply );

  // Fix broken OL appointments
  if( vCalIn.contains( "PRODID:-//Microsoft" ) ) {
    // OL doesn't send the organizer as an attendee as it should
    Attendee* organizer = new KCal::Attendee( i18n("Organizer"),
					      event->organizer(), false,
					      KCal::Attendee::Accepted );
    event->addAttendee( organizer );
  }

  vCalOut = messageText;
  vCalOutOK = true;
  return outState;
}


/*!
  This method processes resource requests. KMail has at this point
  already decided whether the request should be accepted or declined.
*/
void KOGroupware::incomingResourceRequest( const QValueList<QPair<QDateTime, QDateTime> >& busy,
                                           const QCString& resource,
                                           const QString& vCalIn,
                                           bool& vCalInOK,
                                           QString& vCalOut,
                                           bool& vCalOutOK,
                                           bool& isFree,
                                           QDateTime& start,
                                           QDateTime& end )
{
  // Parse the event request into a ScheduleMessage; this needs to
  // be done in any case.
  KCal::ScheduleMessage *message = mFormat.parseScheduleMessage( mCalendar,
								 vCalIn );
  if( message )
    vCalInOK = true;
  else {
    QString errorMessage;
    if( mFormat.exception() ) {
      errorMessage = mFormat.exception()->message();
    }
    kdDebug(5850) << "KOGroupware::incomingResourceRequest() Error parsing "
      "message: " << errorMessage << endl;
    vCalInOK = false;
    // If the message was broken, there's nothing we can do.
    return;
  }

  KCal::Event* event = dynamic_cast<KCal::Event*>( message->event() );
  Q_ASSERT( event );
  if( !event ) {
    // Something has gone badly wrong
    vCalInOK = false;
    return;
  }

  // Now find out whether the resource is free at the requested
  // time, take the opportunity to assign the reference parameters.
  start = event->dtStart();
  end = event->dtEnd();
  isFree = true;
  for( QValueList<QPair<QDateTime, QDateTime> >::ConstIterator it = busy.begin();
       it != busy.end(); ++it ) {
    if( (*it).second <= start || // busy period ends before try
	// period
	(*it).first >= end )  // busy period starts after try
      // period
      continue;
    else {
      isFree = false;
      break; // no need to search further
    }
  }

  // Send back the answer; construct it on the base of state
  QPtrList<KCal::Attendee> attendees = event->attendees();
  KCal::Attendee* resourceAtt = 0;

  // Find the resource addresse, there will always be all attendees
  // listed, even if only one needs to answer it.
  for( KCal::Attendee* current = attendees.first(); current;
       current = attendees.next() )
    if( current->email().utf8() == resource ) {
      resourceAtt = current;
      break;
    }
  Q_ASSERT( resourceAtt );
  if( resourceAtt ) {
    if( isFree )
      resourceAtt->setStatus( KCal::Attendee::Accepted );
    else
      resourceAtt->setStatus( KCal::Attendee::Declined );
  } else {
    vCalOutOK = false;
    return;
  }

  // Create the outgoing vCal
  QString messageText = mFormat.createScheduleMessage( event,
						       KCal::Scheduler::Reply );
  // kdDebug(5850) << "Sending vCal back to KMail: " << messageText << endl;
  vCalOut = messageText;
  vCalOutOK = true;
  return;
}



/*!
  This method loads the specified notes into KOrganizer's calendar.
*/
void KOGroupware::refreshNotes( const QStringList& notes )
{
  mNotes.clear();
  for( QStringList::ConstIterator it = notes.begin(); it != notes.end(); ++it )
    noteAdded( *it );
  emit notesUpdated();
}

QValueList<Note> KOGroupware::notes() const
{
  return mNotes;
}

void KOGroupware::slotNoteNewOrUpdated( const Note& note )
{
  for( QValueList<Note>::Iterator it = mNotes.begin(); it != mNotes.end(); ++it ) {
    if( (*it).id == note.id ) {
      if( *it == note )
	// Nothing changed
	return;

      // Update the note
      *it = note;
      emit newOrUpdatedNote( note.id, note.geometry, note.color,
			     note.text, false );
      return;
    }
  }
  emit newOrUpdatedNote( note.id, note.geometry, note.color,
			 note.text, false );
  mNotes.append(note);
}

void KOGroupware::slotNoteDeleted( const Note& note )
{
  QString uid = note.id;
  noteDeleted( uid );
  emit deletedNote( uid );
}

/*!
  Delete the note specified by the uid
*/
void KOGroupware::noteDeleted( const QString& uid )
{
  for( QValueList<Note>::Iterator it = mNotes.begin(); it != mNotes.end(); ++it )
    if( (*it).id == uid ) {
      mNotes.remove( it );
      break;
    }
  mView->updateView();
}

/*!
  Add a note
*/
bool KOGroupware::noteAdded( const QString& noteString )
{
  // TODO: proper parsing of note
  Note note;
  QRegExp noteid("X-KOrg-Note-Id:([^\n]*)\n");
  if( noteid.search( noteString ) != -1 ) {
    note.id = noteid.cap(1).stripWhiteSpace();
  } else {
    //note.id =
  }
  QRegExp notegeom("X-KOrg-Note-Geometry:([^\n]*)\n");
  if( notegeom.search( noteString ) != -1 ) {
    note.geometry = notegeom.cap(1).stripWhiteSpace();
  }
  QRegExp notecolor("X-KOrg-Note-Color:([^\n]*)\n");
  if( notecolor.search( noteString ) != -1 && !notecolor.capturedTexts().first().isEmpty()) {
    note.color = QColor(notecolor.cap(1).stripWhiteSpace());
  }
  int pos = noteString.find("\n\n");
  if( pos > -1 )
    note.text = noteString.mid(pos+2);

  // Figure out if the note ID already exists
  bool rc = true;
  QValueList<Note>::Iterator it = mNotes.begin();
  for( ; it != mNotes.end(); ++it ) {
    if( (*it).id == note.id ) {
      // We have a match. And that means a conflict
      // Give the new note a new ID, add it to the list and make KMail
      // delete the one it tried to add
      rc = false;
      note.id = CalFormat::createUniqueId();
      emit conflict( "Note" );
      emit newOrUpdatedNote( note.id, note.geometry, note.color,
			     note.text, true );
      break;
    }
  }
  
  mNotes << note;

  return rc;
}

/* Called by KMail when a note is added to the folder */
void KOGroupware::slotNoteAdded( const QString& msg, bool& accepted )
{
  accepted = noteAdded( msg );
  emit notesUpdated();
}

void KOGroupware::slotNoteRemoved( const QString& uid )
{
  noteDeleted( uid );
  emit notesUpdated();
}

/*!
  This method is called when the user has invited people and is now
  receiving the answers to the invitation. Returns the updated vCal
  representing the whole incidence in the vCalOut parameter.

  If returning false because of something going wrong, set vCalOut
  to "false" to signal this to KMail.
*/

bool KOGroupware::incidenceAnswer( const QCString& sender, const QString& vCalIn,
				   QString& vCalOut )
{
  vCalOut = "";

  // Parse the event request into a ScheduleMessage; this needs to
  // be done in any case.
  KCal::ScheduleMessage *message = mFormat.parseScheduleMessage( mCalendar, vCalIn );
  if( !message ) {
    // a parse error of some sort
    KMessageBox::error( mView, i18n("<b>There was a problem parsing the iCal data:</b><br>%1")
			.arg(mFormat.exception()->message()) );
    vCalOut = "false";
    return false;
  }

  KCal::IncidenceBase* incidence = message->event();

  // Enter the answer into the calendar. We just create a
  // Scheduler, because all the code we need is already there. We
  // take a MailScheduler, because we need a concrete one, but we
  // really only want code from Scheduler.
  QString uid = incidence->uid();
  KCal::MailScheduler scheduler( mCalendar );
  if( !scheduler.acceptTransaction( incidence, (KCal::Scheduler::Method)message->method(),
				    message->status(), sender ) ) {
    KMessageBox::error( mView, i18n("Scheduling failed") );
    vCalOut = "false";
    return false;
  }

  mView->updateView();

  // Find the calendar entry that corresponds to this uid.
  if( Event* event = mCalendar->event( uid ) ) {
    vCalOut = mFormat.createScheduleMessage( event, KCal::Scheduler::Reply );
  } else if( Todo* todo = mCalendar->todo( uid ) )
    vCalOut = mFormat.createScheduleMessage( todo, KCal::Scheduler::Reply );
  else if( Journal* journal = mCalendar->journal( uid ) )
    vCalOut = mFormat.createScheduleMessage( journal, KCal::Scheduler::Reply );

  return true;
}


QString KOGroupware::getFreeBusyString()
{
  QDateTime start = QDateTime::currentDateTime();
  QDateTime end = start.addDays( KOPrefs::instance()->mPublishFreeBusyDays );

  FreeBusy freebusy( mCalendar, start, end );
  freebusy.setOrganizer( KOPrefs::instance()->email() );
  mFormat.setTimeZone( mCalendar->timeZoneId(), !mCalendar->isLocalTime() );

//   kdDebug(5850) << "KOGroupware::publishFreeBusy(): startDate: "
// 		<< KGlobal::locale()->formatDateTime( start ) << " End Date: "
// 		<< KGlobal::locale()->formatDateTime( end ) << endl;

  return mFormat.createScheduleMessage( &freebusy, Scheduler::Publish );
}


// When something changed in the calendar, we get this called
void KOGroupware::slotPerhapsUploadFB()
{
  if( mTimerID != 0 )
    // A timer is already running, so we don't need to do anything
    return;

  int now = static_cast<int>( QDateTime::currentDateTime().toTime_t() );
  int eta = static_cast<int>( mNextUploadTime.toTime_t() ) - now;

  if( !mUploadingFreeBusy ) {
    // Not currently uploading
    if( mNextUploadTime.isNull() ||
	QDateTime::currentDateTime() > mNextUploadTime ) {
      // No uploading have been done in this session, or delay time is over
      publishFreeBusy();
      return;
    }

    // We're in the delay time and no timer is running. Start one
    if( eta <= 0 ) {
      // Sanity check failed - better do the upload
      publishFreeBusy();
      return;
    }
  } else {
    // We are currently uploading the FB list. Start the timer
    if( eta <= 0 ) {
      kdDebug(5850) << "This shouldn't happen! eta <= 0\n";
      eta = 10; // whatever
    }
  }

  // Start the timer
  mTimerID = startTimer( eta * 1000 );

  if( mTimerID == 0 )
    // startTimer failed - better do the upload
    publishFreeBusy();
}

// This is used for delayed Free/Busy list uploading
void KOGroupware::timerEvent( QTimerEvent* )
{
  publishFreeBusy();
}

/*!
  This method is called when the user has selected to publish its
  free/busy list or when the delay have passed.
*/
void KOGroupware::publishFreeBusy()
{
  // Already uploading? Skip this one then.
  if ( mUploadingFreeBusy )
    return;
  mUploadingFreeBusy = true;

  // If we have a timer running, it should be stopped now
  if( mTimerID != 0 ) {
    killTimer( mTimerID );
    mTimerID = 0;
  }

  // Save the time of the next free/busy uploading
  mNextUploadTime = QDateTime::currentDateTime();
  if( KOPrefs::instance()->mPublishDelay > 0 )
    mNextUploadTime = mNextUploadTime.addSecs( KOPrefs::instance()->mPublishDelay * 60 );

  QString messageText = getFreeBusyString();

  // We need to massage the list a bit so that Outlook understands
  // it.
  messageText = messageText.replace( QRegExp( "ORGANIZER\\s*:MAILTO:" ), "ORGANIZER:" );

  QString emailHost = KOPrefs::instance()->email().mid( KOPrefs::instance()->email().find( '@' ) + 1 );

  // Create a local temp file and save the message to it
  KTempFile tempFile;
  QTextStream* textStream = tempFile.textStream();
  if( textStream ) {
    *textStream << messageText;
    tempFile.close();

    // Put target string together
    KURL targetURL;
    if( KOPrefs::instance()->mPublishKolab ) {
      // we use Kolab
      QString server;
      if( KOPrefs::instance()->mPublishKolabServer == "%SERVER%" ||
	  KOPrefs::instance()->mPublishKolabServer.isEmpty() )
	server = emailHost;
      else
	server = KOPrefs::instance()->mPublishKolabServer;

      targetURL.setProtocol( "webdavs" );
      targetURL.setHost( server );

      QString fbname = KOPrefs::instance()->mPublishUserName;
      int at = fbname.find('@');
      if( at > 1 && fbname.length() > (uint)at ) {
	fbname = fbname.left(at);
      }
      targetURL.setPath( "/freebusy/" + fbname + ".vfb" );
      targetURL.setUser( KOPrefs::instance()->mPublishUserName );
      targetURL.setPass( KOPrefs::instance()->mPublishPassword );
    } else {
      // we use something else
      targetURL = KOPrefs::instance()->mPublishAnyURL.replace( "%SERVER%", emailHost );
      targetURL.setUser( KOPrefs::instance()->mPublishUserName );
      targetURL.setPass( KOPrefs::instance()->mPublishPassword );
    }

    KURL src;
    src.setPath( tempFile.name() );
    KIO::Job * job = KIO::file_copy( src, targetURL, -1,
                                     true /*overwrite*/,
                                     false /*don't resume*/,
                                     false /*don't show progress info*/ );
    connect( job, SIGNAL( result(KIO::Job *) ),
             this, SLOT( slotUploadFreeBusyResult(KIO::Job *) ) );
  }
}

void KOGroupware::slotUploadFreeBusyResult(KIO::Job *_job)
{
    KIO::FileCopyJob* job = static_cast<KIO::FileCopyJob *>(_job);
    if ( job->error() )
        KMessageBox::sorry( 0,
          i18n( "<qt>The software could not upload your free/busy list to the URL %1. There might be a problem with the access rights, or you specified an incorrect URL. The system said: <em>%2</em>.<br>Please check the URL or contact your system administrator.</qt>" ).arg( job->destURL().prettyURL() ).arg( job->errorString() ) );
    // Delete temp file
    KURL src = job->srcURL();
    Q_ASSERT( src.isLocalFile() );
    if( src.isLocalFile() )
        QFile::remove(src.path());
    mUploadingFreeBusy = false;
}

FBDownloadJob::FBDownloadJob( const QString& email, const KURL& url, KOGroupware* kogroupware, const char* name )
  : QObject( kogroupware, name ), mKogroupware(kogroupware), mEmail( email )
{
  KIO::Job* job = KIO::get( url, false, false );
  connect( job, SIGNAL( result( KIO::Job* ) ),
	   this, SLOT( slotResult( KIO::Job* ) ) );
  connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
	   this, SLOT( slotData( KIO::Job*, const QByteArray& ) ) );
}

FBDownloadJob::~FBDownloadJob()
{
  // empty for now
}

void FBDownloadJob::slotData( KIO::Job*, const QByteArray& data)
{
  QByteArray tmp = data;
  tmp.resize( tmp.size() + 1 );
  tmp[tmp.size()-1] = 0;
  mFBData += tmp;
}

void FBDownloadJob::slotResult( KIO::Job* job )
{
  if( job->error() ) {
    kdDebug(5850) << "FBDownloadJob::slotResult() job error :-(" << endl;
  }

  FreeBusy* fb = mKogroupware->parseFreeBusy( mFBData );
  emit fbDownloaded( mEmail, fb );
  // PENDING(steffen): Is this safe?
  //job->deleteLater();
  delete this;
}

bool KOGroupware::downloadFreeBusyData( const QString& email, QObject* receiver, const char* member )
{
  // Don't do anything with free/busy if the user does not want it.
  if( !KOPrefs::instance()->mRetrieveFreeBusy )
    return false;

  // Sanity check: Don't download if it's not a correct email
  // address (this also avoids downloading for "(empty email)").
  int emailpos = email.find( '@' );
  if( emailpos == -1 )
    return false;

  // Cut off everything left of the @ sign to get the user name.
  QString emailName = email.left( emailpos );
  QString emailHost = email.mid( emailpos + 1 );

  // Put download string together
  KURL sourceURL;
  if( KOPrefs::instance()->mRetrieveKolab ) {
    // we use Kolab
    QString server;
    if( KOPrefs::instance()->mRetrieveKolabServer == "%SERVER%" ||
	KOPrefs::instance()->mRetrieveKolabServer.isEmpty() )
      server = emailHost;
    else
      server = KOPrefs::instance()->mRetrieveKolabServer;

    sourceURL.setProtocol( "webdavs" );
    sourceURL.setHost( server );
    sourceURL.setPass( KOPrefs::instance()->mRetrievePassword );
    sourceURL.setUser( KOPrefs::instance()->mRetrieveUserName );
    sourceURL.setPath( QString::fromLatin1( "/freebusy/" ) + emailName +
		       QString::fromLatin1( ".vfb" ) );
  } else {
    // we use something else
    QString anyurl = KOPrefs::instance()->mRetrieveAnyURL;
    if( anyurl.contains( "%SERVER%" ) )
      anyurl.replace( "%SERVER%", emailHost );
    sourceURL = anyurl;
  }

  FBDownloadJob* job = new FBDownloadJob( email, sourceURL, this, "fb_download_job" );
  connect( job, SIGNAL( fbDownloaded( const QString&, FreeBusy*) ),
	   receiver, member );

  return true;
}

KCal::FreeBusy* KOGroupware::parseFreeBusy( const QCString& data )
{
  KCal::FreeBusy* fb = 0;
  QString freeBusyVCal = QString::fromUtf8(data);
  KCal::ScheduleMessage *message = mFormat.parseScheduleMessage( mCalendar,
								 freeBusyVCal );
  if( message ) {
    KCal::IncidenceBase* event = message->event();
    Q_ASSERT( event );

    if( event ) {
      // Enter the answer into the calendar. We just create a
      // Scheduler, because all the code we need is
      // already there. We take a MailScheduler, because
      // we need a concrete one, but we really only want
      // code from Scheduler.
      KCal::MailScheduler scheduler( mCalendar );
      scheduler.acceptTransaction( event,
				   (KCal::Scheduler::Method)message->method(),
				   message->status() );
      fb = dynamic_cast<KCal::FreeBusy*>( event );
      Q_ASSERT( fb );
    }
  }
  return fb;
}

/*!
  Store the calendar data into filename for the purpose of syncing
*/
void KOGroupware::pullSyncData( const QString& filename )
{
  // Store back all unsaved data into calendar object
  mView->viewManager()->currentView()->flushView();

  // In Groupware mode, we now for sure that we have an CalendarIMAP
  // object. But we use a dynamic_cast to guard against all possibilities
  // anyway.
  CalendarIMAP* mCalendarIMAP = dynamic_cast<CalendarIMAP*>( mCalendar );
  if( mCalendarIMAP )
    mCalendarIMAP->save( filename );
}


/*!
  Reload the calendar data from filename after syncing.
*/
void KOGroupware::pushSyncData( const QString& filename )
{
  // In Groupware mode, we know for sure that we have an CalendarIMAP
  // object. But we use a dynamic_cast to guard all possibilities
  // anyway.
  CalendarIMAP* mCalendarIMAP = dynamic_cast<CalendarIMAP*>( mCalendar );
  if( mCalendarIMAP )
    mCalendarIMAP->loadAndDelete( filename  );
}

/* This function sends mails if necessary, and makes sure the user really
 * want to change his calendar.
 *
 * Return true means accept the changes
 * Return false means revert the changes
 */
bool KOGroupware::sendICalMessage( QWidget* parent,
				   KCal::Scheduler::Method method,
				   Incidence* incidence, bool isDeleting )
{
  bool isOrganizer = KOPrefs::instance()->email() == incidence->organizer();

  int rc = 0;
  if( isOrganizer ) {
    // Figure out if there are other people involved in this incidence
    bool otherPeople = false;
    QPtrList<Attendee> attendees = incidence->attendees();
    for( Attendee* attendee = attendees.first(); attendee; attendee = attendees.next() ) {
      // Don't send email to ourselves
      if( attendee->email() != KOPrefs::instance()->email() ) {
	otherPeople = true;
	break;
      }
    }
    // You never send something out if no others are involved
    if( !otherPeople ) {
      if( incidence->summary().isEmpty() )
	// Fix empty summary
	incidence->setSummary( i18n("<No summary given>") );
      return true;
    }

    QString type;
    if( incidence->type() == "Event") type = i18n("event");
    else if( incidence->type() == "Todo" ) type = i18n("task");
    else if( incidence->type() == "Journal" ) type = i18n("journal entry");
    else type = incidence->type();
    QString txt = i18n("This %1 includes other people. "
		       "Should email be sent out to the attendees?").arg(type);
    rc = KMessageBox::questionYesNoCancel( parent, txt, i18n("Group scheduling email") );
  } else if( incidence->type() == "Todo" ) {
    if( method == Scheduler::Request )
      // This is an update to be sent to the organizer
      method = Scheduler::Reply;

    // Ask if the user wants to tell the organizer about the current status
    QString txt = i18n("Do you want to send a status update to the organizer of this task?");
    rc = KMessageBox::questionYesNo( parent, txt );
  } else if( incidence->type() == "Event" ) {
    // When you're not the organizer of an event, an update mail can never be sent out
    QString txt;
    if( !isDeleting ) {
      txt = i18n("<p>You are not the organizer of this event. "
		 "Editing it will bring your calendar out of sync "
		 "with the organizers calendar.</p>"
		 "<p>Do you really want to edit it?</p>");
      return KMessageBox::questionYesNo( parent, txt ) == KMessageBox::Yes;
    }

    txt = i18n("<p>You are deleting an appointment you have accepted.</p>"
	       "<p>Do you want to notify the organizer that you are "
	       "cancelling the appointment?</p>");
    rc = KMessageBox::questionYesNo( parent, txt );
    if( rc == KMessageBox::Yes ) {
      KCal::Event* event = static_cast<KCal::Event*>(incidence);
      QPtrList<KCal::Attendee> attendees = event->attendees();
      QPtrListIterator<KCal::Attendee> attendeeIt( attendees );
      KCal::Attendee* myself = 0, *current = 0;;
      // Find myself, there will always be all attendees listed, even if
      // only I need to answer it.
      while( ( current = attendeeIt.current() ) != 0 ) {
	++attendeeIt;

	if( current->email() == KOPrefs::instance()->email() ) {
	  // If we are the current one, note that. Still continue to
	  // search in case we find the receiver himself.
	  myself = current;
	  break;
	}
      }

      if( !myself ) {
	return true;
      }

      myself->setStatus( KCal::Attendee::Declined );
      KCal::Attendee* newMyself = new KCal::Attendee( myself->name(),
						      myself->email(),
						      myself->RSVP(),
						      myself->status(),
						      myself->role(),
						      myself->uid() );

      // Send back the answer; construct it on the base of state. We
      // make a clone of the event since we need to manipulate it here.
      Incidence* newIncidence = event->clone();
      Event* newEvent = static_cast<KCal::Event*>( newIncidence );
      newEvent->clearAttendees();
      newEvent->addAttendee( newMyself );

      // Create the outgoing vCal
      QString messageText = mFormat.createScheduleMessage( newEvent,
							   KCal::Scheduler::Reply );
      QString methodStr = KCal::Scheduler::methodName( KCal::Scheduler::Reply );
      QString subject = i18n("Cancelling an appointment");
      QStringList receiver; receiver << event->organizer();
      emit doSendICalMessage( methodStr, subject, receiver, messageText );
      return true;
    }

    txt = i18n("<p>You are not the organizer of this event. "
	       "Deleting it will bring your calendar out of sync "
	       "with the organizers calendar.</p>"
	       "<p>Do you really want to delete it?</p>");
    return KMessageBox::questionYesNo( parent, txt ) == KMessageBox::Yes;
  } else {
    qFatal( "Some unimplemented thing happened" );
  }

  if( rc == KMessageBox::Yes ) {
    // We will be sending out a message here. Now make sure there is some summary
    if( incidence->summary().isEmpty() )
      incidence->setSummary( i18n("<No summary given>") );

    // Send the mail
    static_cast<CalendarIMAP*>( mCalendar )->sendICalMessage( method, incidence );

    return true;
  } else if( rc == KMessageBox::No )
    return true;
  else
    return false;
}


void KOGroupware::slotSyncRunning( const QString& type, bool running )
{
  if( type == "Event" || type == "Calendar" )
    mCanEditEvents = !running;
  else if( type == "Task" || type == "Todo" ) {
    mCanEditTodos = !running;
    if( running && mView && mView->todoList() )
      // Invalidate category selection menu
      mView->todoList()->setCategorySelectionValidity( false );
  } else if( type == "Note" ) {
    emit notesSync(running);
  }
}


void KOGroupware::editingEvent( KOEventEditor* editor )
{
  connect( editor, SIGNAL( closingEventDialog( KOEventEditor* ) ),
	   this, SLOT( slotClosingEventDialog( KOEventEditor* ) ) );
  if( mEventDialogCount++ == 0 )
    emit dialogsUp( "Event", true );
}

void KOGroupware::slotClosingEventDialog( KOEventEditor* editor )
{
  editor->disconnect( this );
  if( --mEventDialogCount == 0 )
    emit dialogsUp( "Event", false );
}

void KOGroupware::editingTodo( KOTodoEditor* editor )
{
  connect( editor, SIGNAL( closingTodoDialog( KOTodoEditor* ) ),
	   this, SLOT( slotClosingTodoDialog( KOTodoEditor* ) ) );
  if( mTodoDialogCount++ == 0 )
    emit dialogsUp( "Todo", true );
}

void KOGroupware::slotClosingTodoDialog( KOTodoEditor* editor )
{
  editor->disconnect( this );
  if( --mTodoDialogCount == 0 )
    emit dialogsUp( "Todo", false );
}
