/***************************************************************************
 *                                                                         *
 *   SPDX-FileCopyrightText: 2015 Sebastian Kügler <sebas@kde.org>                       *
 *                                                                         *
 *   SPDX-License-Identifier: GPL-2.0-or-later
 ***************************************************************************/

#include "themesettings.h"
#include "themelistmodel.h"

#include <QDBusConnection>
#include <QDBusMessage>

#include <KAboutData>
#include <KLocalizedString>
#include <KPluginFactory>

#include <Plasma/Theme>

K_PLUGIN_FACTORY_WITH_JSON(ThemeSettingsFactory, "themesettings.json", registerPlugin<ThemeSettings>();)

ThemeSettings::ThemeSettings(QObject *parent, const QVariantList &args)
    : KQuickAddons::ConfigModule(parent, args)
{
    KAboutData *about = new KAboutData("kcm_mobile_theme", i18n("Appearance"), "1.0", QString(), KAboutLicense::LGPL);
    about->addAuthor(i18n("Sebastian Kügler"), QString(), "sebas@kde.org");
    setAboutData(about);
    setButtons(Apply | Default);

    qDebug() << "ThemeSettings module loaded.";

    qmlRegisterType<ThemeListModel>();
    m_themeListModel = new ThemeListModel(this);
    m_theme = new Plasma::Theme(this);
    m_theme->setUseGlobalSettings(true);
    m_themeName = m_theme->themeName();

    qDebug() << "Current theme: " << m_themeName;

    auto kdeglobals = KSharedConfig::openConfig("kdeglobals");
    KConfigGroup cfg(kdeglobals, "General");
    m_font = cfg.readEntry("font", QFont());
    qDebug() << "Font: " << m_font << m_font.pointSize();
}

ThemeSettings::~ThemeSettings() = default;

void ThemeSettings::setThemeName(const QString &theme)
{
    if (theme != m_themeName) {
        m_themeName = theme;

        /// ...
        m_theme->setThemeName(theme);
        emit themeNameChanged();
    }
}

QString ThemeSettings::themeName() const
{
    return m_themeName;
}

ThemeListModel *ThemeSettings::themeListModel()
{
    return m_themeListModel;
}

int ThemeSettings::fontSize() const
{
    return m_font.pointSize();
}

void ThemeSettings::setFontSize(int pointSize)
{
    m_font.setPointSize(pointSize);
    qDebug() << "Font size set to : " << m_font.pointSize();

    auto kdeglobals = KSharedConfig::openConfig("kdeglobals");
    KConfigGroup cfg(kdeglobals, "General");
    cfg.writeEntry("font", m_font);

    kdeglobals->sync();

    QDBusMessage message = QDBusMessage::createSignal("/KDEPlatformTheme", "org.kde.KDEPlatformTheme", "refreshFonts");
    QDBusConnection::sessionBus().send(message);

    // qApp->processEvents(); // Process font change ourselves
}

#include "themesettings.moc"
