/***************************************************************************
                          utils.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef UTILS_H
#define UTILS_H

#include <qstring.h>

class KURL;

/**
  * Utility class. Provides static utility functions in a namespace
  * given by the class.
  * @author Werner Stille
  */
class Utils
{
public:
  /**
   * Checks a double value for validity.
   * @param x double value.
   * @param logx true for use as argument of log function.
   * @return true for valid.
   */
  static bool valid(double x, bool logx = false);
  /**
   * Calculates minimum and maximum values of an array.
   * @param xmin pointer to minimum value.
   * @param xmax pointer to maximum value.
   * @param x double array.
   * @param n size of array.
   * @return true for error.
   */
  static bool minMaxFile(double* xmin, double* xmax, const double* x, int n);
  /**
   * Calculates scaling parameters.
   * @param xmi pointer to double for rounded minimum value.
   * @param xma pointer to double for rounded maximum value.
   * @param tic pointer to double for major tic distance.
   * @param mtic pointer to int for intervals between major tics.
   * @param ndig pointer to int for number of places after decimal point
   * for lettering.
   * @param xmin minimum value.
   * @param xmax maximum value.
   * @param fx normalization factor.
   * @param c scaling constant, desired value for tic / (xma - xmi).
   * @param log true for logarithmic axis.
   */
  static void autoSc(double* xmi, double* xma, double* tic, int* mtic,
                     int* ndig, double xmin, double xmax, double fx, double c,
                     bool log);

  /**
   * Calculates normalization factor.
   * @param a absolute double value.
   * @param ie pointer to int for logarithm of normalization factor.
   * @param f pointer to double for normalization factor.
   */
  static void expo(double a, int* ie, double* f);
  /**
   * Converts double to integer.
   * @param a double value.
   * @return integer result.
   */
  static int int9(double a);
  /**
   * Constructs relative file path (if possible).
   * @param url URL of plot file.
   * @param path file URL.
   * @param abs return absolute path if true.
   * @return relative path (if possible).
   */
  static QString relPath(const KURL& url, const KURL& path, bool abs);
  /**
   * Saves widget size in configuration.
   * @param w widget.
   * @param groupName name of the configuration group.
   */
  static void saveSize(QWidget* w, const QString& groupName);
  /**
   * Saves widget geometry in configuration.
   * @param w widget.
   * @param groupName name of the configuration group.
   */
  static void saveGeometry(QWidget* w, const QString& groupName);
  /**
   * Reads and sets widget size from configuration.
   * @param w widget.
   * @param groupName name of the configuration group.
   */
  static void setSize(QWidget* w, const QString& groupName);
  /**
   * Reads and sets widget geometry from configuration.
   * @param w widget.
   * @param groupName name of the configuration group.
   */
  static void setGeometry(QWidget* w, const QString& groupName);
  /**
   * Reads bounding box coordinates from an EPS file.
   * @param url URL of EPS file.
   * @param filename path (returned).
   * @param x1 left x value.
   * @param y1 bottom y value.
   * @param x2 right x value.
   * @param y2 top y value.
   */
  static bool boundingBox(const KURL& url, QString& filename, double* x1,
                          double* y1, double* x2, double* y2);
  /**
   * Translates exponent and decimal symbols.
   * @param line line from data or parameter file.
   * @param idec 0 for decimal point, 1 for comma.
   */
  static void translate(QString& line, int idec);
};

#endif
