/***************************************************************************
                          arrayitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Aug 28 1999
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qurl.h>
#include <qregexp.h>
#include <kapp.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include "arrayitem.h"
#include "arraydlg.h"
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "utils.h"

ArrayItem::ArrayItem() : internal(false), ix(0), iy(0), ie(0), istart(0), n(0),
 errbars(0), nrows(0), ncols(0), idec(0), x(0)
{
}

ArrayItem::ArrayItem(const ArrayItem& a) :
 ScaledItem(a), internal(a.internal), ix(a.ix), iy(a.iy), ie(a.ie),
 istart(a.istart), n(a.n), errbars(a.errbars), nrows(a.nrows), ncols(a.ncols),
 idec(a.idec), x(a.x), url(a.url)
{
  x = ArrayItem::copyX(a.x, a.ncols, a.nrows);
}

ArrayItem::ArrayItem(Kpl::AutoStruct* aut) : ScaledItem(aut), internal(false),
 ix(aut->ixAuto), iy(aut->iyAuto), ie(aut->ieAuto), istart(0), n(0),
 errbars(aut->autoErr), nrows(0), ncols(0), idec(aut->iDec), x(0)
{
}

ArrayItem::ArrayItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                     const KURL& uPlo) : ScaledItem(plo, aut),
 ix(0), iy(0), ie(0), istart(0), n(0), errbars(0), x(0)
{
  QString s = plo->readEntry("path", "");
  url = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  url.cleanPath();
  internal = plo->readBoolEntry("internal");
  if (internal) {
    n = nrows = plo->readNumEntry("n");
    errbars = plo->readNumEntry("errbars");
    ncols = 3;
    idec = 0;
    double* x1 = new double[ncols * nrows];
    x = new double*[ncols];
    for (int j = 0; j < ncols; j++)
      x[j] = &x1[j * nrows];
    for (int i = 0; i < nrows; i++) {
      QString s;
      QStringList list = plo->readListEntry(s.sprintf("r%i", i), ' ');
      int cnt = list.count();
      for (int j = 0; j < ncols; j++)
        x[j][i] = (j < cnt) ? list[j].toDouble() : 0.0;
    }
  } else {
    idec = plo->readNumEntry("idec");
    nrows = readFile(url, &ncols, &x, idec);
    if (!nrows) {
      KMessageBox::error(0, url.url());
        return;
    }
    istart = QMIN(plo->readNumEntry("istart"), nrows - 1);
    int nmax =  nrows - istart;
    n = QMIN(plo->readNumEntry("n", nmax), nmax);
    errbars = plo->readNumEntry("errbars", aut->autoErr);
  }
  int icolmax = ncols - 1;
  ix = QMIN(plo->readNumEntry("ix", aut->ixAuto), icolmax);
  iy = QMIN(plo->readNumEntry("iy", aut->iyAuto), icolmax);
  ie = QMIN(plo->readNumEntry("ie", aut->ieAuto), icolmax);
}

ArrayItem::ArrayItem(bool act, int fill, int sym, const QString& col,
                     double xn, double yn, int i_x, int i_y, int i_e,
                     int i_start, int np, int errb, const KURL& u,
                     bool _int, double relSize, int iDec) :
 ScaledItem(act, fill, sym, col, xn, yn, relSize), internal(_int), ix(i_x),
 iy(i_y), ie(i_e), istart(0), n(0), errbars(errb), idec(iDec), x(0), url(u)
{
  if (internal) {
    n = nrows = np;
    ncols = 3;
    idec = 0;
    int dim = ncols * nrows;
    double* x1 = new double[dim];
    memset(x1, 0, dim * sizeof(double));
    x = new double*[ncols];
    for (int j = 0; j < ncols; j++)
      x[j] = &x1[j * nrows];
  } else {
    nrows = readFile(url, &ncols, &x, idec);
    if (!nrows) {
      KMessageBox::error(0, url.url());
      return;
    }
    istart = QMIN(i_start, nrows - 1);
    n = QMIN(np, nrows - istart);
  }
}

ArrayItem::~ArrayItem()
{
  freeX(&x);
}

ArrayItem& ArrayItem::operator=(const ArrayItem& a)
{
  if (this != &a) {
    *(ScaledItem*)this = a;
    internal = a.internal;
    ix = a.ix;
    iy = a.iy;
    ie = a.ie;
    istart = a.istart;
    n = a.n;
    errbars = a.errbars;
    nrows = a.nrows;
    ncols = a.ncols;
    idec = a.idec;
    url = a.url;
    freeX(&x);
    x = ArrayItem::copyX(a.x, a.ncols, a.nrows);
  }
  return *this;
}

double** ArrayItem::copyX(double **xv, int nc, int nr)
{
  double** x2 = 0;
  if (xv) {
    int n = nc * nr;
    double* x1 = new double[n];
    memcpy(x1, xv[0], n * sizeof(double));
    x2 = new double*[nc];
    for (int i = 0; i < nc; i++)
      x2[i] = &x1[i * nr];
  }
  return x2;
}

void ArrayItem::freeX(double ***xv)
{
  if (*xv) {
    delete [] *xv[0];
    delete [] *xv;
    *xv = 0;
  }
}

int ArrayItem::nColumns(QTextStream& t, int idec)
{
  int nc = 0;
  while (!t.eof()) {
    QString line = t.readLine().stripWhiteSpace();
    if ((!line.isEmpty()) && (line[0] != '#')) {
      nc = QStringList::split(QRegExp(idec ? "[ \t]" : "[ ,\t]"),
                              line).count();
      break;
    }
  }
  return nc;
}

int ArrayItem::readLines(QTextStream& t, int nc, double ***xv, int idec)
{
  QArray<double> x2;
  int np = 0;
  int k = 0;
  int nr = 0;
  while (!t.eof()) {
    QString line = t.readLine().stripWhiteSpace();
    if ((!line.isEmpty()) && (line[0] != '#')) {
      Utils::translate(line, idec);
      QStringList list = QStringList::split(QRegExp(idec ? "[ \t]" : "[ ,\t]"),
                                            line);
      nr++;
      np += nc;
      x2.resize(np);
      int cnt = list.count();
      for (int i = 0; i < nc; i++) {
        x2[k] = (i < cnt) ? list[i].toDouble() : 0.0;
        k++;
      }
    }
  }
  double* x1 = new double[np];
  *xv = new double*[nc];
  for (int i = 0; i < nc; i++)
    (*xv)[i] = &x1[i * nr];
  k = 0;
  for (int j = 0; j < nr; j++)
    for (int i = 0; i < nc; i++)
      (*xv)[i][j] = x2[k++];
  return nr;
}

int ArrayItem::readFile(const KURL& u, int *nc, double ***xv, int idec)
{
  freeX(xv);
  int nr = 0;
  *nc = 0;
  QString fn;
  if (u.isLocalFile())
    fn = u.path();
  else
#if KDE_VERSION >= 0x03015a
    if (!KIO::NetAccess::download(u, fn, 0))
#else
    if (!KIO::NetAccess::download(u, fn))
#endif
      return nr;
  QFile f(fn);
  if (f.open(IO_ReadOnly)) {
    QTextStream t(&f);
    *nc = nColumns(t, idec);
    if (*nc) {
      f.reset();
      nr = readLines(t, *nc, xv, idec);
    }
    f.close();
  }
  if (!u.isLocalFile())
    KIO::NetAccess::removeTempFile(fn);
  return nr;
}

int ArrayItem::readFile(QString* text, int *nc, double ***xv)
{
  QTextStream t1(text, IO_ReadOnly);
  int nr = 0;
  *nc = nColumns(t1);
  if (*nc) {
    QTextStream t2(text, IO_ReadOnly);
    nr = readLines(t2, *nc, xv);
  }
  return nr;
}

KplItem::ItemTypes ArrayItem::iType() const
{
  return Array;
}

void ArrayItem::draw(KplGraph* g)
{
  if (x && active) {
    setProperties(g);
    double sav = g->relSize();
    g->setRelSize(relsiz * sav);
    g->plArray(&x[ix][istart], &x[iy][istart], fx, fy, n, fillStyle);
    if (errbars)
      g->plError(&x[ix][istart], &x[iy][istart], &x[ie][istart], fx, fy, n);
    g->setRelSize(sav);
  }
}

void ArrayItem::writePlo(KSimpleConfig* plo, const KURL& uPlo, bool _abs,
                         KplDoc* m) const
{
  plo->writeEntry("Type", "ARRAYITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("internal", internal);
  plo->writeEntry("ix", ix);
  plo->writeEntry("iy", iy);
  plo->writeEntry("ie", ie);
  plo->writeEntry("errbars", errbars);
  plo->writeEntry("n", n);
  if (internal) {
    plo->writeEntry("istart", 0);
    for (int i = 0; i < n; i++) {
      QStrList list;
      for (int j = 0; j < ncols; j++)
        list.insert(j, QString::number(x[j][i + istart]));
      QString s;
      plo->writeEntry(s.sprintf("r%i", i), list, ' ');
    }
  } else {
    plo->writeEntry("idec", idec);
    plo->writeEntry("istart", istart);
    plo->writeEntry("path", Utils::relPath(uPlo, url, _abs));
  }
}

void ArrayItem::setText(KplCheckListItem* it, bool* arrays, bool*) const
{
  *arrays = true;
  it->setText(1, i18n("Array"));
  QString s = (internal ? i18n("internal data") :
                         (i18n("file") + " " + url.fileName())) +
              ", " + i18n("columns") + " " +
              QString::number(ix) + ", " + QString::number(iy);
  if (errbars)
    s += ", " + QString::number(ie);
  it->setText(2, s);
}

int ArrayItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ArrayDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* ArrayItem::copy() const
{
  return new ArrayItem(*this);
}

void ArrayItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (x) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ArrayItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (x) {
    Utils::minMaxFile(xmi, xma, &x[ix][istart], n);
    Utils::minMaxFile(ymi, yma, &x[iy][istart], n);
  }
}
