/***************************************************************************
                          psgraph.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Wed Sep 08 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qglobal.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <math.h>
#include "psgraph.h"

const double PSGraph::pts = 28.34645669;

PSGraph::PSGraph() : x0(0.0), xl(0.0), y0(0.0), yl(0.0), xmin(0.0), xmax(0.0),
 ymin(0.0), ymax(0.0), ts(0)
{
  devType = PS;
}

PSGraph::~PSGraph()
{
}

void PSGraph::psInit(QTextStream* t, double xf, double yf)
{
  ts = t;
  logx = logy = false;
  kSymb = 0;
  relSize = 1.0;
  xmin = ymin = x0 = y0 = 0.0;
  xmax = xl = 21.0;
  ymax = yl = 29.7;
  *ts << "%!PS-Adobe-3.0 EPSF-3.0\n"
    "%%BoundingBox: 0 0 "
    << QString::number(qRound(xf * pts)) << " "
    << QString::number(qRound(yf * pts)) << "              \n"
    "%%Creator: Kpl\n"
    "%%CreationDate: " << QDateTime::currentDateTime().toString() << "\n"
    "%%Pages: 1\n\n"
    "%User code starts at line 392\n\n"
    << "/xmin 0 def\n"
    "/ymin 0 def\n"
    "/xmax 21 def\n"
    "/ymax 29.7 def\n"
    "/scalx 28.346456 def\n"
    "/xscal scalx def\n"
    "/yscal xscal def\n"
    "/minx 0 def\n"
    "/miny 0 def\n"
    "/maxx xscal xmax mul def\n"
    "/maxy yscal ymax mul def\n"
    "/xcur 0 def\n"
    "/ycur 0 def\n"
    "/colframe 0 def\n"
    "/colgrid 0 def\n"
    "/coldata 0 def\n"
    "/dir 0 def\n"
    "/symbol 0 def\n"
    "/dpos 0 def\n"
    "/relSize 1 def\n"
    "/ndigx -1 def\n"
    "/ndigy -1 def\n"
    "/logx false def\n"
    "/logy false def\n\n"
    << "/ed {exch def} bind def\n\n"
    << "/encfont {\n"
    " findfont dup length dict begin {\n"
    "  1 index dup /FID eq exch /Encoding eq or {pop pop} {def} ifelse\n"
    " } forall /Encoding ISOLatin1Encoding def currentdict end definefont\n"
    "} bind def\n\n"
    << "/strcat {\n"
    " /s2 ed /s1 ed s1 length dup s2 length add string /s3 ed s3 exch\n"
    " s2 putinterval s3 0 s1 putinterval s3\n"
    "} bind def\n\n"
    << "/r2r {ymin sub yscal mul miny add exch\n"
    " xmin sub xscal mul minx add exch\n"
    "} bind def\n\n"
    << "/mv {newpath r2r moveto} bind def\n\n"
    << "/rect {\n /y2 ed /x2 ed /y1 ed /x1 ed\n"
    " newpath x1 y1 moveto x2 y1 lineto x2 y2 lineto x1 y2 lineto closepath\n"
    "} bind def\n\n"
    << "/showc {\n dup stringwidth exch 2 div neg exch rmoveto currentpoint\n"
    " 3 2 roll show moveto\n"
    "} bind def\n\n"
    << "/showr {\n dup stringwidth exch neg exch rmoveto currentpoint\n"
    " 3 2 roll show moveto\n"
    "} bind def\n\n"
    << "/rel {maxx minx sub mul relSize mul} bind def\n\n"
    << "/slw {rel setlinewidth} bind def\n\n"
    << "/scf {rel scalefont setfont} bind def\n\n"
    << "/mrk {\n /sh 0.01 rel def sh mul exch sh mul exch rmoveto {\n"
    "  sh mul exch sh mul exch rlineto\n } repeat\n} bind def\n\n"
    << "/m0 {newpath r2r 0.01 rel 0 360 arc} bind def\n\n"
    "/m1 {0.004 slw mv 0 0 rlineto stroke 0.00295 slw} bind def\n\n"
    "/m2 {mv -1 1 1 1 1 -1 -1 -1 4 0 1 mrk} bind def\n\n"
    "/m3 {mv 0 2 2 0 0 -2 -2 0 4 1 1 mrk} bind def\n\n"
    "/m4 {mv -1 1.732 2 0 -1 -1.732 3 0 0.866 mrk} bind def\n\n"
    "/m5 {mv 2 2 -2 0 2 -2 -2 0 4 1 1 mrk} bind def\n\n"
    "/m6 {mv 0 1 0.5 0.5 1 0 0.5 -0.5 0 -1 -0.5 -0.5 -1 0 -0.5"
    " 0.5 8 1 0.5 mrk} bind def\n\n"
    "/m7 {mv 2 -2 -1 1 1 1 -2 -2 4 1 1 mrk} bind def\n\n"
    "/m8 {mv 2 0 -1 0 0 1 0 -2 4 0 1 mrk} bind def\n\n"
    "/m9 {mv -1 1 -1 -1 1 1 0 0.866 4 0 -0.866 mrk} bind def\n\n"
    "/m10 {mv 1 1 -2 0 1 -1 0 2 4 0 -1 mrk} bind def\n\n"
    << "/lintyp {\n 1 sub dup 0 lt {pop 0} if 9 mod dup\n"
    " [8 7 2 1 9 8 7 4 1] exch get /p0 ed\n"
    " [8 0 0 0 0 1 2 4 1] exch get /p1 ed\n"
    " /f 0.003125 rel def 4 array dup 0 p0 f mul put dup 1 16 p0 sub p1 sub\n"
    " f mul p1 0 eq {put dup dup 0 get 2 exch} {0.5 mul put dup 2 p1 f mul}\n"
    " ifelse put dup dup 1 get 3 exch put dpos setdash\n"
    "} bind def\n\n"
    << "/err {\n fy mul yscal mul /dy ed fy mul /y ed fx mul /x ed logy\n"
    " {y 0 gt {/dy dy y div 0.4343 mul def} {/dy 0 def} ifelse} if\n"
    " x 0 gt logx not or dy sh ge and {\n"
    "  x logx {log} if y logy {log} if mv 0 sh rmoveto 0 dy sh sub\n"
    "  rlineto 0.5 sh mul 0 rmoveto sh neg 0 rlineto 0 -2 dy mul rmoveto sh\n"
    "  0 rlineto -0.5 sh mul 0 rmoveto 0 dy sh sub rlineto stroke\n"
    " } if\n"
    "} bind def\n\n"
    << "/format {\n /ymax ed /xmax ed /xmin 0 def /ymin 0 def\n"
    " /xscal scalx def /yscal xscal def /minx 0 def /miny 0 def /maxx xscal\n"
    " xmax mul def /maxy yscal ymax mul def /xcur 0 def /ycur 0 def\n"
    "} bind def\n\n"
    << "/window {\n /ymax ed /ymin ed /xmax ed /xmin ed\n"
    " /xmax xmax xmin add def /ymax ymax ymin add def\n"
    " /minx xscal xmin mul def /miny yscal ymin mul def\n"
    " /maxx xscal xmax mul def /maxy yscal ymax mul def\n"
    " /xcur 0 def /ycur 0 def 0.00295 slw\n"
    " /latin1 findfont 0.033333 scf\n"
    "} bind def\n\n"
    << "/psscale {\n /logy ed /logx ed /ymax ed /ymin ed /xmax ed /xmin ed\n"
    " logx {/xmin xmin log def /xmax xmax log def} if\n"
    " logy {/ymin ymin log def /ymax ymax log def} if\n"
    " /xscal maxx minx sub xmax xmin sub div def\n"
    " /yscal maxy miny sub ymax ymin sub div def\n"
    " /xcur xmin def /ycur ymin def\n"
    "} bind def\n\n"
    << "/scol {\n"
    " /col ed col 16#ff and 16#ff div col -8 bitshift 16#ff and 16#ff div\n"
    " col -16 bitshift 16#ff div setrgbcolor\n"
    "} bind def\n\n"
    << "/frame {minx miny maxx maxy rect stroke} bind def\n\n"
    "/psclip {\n"
    " /d 0.00208 rel def minx d add miny d add maxx d sub maxy d sub\n"
    " rect clip\n"
    "} bind def\n\n"
    << "/prenum {\n"
    " /ndig ed /v ed v 0 lt {/v v neg def (-)} {()} ifelse /v ndig 0 gt\n"
    " {10 ndig exp dup v mul round dup cvi /iv ed exch div}\n"
    " {v round dup cvi /iv ed} ifelse def /ie v 10 lt {0} {v log cvi}\n"
    " ifelse def /iend ndig 0 gt {ndig neg} {0} ifelse def\n"
    " /id 10 ie iend sub exp round cvi def {\n"
    "  iv dup id mod 10 mul /iv ed id idiv 256 string cvs\n"
    "  strcat ie 0 eq {ndig 0 ge {(.) strcat} if} if ie iend eq\n"
    "  {exit} if /ie ie 1 sub def\n"
    " } loop\n"
    "} bind def\n\n"
    << "/incLog {\n"
    " log /lgf ed /xlg ed /mant xlg dup dup abs 1.0e-5 mul add cvi sub def\n"
    " mant abs 1.0e-6 lt {/mant 0 def} if mant 0 lt {/mant mant 1 add def}\n"
    " if xlg mant 0.1 lt {lgf add} {mant 0.4 lt {2.5 log add}\n"
    "  {mant 0.6 lt {10 3 div log add} {2 log add} ifelse} ifelse} ifelse\n"
    "} bind def\n\n"
    << "/raster {\n"
    " /yRight ed /yLeft ed /xTop ed /xBottom ed /yOff ed /xOff ed /mode ed\n"
    " /inty ed /intx ed /ytic ed /xtic ed mode -1 ne {mode 0 ne {\n"
    "  logx {xtic 2.5 lt {/xtic 2 def} {xtic 5.5 lt {/xtic 3 def}\n"
    "   {xtic 99 lt {/xtic 10 def} if} ifelse} ifelse} if\n"
    "  logy {ytic 2.5 lt {/ytic 2 def} {ytic 5.5 lt {/ytic 3 def}\n"
    "   {ytic 99 lt {/ytic 10 def} if} ifelse} ifelse} if\n"
    "  colgrid scol 0.00208 slw /dx maxx minx sub def /dy maxy miny sub def\n"
    "  mode 2 le {/dx dx 0.02 mul def /dy dx def} if\n"
    "  mode 2 le intx 1 gt and xOff 0 gt and logx not and {\n"
    "   /d xtic intx div def /xm xmin xOff add d sub def {xm xmin le {exit} if\n"
    "    xBottom {xm ymin mv 0 dy 0.5 mul rlineto stroke} if\n"
    "    xTop {xm ymax mv 0 dy -0.5 mul rlineto stroke} if\n"
    "    /xm xm d sub def} loop} if\n"
    "  /x xmin xOff add def {logx\n"
    "   {/xa x def /x x xtic incLog def\n"
    "    x xmax lt {xBottom {x ymin mv 0 dy rlineto stroke} if\n"
    "    xTop {x ymax mv 0 dy neg rlineto stroke} if} if intx 1 gt {\n"
    "     /xd 10 xa mant sub exp def {/xa 10 xa exp xd add log def\n"
    "      xa x ge xa xmax ge or {exit} if\n"
    "      xBottom {xa ymin mv 0 dy mode 2 le {0.5 mul} if rlineto stroke} if\n"
    "      xTop {xa ymax mv 0 dy neg mode 2 le {0.5 mul} if rlineto stroke} if\n"
    "     } loop\n"
    "    } if} {mode 2 le intx 1 gt and {\n"
    "     /xm x def intx 1 sub {\n"
    "      /xm xtic intx div xm add def xm xmax ge {exit} if\n"
    "      xBottom {xm ymin mv 0 dy 0.5 mul rlineto stroke} if\n"
    "      xTop {xm ymax mv 0 dy -0.5 mul rlineto stroke} if\n"
    "     } repeat\n"
    "    } if x xmin gt {\n"
    "     xBottom {x ymin mv 0 dy rlineto stroke} if\n"
    "     xTop {x ymax mv 0 dy neg rlineto stroke} if\n"
    "    } if /x x xtic add def\n"
    "   } ifelse x xmax ge {exit} if} loop\n"
    "  mode 2 le inty 1 gt and yOff 0 gt and logy not and {\n"
    "   /d ytic inty div def /ym ymin yOff add d sub def {\n"
    "    ym ymin le {exit} if\n"
    "    yLeft {xmin ym mv dx 0.5 mul 0 rlineto stroke} if\n"
    "    yRight {xmax ym mv dx -0.5 mul 0 rlineto stroke} if\n"
    "    /ym ym d sub def} loop} if\n"
    "   /y ymin yOff add def {logy\n"
    "   {/ya y def /y y ytic incLog def\n"
    "    y ymax lt {yLeft {xmin y mv dx 0 rlineto stroke} if\n"
    "    yRight {xmax y mv dx neg 0 rlineto stroke} if} if inty 1 gt {\n"
    "     /yd 10 ya mant sub exp def {/ya 10 ya exp yd add log def\n"
    "      ya y ge ya ymax ge or {exit} if\n"
    "      yLeft {xmin ya mv dx mode 2 le {0.5 mul} if 0 rlineto stroke} if\n"
    "      yRight {xmax ya mv dx neg mode 2 le {0.5 mul} if 0 rlineto stroke} if\n"
    "     } loop\n"
    "    } if} {mode 2 le inty 1 gt and {\n"
    "     /ym y def inty 1 sub {\n"
    "      /ym ytic inty div ym add def ym ymax ge {exit} if\n"
    "      yLeft {xmin ym mv dx 0.5 mul 0 rlineto stroke} if\n"
    "      yRight {xmax ym mv dx -0.5 mul 0 rlineto stroke} if\n"
    "     } repeat} if y ymin gt {\n"
    "     yLeft {xmin y mv dx 0 rlineto stroke} if\n"
    "     yRight {xmax y mv dx neg 0 rlineto stroke} if\n"
    "    } if /y y ytic add def\n"
    "   } ifelse y ymax ge {exit} if} loop\n"
    "  mode 2 eq mode 4 eq or {\n"
    "   xBottom {/yB ymin xmax xmin sub relSize mul xscal mul yscal div\n"
    "    0.05 mul sub def} if\n"
    "   xTop {/yT ymax xmax xmin sub relSize mul xscal mul yscal div\n"
    "    0.025 mul neg sub def} if\n"
    "   /e xmax 2.99 gt xmin -2.99 lt or xtic 9.9 gt and\n"
    "   logx and def /nd ndigx def /x xmin xOff add def /xe xmax xmin sub\n"
    "   1e-3 mul xmax add def {\n"
    "    x xmax gt {/x xmax def} if e {(10#nu) x round cvi 4 string cvs strcat\n"
    "     xBottom {dup x yB r2r 2 text} if\n"
    "     xTop {x yT r2r 2 text} {pop} ifelse}\n"
    "     {logx {x 0 lt {/nd 1 x 0.01 add cvi sub def} {/nd -1 def} ifelse}\n"
    "     if x logx {10 exch exp} if nd prenum colframe scol\n"
    "     xBottom {dup x yB mv showc} if xTop {x yT mv showc} {pop} ifelse}\n"
    "    ifelse logx {/x x xtic incLog def} {/x x xtic add def} ifelse\n"
    "    x xe gt {exit} if\n"
    "   } loop\n"
    "   yLeft {/xL xmin xmax xmin sub relSize mul 0.025 mul sub def} if\n"
    "   yRight {/xR xmax xmax xmin sub relSize mul -0.025 mul sub def} if\n"
    "   /dy xmax xmin sub 0.0125 mul relSize mul xscal mul yscal div def\n"
    "   /e ymax 2.99 gt ymin -2.99 lt or ytic 9.9 gt and logy and def\n"
    "   /nd ndigy def /y ymin yOff add def /ye ymax ymin sub 1e-3 mul\n"
    "   ymax add def {\n"

    "    y ymax gt {/y ymax def} if\n"
    "    e {(10#nu) y round cvi 4 string cvs strcat\n"
    "     yLeft {dup xL y dy sub r2r 3 text} if\n"
    "     yRight {xR y dy sub r2r 1 text} {pop} ifelse}\n"
    "     {logy {y 0 lt {/nd 1 y 0.01 add cvi sub def} {/nd -1 def} ifelse}\n"
    "     if y logy {10 exch exp} if nd prenum colframe scol\n"
    "     yLeft {dup xL y dy sub mv showr} if\n"
    "     yRight {xR y dy sub mv show} {pop} ifelse}\n"
    "    ifelse\n"
    "    logy {/y y ytic incLog def} {/y y ytic add def} ifelse\n"
    "    y ye gt {exit} if\n"
    "   } loop\n"
    "  } if\n"
    " } if colframe scol 0.00417 slw frame coldata scol 0.00295 slw} if\n"
    "} bind def\n\n"
    << "/text {\n"
    " /iorg ed gsave translate dir rotate 0 0 moveto /str ed\n"
    " /latin1 findfont 0.033333 scf colframe scol str (#) search {\n"
    "  pop pop pop /dx 0 def iorg 1 eq {1} {0} ifelse 1 1 {\n"
    "   /j ed /ilev 0 def /s str def\n"
    "   j 1 eq {dx iorg 2 eq {0.5 mul} if neg 0 rmoveto} if {\n"
    "    s (#) search exch /s1 ed {pop /c ed} {/c 0 string def} ifelse\n"
    "    j 0 eq {s1 stringwidth pop dx add /dx ed} {\n"
    "     ilev 0 eq {s1 show} {\n"
    "      ilev 1 eq {0.015} {-0.01} ifelse\n"
    "      rel dup 0 exch rmoveto s1 show neg 0 exch rmoveto\n"
    "     } ifelse\n"
    "    } ifelse c length 2 lt {exit} {\n"
    "     c 0 get dup 96 gt {32 sub} if 1 string dup 0 4 -1 roll put\n"
    "     (NHBOSZI) exch search {\n"
    "      length [\n"
    "       /latin1 /latin1 /l1B /l1O /Symbol /ZapfDingbats /l1I\n"
    "      ] exch get findfont c 1 get dup 96 gt {32 sub} if 1 string dup\n"
    "      0 4 -1 roll put (NUD) exch search {length /ilev ed pop} if\n"
    "      pop ilev 0 eq {0.033333} {0.022222} ifelse scf pop\n"
    "     } if pop /s c 2 c length 2 sub getinterval def\n"
    "    } ifelse s length 0 eq {exit} if\n"
    "   } loop /latin1 findfont 0.033333 scf\n"
    "  } for\n"
    " } {iorg 1 eq {show} {iorg 2 eq {showc} {showr} ifelse} ifelse} ifelse\n"
    " grestore coldata scol\n"
    "} bind def\n\n"
    << "/arrow {\n"
    " colframe scol gsave scalx mul exch scalx mul exch translate newpath\n"
    " 0 0 moveto rotate -0.03333 rel dup dup\n"
    " 0.2 mul dup neg 3 1 roll lineto lineto closepath fill scalx mul dup\n"
    " 0.03333 rel ge {\n"
    "  0.00417 slw neg 0 moveto -0.03333 rel 0 lineto closepath stroke\n"
    "  0.00295 slw\n"
    " } {pop} ifelse grestore coldata scol} bind def\n\n"
    << "/arccm {\n"
    " gsave scalx mul exch scalx mul exch translate rotate /h ed h div\n"
    " /a ed /wl ed /wa ed 6 array currentmatrix a 1 scale newpath 0 0 h\n"
    " 0.5 mul scalx mul wa wl wa add arc setmatrix stroke grestore\n"
    " } bind def\n\n"
    << "/rectcm {\n"
    " gsave scalx mul exch scalx mul exch translate rotate scalx mul exch\n"
    " scalx mul exch 0 0 rect {fill} {stroke} ifelse grestore\n"
    " } bind def\n\n"
    << "/ellcm {\n"
    " gsave scalx mul exch scalx mul exch translate rotate /h ed h div\n"
    " /a ed 6 array currentmatrix a 1 scale newpath 0 0 h 0.5 mul\n"
    " scalx mul 0 360 arc setmatrix {fill} {stroke} ifelse grestore\n"
    " } bind def\n\n"
    << "/letH {\n"
    " {0.16975} {0.025} ifelse rel maxy add minx maxx add 0.5 mul exch\n"
    " 2 text\n"
    "} bind def\n\n"
    << "/letXy {\n"
    " /iax ed /opposite ed /iex ed /str ed str length iex or 0 ne {\n"
    "  iex 0 eq {str} {(10#nu) iex 4 string cvs strcat ( #nn) strcat str\n"
    "   strcat} ifelse iax 2 eq {/l 3 logy {ymax 2.99 gt ymin -2.99 lt\n"
    "   or {ymin 0 lt {4} {3} ifelse} {ymax 1.99 ymin sub 2 copy lt\n"
    "   {exch} if pop cvi 1 add} ifelse} {ymin ndigy prenum length\n"
    "   ymax ndigy prenum length 2 copy lt {exch} if pop} ifelse add def\n"
    "   opposite {maxx} {minx} ifelse l 0.01875 rel mul opposite\n"
    "   {neg} if sub miny maxy add 0.5 mul /dir 90 opposite {neg} if def\n"
    "  } {minx maxx add 0.5 mul opposite {maxy -0.085} {miny 0.11} ifelse\n"
    "  rel sub} ifelse 2 text\n"
    "  /dir 0 def\n"
    " } if\n"
    "} bind def\n\n"
    << "/letX {1 letXy} bind def\n\n"
    << "/letY {2 letXy} bind def\n\n"
    << "/plarry {\n"
    " gsave {psclip} if 3 1 roll /fy ed /fx ed symbol 0 gt {\n"
    "  symbol lintyp 3 1 roll fy mul /ya ed fx mul /xa ed /p1 false def\n"
    "  xa 0 gt logx not or ya 0 gt logy not or and {\n"
    "   xa logx {log} if ya logy {log} if newpath r2r /y ed /x ed\n"
    "   x y moveto /p1 true def\n"
    "  } if 1 sub {\n"
    "   fy mul /ya ed fx mul /xa ed\n"
    "   xa 0 gt logx not or ya 0 gt logy not or and {\n"
    "    xa logx {log} if ya logy {log} if r2r dup y sub  dup mul\n"
    "    2 index x sub dup mul add sqrt dpos add /dpos ed dup /y ed\n"
    "    /x 2 index def p1 {lineto} {moveto /p1 true def} ifelse\n"
    "   } if\n"
    "  } repeat stroke /dpos dpos dup 0.05 rel div floor 0.05 rel mul\n"
    "  sub def [] dpos setdash\n"
    " } {\n"
    "  /m [/m0 /m1 /m2 /m3 /m4 /m5 /m6 /m7 /m8 /m9 /m10 /m0 /m2 /m3 /m4\n"
    "  /m5 /m6] symbol neg 17 mod get load def {\n"
    "   symbol -17 eq {pop pop} {fy mul /ya ed fx mul /xa ed xa 0 gt\n"
    "   logx not or ya 0 gt logy not or and {xa logx {log} if ya logy\n"
    "   {log} if m symbol neg 17 mod 10 le {stroke} {fill} ifelse} if}"
    " ifelse\n  } repeat\n"
    " } ifelse grestore\n"
    "} bind def\n\n"
    << "/plerry {\n"
    " gsave psclip 3 1 roll /fy ed /fx ed /sh 0.01 rel def {err} repeat"
    " grestore\n} bind def\n\n"
    << "%%Page: 1 1\n\n"
    "gsave\n"
    "/latin1 /Helvetica encfont findfont 0.033333 scf\n"
    "/l1B /Helvetica-Bold encfont pop\n"
    "/l1O /Helvetica-Oblique encfont pop\n"
    "/l1I /Helvetica-BoldOblique encfont pop\n"
    "1 setlinecap 1 setlinejoin\n\n"
    "0.00295 slw\n\n%User code starts here\n\n";
}

void PSGraph::format(double x, double y)
{
  QCString s;
  *ts << s.sprintf("%.2f %.2f format\n", x, y);
}

double PSGraph::setRelSize(double r)
{
  double old = relSize;
  relSize = r;
  QCString s;
  *ts << s.sprintf("/relSize %.2f def 0.00295 slw\n", r);
  return old;
}

void PSGraph::Window(double x0a, double xla, double y0a, double yla)
{
  x0 = x0a;
  xl = xla;
  y0 = y0a;
  yl = yla;
  xmin = xmax = 0;
  xmax = xl;
  ymax = yl;
  QCString s;
  *ts << s.sprintf("%.2f %.2f %.2f %.2f window\n", x0, xl, y0, yl);
}

void PSGraph::scale(double _xl, double xr, double yb, double yt,
                    bool lgx, bool lgy)
{
  logx = lgx;
  logy = lgy;
  if (logx) {
    xmin = log10(_xl);
    xmax = log10(xr);
  } else {
    xmin = _xl;
    xmax = xr;
  }
  if (logy) {
    ymin = log10(yb);
    ymax = log10(yt);
  } else {
    ymin = yb;
    ymax = yt;
  }
  *ts << QString::number(_xl) << " " << QString::number(xr) << " "
      << QString::number(yb) << " " << QString::number(yt)
      << (logx ? " true" : " false") << (logy ? " true" : " false")
      << " psscale\n";
}

void PSGraph::setDig(int nx, int ny)
{
  *ts << "/ndigx " << QString::number(nx)
      << " def\n/ndigy " << QString::number(ny) << " def\n";
}

void PSGraph::setColFrame(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/colframe 16#%x def\n", rgbQt1(icol));
}

void PSGraph::setColGrid(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/colgrid 16#%x def\n", rgbQt1(icol));
}

void PSGraph::setColData(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/coldata 16#%x def\ncoldata scol\n",  rgbQt1(icol));
}

void PSGraph::setCol(unsigned icol1, unsigned icol2, unsigned icol3)
{
  setColFrame(icol1);
  setColGrid(icol2);
  setColData(icol3);
}

void PSGraph::frame(void)
{
  *ts << "frame\n";
}

void PSGraph::raster(double xtic, double ytic, int intx, int inty, int mode,
                     double xOff, double yOff, bool xBottom, bool xTop,
                     bool yLeft, bool yRight)
{
  *ts << QString::number(xtic) << " " << QString::number(ytic) << " "
      << QString::number(intx) << " " << QString::number(inty) << " "
      << QString::number(mode) << " " << QString::number(xOff) << " "
      << QString::number(yOff) << (xBottom ? " true" : " false")
      << (xTop ? " true" : " false") << (yLeft ? " true" : " false")
      << (yRight ? " true" : " false") << " raster\n";
}

void PSGraph::setDir(double ang)
{
  *ts << "/dir " << QString::number(ang) << " def\n";
}

void PSGraph::text(const QString& str, double x, double y, int iorg)
{
  psStr(str);
  *ts << " " << QString::number(x) << " " << QString::number(y) << " "
      << QString::number(iorg) << " text\n";
}

void PSGraph::textcm(const QString& str, double x, double y, int iorg, QRect*)
{
  text(str, pts * x, pts * y, iorg);
}

void PSGraph::arrow(double x, double y, double dir, double len, QRect*)
{
  *ts << QString::number(len) << " " << QString::number(dir) << " "
      << QString::number(x) << " " << QString::number(y) << " arrow\n";
}

void PSGraph::arc(double x, double y, double w, double h, double a,
                  double alen, double dir, QRect*)
{
  *ts << QString::number(a) << " " << QString::number(alen) << " "
      << QString::number(w) << " " << QString::number(h) << " "
      << QString::number(dir) << " " << QString::number(x) << " "
      << QString::number(y) << " arccm\n";
}

void PSGraph::rectangle(double x, double y, double w, double h,
                        bool filled, double dir, QRect*)
{
  *ts << (filled ? "true " : "false ")
      << QString::number(w) << " " << QString::number(h) << " "
      << QString::number(dir) << " " << QString::number(x) << " "
      << QString::number(y) << " rectcm\n";
}

void PSGraph::ellipse(double x, double y, double w, double h,
                     bool filled, double dir, QRect*)
{
  *ts << (filled ? "true " : "false ")
      << QString::number(w) << " " << QString::number(h) << " "
      << QString::number(dir) << " " << QString::number(x) << " "
      << QString::number(y) << " ellcm\n";
}

void PSGraph::letH(const QString& str, bool xTop)
{
  psStr(str);
  *ts << (xTop ? " true" : " false") << " letH\n";
}

void PSGraph::letX(const QString& str, int iex, bool xTop)
{
  psStr(str);
  *ts << " " << QString::number(iex) << (xTop ? " true" : " false")
      << " letX\n";
}

void PSGraph::letY(const QString& str, int iey, bool yRight)
{
  psStr(str);
  *ts << " " << QString::number(iey) << (yRight ? " true" : " false")
      << " letY\n";
}

void PSGraph::setSymbol(int is)
{
  kSymb = is;
  *ts <<"/symbol " << QString::number(is) << " def\n";
}

void PSGraph::plArray(const double* x, const double* y,
                      double fx, double fy, int n, bool clip)
{
  if (n > 0) {
    int ie = 0;
    int ia;
    bool xNorm = (fx > 1.0e33) || (fx < 1.0e-33);
    bool yNorm = (fy > 1.0e33) || (fy < 1.0e-33);
    do {
      ia = QMIN(n - 1, ie + 127);
      *ts << "\n";
      int fail = 0;
      QCString s;
      for (int i = ia; i >= ie; i--) {
        s.sprintf("%g %g\n", xNorm ? (fx * x[i]) : x[i],
                  yNorm ? (fy * y[i]) : y[i]);
        if (s.find("nan") != -1) {
          s.prepend("%%");
          fail++;
        }
        *ts << s;
      }
      int np = ia - ie - fail + 1;
      if (np)
        *ts << QString::number(xNorm ? 1.0 : fx) << " "
            << QString::number(yNorm ? 1.0 : fy) << " "
            << QString::number(np) << (clip ? " true" : " false")
            << " plarry\n";
      if (kSymb > 0)
        ie = ia;
      else
        ie = ia + 1;
    } while (ia != (n - 1));
  }
}

void PSGraph::plError(const double* x, const double* y, const double* e,
                      double fx, double fy, int n)
{
  if (n > 0) {
    int ie = 0;
    int ia;
    bool xNorm = (fx > 1.0e33) || (fx < 1.0e-33);
    bool yNorm = (fy > 1.0e33) || (fy < 1.0e-33);
    do {
      ia = QMIN(n - 1, ie + 127);
      *ts << "\n";
      int fail = 0;
      QCString s;
      for (int i = ia; i >= ie; i--) {
        s.sprintf("%g %g %g\n", xNorm ? (fx * x[i]) : x[i],
                  yNorm ? (fy * y[i]) : y[i],
                  yNorm ? (fy * e[i]) : e[i]);
        if (s.find("nan") != -1) {
          s.prepend("%%");
          fail++;
        }
        *ts << s;
      }
      *ts << QString::number(xNorm ? 1.0 : fx) << " "
          << QString::number(yNorm ? 1.0 : fy) << " "
          << QString::number(ia - ie - fail + 1) << " plerry\n";
      ie = ia + 1;
    } while (ia != (n - 1));
  }
}

void PSGraph::resetDash()
{
  *ts << "/dpos 0 def\n";
}

void PSGraph::psExit(void)
{
  *ts << "\nshowpage grestore\n\n%%Trailer\n%%EOF\n";
}

void PSGraph::psStr(const char* str)
{
  *ts << "(";
  QCString s;
  const unsigned char* c = (const unsigned char*) str;
  while (*c) {
    if (*c > 127)
      *ts << s.sprintf("\\%03o", *c);
    else
      switch (*c) {
        case 40:
        case 41:
        case 92:
          *ts << "\\";
        default:
          *ts << QChar(*c);
      }
    c++;
  }
  *ts << ")";
}

void PSGraph::cm2r(double xcm, double ycm, double _fxn, double _fyn,
                   double* xr, double* yr) const
{
  *xr = xmin + (xcm - x0) * _fxn * (xmax - xmin) / xl;
  if (logx)
    *xr = pow(10, *xr);
  *yr = ymin + (ycm - y0) * _fyn * (ymax - ymin) / yl;
  if (logy)
    *yr = pow(10, *yr);
}
