// other_imp.cc
// Copyright (C)  2003  Dominique Devriese <devriese@kde.org>

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.

#include "other_imp.h"

#include "bogus_imp.h"
#include "point_imp.h"
#include "line_imp.h"

#include "../misc/screeninfo.h"
#include "../misc/common.h"
#include "../misc/kigtransform.h"
#include "../misc/kigpainter.h"
#include "../misc/i18n.h"
#include "../kig/kig_view.h"

#include <cmath>
#include <utility>
using namespace std;

AngleImp::~AngleImp()
{
}

ObjectImp* AngleImp::transform( const Transformation& ) const
{
  // TODO ?
  return new InvalidImp;
}

void AngleImp::draw( KigPainter& p ) const
{
  p.drawAngle( mpoint, mstartangle, mangle );
}

AngleImp::AngleImp( const Coordinate& pt, double start_angle_in_radials,
                    double angle_in_radials )
  : mpoint( pt ), mstartangle( start_angle_in_radials ),
    mangle( angle_in_radials )
{
}

bool AngleImp::contains( const Coordinate& p, int width, const KigWidget& w ) const
{
  double radius = 50*w.screenInfo().pixelWidth();

  if ( fabs( (p-mpoint).length() - radius ) > w.screenInfo().normalMiss( width ) )
    return false;

  // and next we check if the angle is appropriate...
  Coordinate vect = p - mpoint;
  double angle = atan2( vect.y, vect.x );
  while ( angle < mstartangle ) angle += 2*M_PI;
  return angle <= mstartangle + mangle;
}

bool AngleImp::inRect( const Rect& r, int width, const KigWidget& w ) const
{
  // TODO ?
  return r.contains( mpoint, w.screenInfo().normalMiss( width ) );
}

const uint AngleImp::numberOfProperties() const
{
  return Parent::numberOfProperties() + 3;
}

const QCStringList AngleImp::propertiesInternalNames() const
{
  QCStringList l = Parent::propertiesInternalNames();
  l << I18N_NOOP( "angle-radian" );
  l << I18N_NOOP( "angle-degrees" );
  l << I18N_NOOP( "angle-bisector" );
  assert( l.size() == AngleImp::numberOfProperties() );
  return l;
}

const QCStringList AngleImp::properties() const
{
  QCStringList l = Parent::properties();
  l << I18N_NOOP( "Angle in Radians" );
  l << I18N_NOOP( "Angle in Degrees" );
  l << I18N_NOOP( "Angle Bisector" );
  assert( l.size() == AngleImp::numberOfProperties() );
  return l;
}

const ObjectImpType* AngleImp::impRequirementForProperty( uint which ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::impRequirementForProperty( which );
  else return AngleImp::stype();
}

const char* AngleImp::iconForProperty( uint which ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::iconForProperty( which );
  if ( which == Parent::numberOfProperties() )
    return "angle-size"; // size in radians
  else if ( which == Parent::numberOfProperties() + 1 )
    return "angle-size"; // size in degrees
  else if ( which == Parent::numberOfProperties() + 2 )
    return "angle-bisector"; // angle bisector..
  else assert( false );
  return "";
}

ObjectImp* AngleImp::property( uint which, const KigDocument& w ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::property( which, w );
  if ( which == Parent::numberOfProperties() )
    return new DoubleImp( size() );
  else if ( which == Parent::numberOfProperties() + 1 )
    return new DoubleImp( size() * 180 / M_PI );
  else if ( which == Parent::numberOfProperties() + 2 )
  {
    const double angle = mstartangle + mangle / 2;
    Coordinate p2 = mpoint + Coordinate( cos( angle ), sin( angle ) ) * 10;
    return new RayImp( mpoint, p2 );
  }
  else assert( false );
  return new InvalidImp;
}

const double AngleImp::size() const
{
  return mangle;
}

ObjectImp* AngleImp::copy() const
{
  return new AngleImp( mpoint, mstartangle, mangle );
}

VectorImp::VectorImp( const Coordinate& a, const Coordinate& b )
  : ma( a ), mb( b )
{
}

VectorImp::~VectorImp()
{
}

ObjectImp* VectorImp::transform( const Transformation& t ) const
{
  Coordinate ta = t.apply( ma );
  Coordinate tb = t.apply( mb );
  if ( ta.valid() && tb.valid() ) return new VectorImp( ta, tb );
  else return new InvalidImp;
}

void VectorImp::draw( KigPainter& p ) const
{
  p.drawVector( ma, mb );
}

bool VectorImp::contains( const Coordinate& o, int width, const KigWidget& w ) const
{
  return isOnSegment( o, ma, mb, w.screenInfo().normalMiss( width ) );
}

bool VectorImp::inRect( const Rect& r, int width, const KigWidget& w ) const
{
  return lineInRect( r, ma, mb, width, this, w );
}

const uint VectorImp::numberOfProperties() const
{
  return Parent::numberOfProperties();
}

const QCStringList VectorImp::propertiesInternalNames() const
{
  return Parent::propertiesInternalNames();
}

const QCStringList VectorImp::properties() const
{
  return Parent::properties();
}

const ObjectImpType* VectorImp::impRequirementForProperty( uint which ) const
{
  return Parent::impRequirementForProperty( which );
}

const char* VectorImp::iconForProperty( uint which ) const
{
  return Parent::iconForProperty( which );
}

ObjectImp* VectorImp::property( uint which, const KigDocument& w ) const
{
  return Parent::property( which, w );
}

ObjectImp* VectorImp::copy() const
{
  return new VectorImp( ma, mb );
}

const Coordinate VectorImp::dir() const
{
  return mb - ma;
}

void AngleImp::visit( ObjectImpVisitor* vtor ) const
{
  vtor->visit( this );
}

void VectorImp::visit( ObjectImpVisitor* vtor ) const
{
  vtor->visit( this );
}

ArcImp::ArcImp( const Coordinate& center, const double radius,
                const double startangle, const double angle )
  : CurveImp(), mcenter( center ), mradius( radius ),
    msa( startangle ), ma( angle )
{
  if ( ma < 0 )
  {
    // we want a positive angle..
    msa = msa + ma;
    ma = -ma;
  };
}

ArcImp::~ArcImp()
{
}

ArcImp* ArcImp::copy() const
{
  return new ArcImp( mcenter, mradius, msa, ma );
}

ObjectImp* ArcImp::transform( const Transformation& t ) const
{
  //
  // we don't have conic arcs! So it is invalid to transform an arc
  // with a nonhomothetic transformation
  //
  if ( ! t.isHomothetic() ) return new InvalidImp();

  Coordinate nc = t.apply( mcenter );
  double nr = t.apply( mradius );
  // transform msa...
  double nmsa = msa;
  if ( t.getAffineDeterminant() > 0 )
  {
    nmsa = msa - t.getRotationAngle();
  } else
  {
    Coordinate ar = t.apply2by2only( Coordinate( cos(msa), sin(msa) ) );
    nmsa = atan2( ar.y, ar.x );
    nmsa -= ma;
  }
  while ( nmsa < -M_PI ) nmsa += 2*M_PI;
  while ( nmsa > M_PI ) nmsa -= 2*M_PI;
  if ( nc.valid() ) return new ArcImp( nc, nr, nmsa, ma );
  else return new InvalidImp;
}

void ArcImp::draw( KigPainter& p ) const
{
  p.drawArc( mcenter, mradius, msa, ma );
}

bool ArcImp::contains( const Coordinate& p, int width, const KigWidget& w ) const
{
  if( fabs( (mcenter - p).length() - mradius ) > w.screenInfo().normalMiss( width ) )
    return false;
  Coordinate d = p - mcenter;
  double angle = atan2( d.y, d.x );

  if ( angle < msa ) angle += 2 * M_PI;
  return angle - msa - ma < 1e-4;
}

bool ArcImp::inRect( const Rect&, int, const KigWidget& ) const
{
  // TODO
  return false;
}

bool ArcImp::valid() const
{
  return true;
}

const uint ArcImp::numberOfProperties() const
{
  return Parent::numberOfProperties() + 6;
}

const QCStringList ArcImp::properties() const
{
  QCStringList ret = Parent::properties();
  ret << I18N_NOOP( "Center" );
  ret << I18N_NOOP( "Radius" );
  ret << I18N_NOOP( "Angle in Degrees" );
  ret << I18N_NOOP( "Angle in Radians" );
  ret << I18N_NOOP( "Sector Surface" );
  ret << I18N_NOOP( "Arc Length" );
  assert( ret.size() == ArcImp::numberOfProperties() );
  return ret;
}

const QCStringList ArcImp::propertiesInternalNames() const
{
  QCStringList ret = Parent::propertiesInternalNames();
  ret << "center";
  ret << "radius";
  ret << "angle-degrees";
  ret << "angle-radians";
  ret << "sector-surface";
  ret << "arc-length";
  return ret;
}

const char* ArcImp::iconForProperty( uint which ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::iconForProperty( which );
  else if ( which == Parent::numberOfProperties() )
    return "arc-center"; // center
  else if ( which == Parent::numberOfProperties() + 1 )
    return "";
  else if ( which == Parent::numberOfProperties() + 2 )
    return "angle-size";
  else if ( which == Parent::numberOfProperties() + 3 )
    return "angle-size";
  else if ( which == Parent::numberOfProperties() + 4 )
    return "";
  else if ( which == Parent::numberOfProperties() + 5 )
    return "";
  else assert( false );
  return "";
}

ObjectImp* ArcImp::property( uint which, const KigDocument& d ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::property( which, d );
  else if ( which == Parent::numberOfProperties() )
    return new PointImp( mcenter );
  else if ( which == Parent::numberOfProperties() + 1 )
    return new DoubleImp( mradius );
  else if ( which == Parent::numberOfProperties() + 2 )
    return new IntImp( static_cast<int>( ma * 180 / M_PI ) );
  else if ( which == Parent::numberOfProperties() + 3 )
    return new DoubleImp( ma );
  else if ( which == Parent::numberOfProperties() + 4 )
    return new DoubleImp( mradius * mradius * ma / 2 );
  else if ( which == Parent::numberOfProperties() + 5 )
    return new DoubleImp( mradius * ma );
  else assert( false );
  return new InvalidImp;
}

const ObjectImpType* ArcImp::impRequirementForProperty( uint which ) const
{
  if ( which < Parent::numberOfProperties() )
    return Parent::impRequirementForProperty( which );
  else
    return ArcImp::stype();
}

void ArcImp::visit( ObjectImpVisitor* vtor ) const
{
  vtor->visit( this );
}

double ArcImp::getParam( const Coordinate& c, const KigDocument& ) const
{
  Coordinate d = (c - mcenter).normalize();
  double angle = atan2( d.y, d.x );
  angle -= msa;
// mp: problems with large arcs
  while ( angle > ma/2 + M_PI ) angle -= 2*M_PI;
  while ( angle < ma/2 - M_PI ) angle += 2*M_PI;
//
  angle = max( 0., min( angle, ma ) );
  angle /= ma;
  return angle;
}

const Coordinate ArcImp::getPoint( double p, bool& valid, const KigDocument& ) const
{
  valid = true;
  double angle = msa + p * ma;
  Coordinate d = Coordinate( cos( angle ), sin( angle ) ) * mradius;
  return mcenter + d;
}

const Coordinate ArcImp::center() const
{
  return mcenter;
}

double ArcImp::radius() const
{
  return mradius;
}

double ArcImp::startAngle() const
{
  return msa;
}

double ArcImp::angle() const
{
  return ma;
}

const Coordinate VectorImp::a() const
{
  return ma;
}

const Coordinate VectorImp::b() const
{
  return mb;
}

bool ArcImp::equals( const ObjectImp& rhs ) const
{
  return rhs.inherits( ArcImp::stype() ) &&
    static_cast<const ArcImp&>( rhs ).radius() == radius() &&
    static_cast<const ArcImp&>( rhs ).startAngle() == startAngle() &&
    static_cast<const ArcImp&>( rhs ).angle() == angle();
}

bool AngleImp::equals( const ObjectImp& rhs ) const
{
  return rhs.inherits( AngleImp::stype() ) &&
    static_cast<const AngleImp&>( rhs ).point() == point() &&
    static_cast<const AngleImp&>( rhs ).startAngle() == startAngle() &&
    static_cast<const AngleImp&>( rhs ).angle() == angle();
}

bool VectorImp::equals( const ObjectImp& rhs ) const
{
  return rhs.inherits( VectorImp::stype() ) &&
    static_cast<const VectorImp&>( rhs ).a() == a() &&
    static_cast<const VectorImp&>( rhs ).b() == b();
}

const ObjectImpType* AngleImp::stype()
{
  static const ObjectImpType t(
    Parent::stype(), "angle",
    I18N_NOOP( "angle" ),
    I18N_NOOP( "Select this angle" ),
    I18N_NOOP( "Remove an Angle" ),
    I18N_NOOP( "Add an Angle" ),
    I18N_NOOP( "Move an Angle" ),
    I18N_NOOP( "Attach to this angle" )
    );
  return &t;
}
const ObjectImpType* VectorImp::stype()
{
  static const ObjectImpType t(
    Parent::stype(), "vector",
    I18N_NOOP( "vector" ),
    I18N_NOOP( "Select this vector" ),
    I18N_NOOP( "Remove a Vector" ),
    I18N_NOOP( "Add a Vector" ),
    I18N_NOOP( "Move a Vector" ),
    I18N_NOOP( "Attach to this vector" )
    );
  return &t;
}
const ObjectImpType* ArcImp::stype()
{
  static const ObjectImpType t(
    Parent::stype(), "arc",
    I18N_NOOP( "arc" ),
    I18N_NOOP( "Select this arc" ),
    I18N_NOOP( "Remove an Arc" ),
    I18N_NOOP( "Add an Arc" ),
    I18N_NOOP( "Move an Arc" ),
    I18N_NOOP( "Attach to this arc" )
    );
  return &t;
}

const ObjectImpType* AngleImp::type() const
{
  return AngleImp::stype();
}

const ObjectImpType* VectorImp::type() const
{
  return VectorImp::stype();
}

const ObjectImpType* ArcImp::type() const
{
  return ArcImp::stype();
}
