/* -*- C++ -*-
 * This file implements the main widget for kab II.
 * 
 * the KDE addressbook
 * copyright:  (C) Mirko Sucker, 1998 1999
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@unibw-hamburg.de>
 * requires:   recent C++-compiler, at least Qt 1.4, STL
 * $Revision: 1.3 $
 */

#include "kab.h"
#include <addressbook.h>
#include <kdatanavigator.h>
#include <kfiledialog.h>
#include <qmessagebox.h>
#include <qkeycode.h>
#include <qtimer.h>
#include <qfileinfo.h>
#include <kapp.h>
#include <functions.h>
#include "debug.h"

// #############################################################################
// MOC OUTPUT FILES:
#include "kab.moc"
// #############################################################################

KabMainWindow::KabMainWindow(const char* name=0)
  : KTMainWindow(name),
    kab(new AddressBook(this)),
    timer(new QTimer(this))
{
  // ###########################################################################
  // ----- do memory check:
  if(kab==0 || timer==0)
    {
      QMessageBox::critical
	(this, i18n("kab: Out of memory"),
	 i18n("Cannot initialize local variables."));
      kapp->quit(); // It is critical, but will possibly never happen. 
    }
  // -----
  createMenuBar();
  createToolBar();
  createStatusBar();
  createConnections();
  setView(kab);
  updateRects();
  updateGUI();
  // -----
  setStatus(i18n("KDE addressbook, version 2."));
  // ###########################################################################
}

KabMainWindow::~KabMainWindow()
{
  register bool GUARD; GUARD=true;
  // ###########################################################################
  LG(GUARD, "KabMainWindow::~KabMainWindow: called.\n");
  delete timer; timer=0;
  delete nav; nav=0; // make sure updateGUI does not use it after destruction
  delete kab; kab=0; // kab emits changed on destruction
  LG(GUARD, "KabMainWindow::~KabMainWindow: done.\n");
  // ###########################################################################
}

void KabMainWindow::createMenuBar()
{ // (consider this method to be part of the constructor)
  // ###########################################################################
  // ----- the "file" menu:
  file=new QPopupMenu;
  CHECK_PTR(file);
  newFileItemID=file->insertItem(i18n("&New File"), this, SLOT(newFile()),
				 CTRL+Key_N);
  openItemID=file->insertItem(i18n("&Open"), this, SLOT(open()),
			      CTRL+Key_O);
  saveItemID=file->insertItem(i18n("&Save"), this, SLOT(save()), CTRL+Key_S);
  QPopupMenu *exp=new QPopupMenu;
  exp->insertItem(i18n("HTML table"), this, SLOT(exportHTML()));
  exportItemID=file->insertItem(i18n("Export"), exp);
  file->insertSeparator();
  searchItemID=file->insertItem
    (i18n("Search &entries"), this, SLOT(search()), CTRL+Key_F);
  printItemID=file->insertItem(i18n("&Print"), this, SLOT(print()), CTRL+Key_P);
  file->insertSeparator();
  file->insertItem(i18n("&Quit"), kapp, SLOT(quit()), CTRL+Key_Q);
  // ----- the "edit" menu:
  edit=new QPopupMenu;
  CHECK_PTR(edit);
  copyItemID=edit->insertItem(i18n("&Copy"), this, SLOT(copy()), CTRL+Key_W);
  edit->insertSeparator();
  edit->insertItem(i18n("&Add entry"), this, SLOT(add()), CTRL+Key_N);
  editItemID=edit->insertItem
    (i18n("&Edit entry"), this, SLOT(editEntry()), CTRL+Key_E);
  removeItemID=edit->insertItem
    (i18n("&Remove entry"), this, SLOT(remove()), CTRL+Key_R);
  edit->insertSeparator();
  importVCardID=edit->insertItem
    (i18n("Import &vCard"), this, SLOT(importVCard()), CTRL+Key_V);
  edit->insertSeparator();
  mailItemID=edit->insertItem(i18n("&Mail"), this, SLOT(mail()), CTRL+Key_M);
  talkItemID=edit->insertItem(i18n("&Talk"), this, SLOT(talk()), CTRL+Key_T);
  browseItemID=edit->insertItem
    (i18n("&Browse"), this, SLOT(browse()), CTRL+Key_B);
  // ----- the "about" menu:
  QPopupMenu* about=new QPopupMenu;
  about->insertItem
    (i18n("Help on Addressbook"), kapp, SLOT(appHelpActivated()), Key_F1);
  about->insertSeparator();
  about->insertItem(i18n("About Addressbook"), this, SLOT(about()), CTRL+Key_A);
  about->insertItem(i18n("About &Qt"), this, SLOT(aboutQt()));
  about->insertItem(i18n("About &KDE"), kapp, SLOT(aboutKDE()));
  // ----- fill the menu:
  menuBar()->insertItem(i18n("&File"), file);
  menuBar()->insertItem(i18n("&Edit"), edit);
  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&About"), about);
  // ###########################################################################
}

void KabMainWindow::createStatusBar()
{ // (consider this method to be part of the constructor)
  // ###########################################################################
  statusBar()->setInsertOrder(KStatusBar::RightToLeft);
  statusBar()->insertItem("20000/20000", Number);
  statusBar()->insertItem("", Text);
  // ###########################################################################
}

void KabMainWindow::createToolBar()
{ // (consider this method to be part of the constructor)
  // ###########################################################################
  int id;
  // ----- create the navigation toolbar:
  nav=new KDataNavigator(toolBar());
  CHECK_PTR(nav);
  connect(nav, SIGNAL(newItem()), SLOT(add()));
  connect(nav, SIGNAL(itemSelected(int)), SLOT(entrySelected(int)));
  // ----- configure the navigation toolbar:
  nav->setFrameStyle(QFrame::NoFrame);
  nav->setMode(KDataNavigator::List);
  id=toolBar()->insertWidget(0, 100, nav);
  toolBar()->setItemAutoSized(id);
  // ###########################################################################
}

void KabMainWindow::createConnections()
{ // (consider this method to be part of the constructor)
  REQUIRE(kab!=0);
  // ###########################################################################
  connect(timer, SIGNAL(timeout()), SLOT(timeOut()));
  connect(kab, SIGNAL(setStatus(const char*)), SLOT(setStatus(const char*)));
  connect(kab, SIGNAL(changed()), SLOT(updateGUI()));
  connect(kab, SIGNAL(browse(const char*)), SLOT(browse(const char*)));
  connect(kab, SIGNAL(mail(const char*)), SLOT(mail(const char*)));
  // ###########################################################################
}

void KabMainWindow::updateGUI()
{
  REQUIRE(kab!=0);
  // ###########################################################################
  int current;
  int number;
  QString temp;
  QStrList strings;
  // -----
  if(nav==0 || kab==0)
    {
      L("KabMainWindow::updateGUI: in destruction, done.\n");
      return;
    }
  current=1; // WORK_TO_DO
  number=kab->noOfEntries();
  if(kab->getListOfNames(&strings)!=AddressBook::NoError)
    {
      L("KabMainWindow::updateGUI: error retrieving name list.\n");
    } else {
      CHECK(strings.count()==(unsigned)number);
      nav->setList(&strings);
    }
  // ----- show the last displayed or the first entry:
  if(kab->getState()==AddressBook::NoError)
    {
      if(kab->displayEntry(0, 0)!=AddressBook::NoError)
 	{ // possibly there are no entries, clear the display
	  AddressBook::Entry entry;
	  // -----
	  kab->displayEntry(entry, 0);
	}
    }
  temp.sprintf("%i/%i", current, number);
  statusBar()->changeItem(temp, Number);
  // ###########################################################################
}

void KabMainWindow::setStatus(const char* text)
{
  // ###########################################################################
  statusBar()->changeItem(text, Text);
  timer->start(8000, false);
  // ###########################################################################
}

void KabMainWindow::timeOut()
{
  // ###########################################################################
  statusBar()->changeItem("", Text);
  // ###########################################################################
}

void KabMainWindow::aboutQt()
{
  // ###########################################################################
  QMessageBox::aboutQt(this, i18n("About Qt"));
  // ###########################################################################
}

void KabMainWindow::about()
{
  // ###########################################################################
  L("KabMainWindow::about: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::saveOurselfes()
{
  // ###########################################################################
  L("KabMainWindow::saveOurSelfes: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::enableWidgets(int)
{
  // ###########################################################################
  L("KabMainWindow::enableWidgets: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::browse()
{
  // ###########################################################################
  L("KabMainWindow::browse: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::browse(const char*)
{
  // ###########################################################################
  browse();
  // ###########################################################################
}

void KabMainWindow::print()
{
  // ###########################################################################
  L("KabMainWindow::print: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::search()
{
  // ###########################################################################
  L("KabMainWindow::search: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::newFile()
{
  register bool GUARD; GUARD=true;
  // ###########################################################################
  LG(GUARD, "KabMainWindow::newFile: called.\n");
  QString home, dummy;
  QFileInfo info;
  // ----- select the filename:
  if(!getHomeDirectory(home))
    {
      QMessageBox::information
	(this, i18n("Sorry"), i18n("Could not find the users home directory."));
      emit(setStatus(i18n("Intern error!"))); 
      qApp->beep();
      return;
    }
  for(;;) // do forever
    {
      dummy=KFileDialog::getOpenFileName(home, "*kab", this);
      if(dummy.isEmpty()) // dialog has been cancelled
	{
	  emit(setStatus(i18n("Cancelled.")));
	  qApp->beep();
	  return;
	}
      info.setFile(dummy);
      if(info.isDir() || (info.exists() && !info.isWritable()))
	{
	  QMessageBox::information
	    (this, i18n("kab: File error"),
	     i18n("This file is a directory, or you do not have writing "
		  "permissions for it.\n"
		  "You must have permission to write to create a new file.\n"
		  "Please select another file name."));
	} else {
	  LG(GUARD, "KabMainWindow::newFile: filename is %s.\n", dummy.data());
	  break;
	}
    }
  // ----- adjust name:
  if(dummy.size()<4 || dummy.mid(dummy.size()-4, 4)!=".kab")
    {
      dummy+=".kab";
      QMessageBox::information
	(this, i18n("kab: Note"),
	 i18n("The filename did not end with \".kab\".\n"
	      "It has been changed to\n")+dummy);
    }
  // ----- create the new database file:
  if(kab->createNew(dummy)!=AddressBook::NoError)
    {
      QMessageBox::critical
	(this, i18n("kab: File error"),
	 i18n("The file could not be created."));
      qApp->beep();
      return;
    }
  // ----- load this file:
  if(kab->load(dummy)!=AddressBook::NoError)
    {
      QMessageBox::critical
	(this, i18n("kab: File error"),
	 i18n("The file has been created, but could not be loaded."));
      qApp->beep();
      return;
    }
  // -----
  updateGUI();
  LG(GUARD, "KabMainWindow::newFile: done.\n");  
  // ###########################################################################
}

void KabMainWindow::open()
{
  register bool GUARD; GUARD=true;
  // ###########################################################################
  LG(GUARD, "KabMainWindow::open: called.\n");
  QString home, dummy;
  QFileInfo info;
  // ----- select the filename:
  if(!getHomeDirectory(home))
    {
      QMessageBox::information
	(this, i18n("Sorry"), i18n("Could not find the users home directory."));
      emit(setStatus(i18n("Intern error!"))); 
      qApp->beep();
      return;
    }
  for(;;) // do forever
    {
      dummy=KFileDialog::getOpenFileName(home, "*kab", this);
      if(dummy.isEmpty()) // dialog has been cancelled
	{
	  emit(setStatus(i18n("Cancelled.")));
	  qApp->beep();
	  return;
	}
      info.setFile(dummy);
      if(info.isDir() || !info.exists())
	{
	  QMessageBox::information
	    (this, i18n("kab: File error"),
	     i18n("The file does not exist or is a directory.\n"
		  "Use \"New File\" to create a new one.\n"
		  "Select an existing file to load it."));
	} else {
	  LG(GUARD, "KabMainWindow::newFile: filename is %s.\n", dummy.data());
	  break;
	}
    }
  // ----- load the file:
  if(kab->load(dummy)!=AddressBook::NoError)
    {
      QMessageBox::critical
	(this, i18n("kab: File error"),
	 i18n("The file could not be loaded."));
      qApp->beep();
      return;
    }
  // -----
  updateGUI();
  LG(GUARD, "KabMainWindow::newFile: done.\n");  
  // ###########################################################################
}

void KabMainWindow::save()
{
  // ###########################################################################
  if(kab->save()!=AddressBook::NoError)
    {
      QMessageBox::information
	(this, i18n("kab: File error"),
	 i18n("The file could not be saved (permission denied)."));
    }
  // ###########################################################################
}

void KabMainWindow::copy()
{
  // ###########################################################################
  L("KabMainWindow::copy: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::add()
{
  // ###########################################################################
  L("KabMainWindow::add: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::editEntry()
{
  // ###########################################################################
  L("KabMainWindow::edit: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::remove()
{
  // ###########################################################################
  L("KabMainWindow::remove: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::importVCard()
{
  // ###########################################################################
  L("KabMainWindow::importVCard: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::mail()
{
  // ###########################################################################
  L("KabMainWindow::mail: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::mail(const char*)
{
  // ###########################################################################
  mail();
  // ###########################################################################
}

void KabMainWindow::talk()
{
  // ###########################################################################
  L("KabMainWindow::talk: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::exportHTML()
{
  // ###########################################################################
  L("KabMainWindow::exportHTML: called (not implemented).\n");
  // ###########################################################################
}

void KabMainWindow::entrySelected(int index)
{
  // ###########################################################################
  if(kab->displayEntry(index, 0)!=AddressBook::NoError)
    {
      CHECK(false);
    }
  // ###########################################################################
}

int main(int argc, char **argv)
{
  // ###########################################################################
  KApplication app(argc, argv, "kab");
  KabMainWindow *mw=new KabMainWindow;
  if(mw!=0)
    {
      app.setMainWidget(mw);
      mw->show();
      return app.exec();
    } else {
      return -1;
    }
  // ###########################################################################
}
