# -*- coding: utf-8 -*-

"""
Copyright (C) 2008-2012 Wolfgang Rohdewald <wolfgang@rohdewald.de>

kajongg is free software you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
"""

from common import Internal, Preferences, ZValues
from PyQt4.QtCore import QRectF, QPointF
from PyQt4.QtGui import QGraphicsSimpleTextItem

from board import PlayerWind, YellowText, Board, rotateCenter
from wall import Wall, KongBox
from tile import Tile
from uitile import UITile
from animation import animate, afterCurrentAnimationDo, Animated, \
    ParallelAnimationGroup

class UIWallSide(Board):
    """a Board representing a wall of tiles"""
    def __init__(self, tileset, boardRotation, length):
        Board.__init__(self, length, 1, tileset, boardRotation=boardRotation)
        self.length = length

    def name(self):
        """name for debug messages"""
        game = Internal.field.game
        if not game:
            return 'NOGAME'
        for player in game.players:
            if player.front == self:
                return 'wall %s'% player.name

    def center(self):
        """returns the center point of the wall in relation to the faces of the upper level"""
        faceRect = self.tileFaceRect()
        result = faceRect.topLeft() + self.shiftZ(1) + \
            QPointF(self.length // 2 * faceRect.width(), faceRect.height()/2)
        result.setX(result.x() + faceRect.height()/2) # corner tile
        return result

    def hide(self):
        """hide all my parts"""
        self.windTile.hide()
        self.nameLabel.hide()
        Board.hide(self)

class UIKongBox(KongBox):
    """Kong box with UITiles"""
    # pylint: disable=incomplete-protocol
    def __init__(self):
        KongBox.__init__(self)

    def fill(self, tiles):
        """fill the box"""
        for tile in self._tiles:
            tile.cross = False
        KongBox.fill(self, tiles)
        for tile in self._tiles:
            tile.cross = True

    def pop(self, count):
        """get count tiles from kong box"""
        result = KongBox.pop(self, count)
        for tile in result:
            tile.cross = False
        return result

class UIWall(Wall):
    """represents the wall with four sides. self.wall[] indexes them counter clockwise, 0..3. 0 is bottom."""
    tileClass = UITile
    kongBoxClass = UIKongBox

    def __init__(self, game):
        """init and position the wall"""
        # we use only white dragons for building the wall. We could actually
        # use any tile because the face is never shown anyway.
        game.wall = self
        Wall.__init__(self, game)
        self.__square = Board(1, 1, Internal.field.tileset)
        self.__square.setZValue(ZValues.marker)
        sideLength = len(self.tiles) // 8
        self.__sides = [UIWallSide(Internal.field.tileset, boardRotation, sideLength) \
            for boardRotation in (0, 270, 180, 90)]
        for side in self.__sides:
            side.setParentItem(self.__square)
            side.lightSource = self.lightSource
            side.windTile = PlayerWind('E', Internal.field.windTileset, parent=side)
            side.windTile.hide()
            side.nameLabel = QGraphicsSimpleTextItem('', side)
            font = side.nameLabel.font()
            font.setPointSize(48)
            side.nameLabel.setFont(font)
            side.message = YellowText(side)
            side.message.setZValue(ZValues.popup)
            side.message.setVisible(False)
            side.message.setPos(side.center())
        self.__sides[0].setPos(yWidth=sideLength)
        self.__sides[3].setPos(xHeight=1)
        self.__sides[2].setPos(xHeight=1, xWidth=sideLength, yHeight=1)
        self.__sides[1].setPos(xWidth=sideLength, yWidth=sideLength, yHeight=1 )
        self.showShadows = Preferences.showShadows
        Internal.field.centralScene.addItem(self.__square)

    @staticmethod
    def name():
        """name for debug messages"""
        return 'wall'

    def __getitem__(self, index):
        """make Wall index-able"""
        return self.__sides[index]

    def __setitem__(self, index, value):
        """only for pylint, currently not used"""
        self.__sides[index] = value

    def __delitem__(self, index):
        """only for pylint, currently not used"""
        del self.__sides[index]

    def __len__(self):
        """only for pylint, currently not used"""
        return len(self.__sides)

    def hide(self):
        """hide all four walls and their decorators"""
        self.living = []
        self.kongBox.fill([])
        for side in self.__sides:
            side.hide()
        self.tiles = []
        Internal.field.centralScene.removeItem(self.__square)

    def __shuffleTiles(self):
        """shuffle tiles for next hand"""
        discardBoard = Internal.field.discardBoard
        places = [(x, y) for x in range(-3, discardBoard.width+3) for y in range(-3, discardBoard.height+3)]
        places = self.game.randomGenerator.sample(places, len(self.tiles))
        for idx, tile in enumerate(self.tiles):
            assert isinstance(tile, UITile)
            tile.dark = True
            tile.setBoard(discardBoard, *places[idx])

    def build(self):
        """builds the wall without dividing"""
        # recycle used tiles
        for tile in self.tiles:
            tile.tile = Tile('Xy')
            tile.dark = True
#        field = Internal.field
#        animateBuild = not field.game.isScoringGame() and not self.game.isFirstHand()
        animateBuild = False
        with Animated(animateBuild):
            self.__shuffleTiles()
            for tile in self.tiles:
                tile.focusable = False
            return animate().addCallback(self.__placeWallTiles)

    def __placeWallTiles(self, dummyResult=None):
        """place all wall tiles"""
        tileIter = iter(self.tiles)
        tilesPerSide = len(self.tiles) // 4
        for side in (self.__sides[0], self.__sides[3], self.__sides[2], self.__sides[1]):
            upper = True # upper tile is played first
            for position in range(tilesPerSide-1, -1, -1):
                tile = tileIter.next()
                tile.setBoard(side, position//2, 0, level=int(upper))
                upper = not upper
        self.__setDrawingOrder()
        return animate()

    @property
    def lightSource(self):
        """For possible values see LIGHTSOURCES"""
        return self.__square.lightSource

    @lightSource.setter
    def lightSource(self, lightSource):
        """setting this actually changes the visuals"""
        if self.lightSource != lightSource:
            assert ParallelAnimationGroup.current is None
            self.__square.lightSource = lightSource
            for side in self.__sides:
                side.lightSource = lightSource
            self.__setDrawingOrder()

    @property
    def tileset(self):
        """The tileset of this wall"""
        return self.__square.tileset

    @tileset.setter
    def tileset(self, value):
        """setting this actually changes the visuals."""
        if self.tileset != value:
            assert ParallelAnimationGroup.current is None
            self.__square.tileset = value
            for side in self.__sides:
                side.tileset = value
            self.__resizeHandBoards()

    @property
    def showShadows(self):
        """The current value"""
        return self.__square.showShadows

    @showShadows.setter
    def showShadows(self, showShadows):
        """setting this actually changes the visuals."""
        if self.showShadows != showShadows:
            assert ParallelAnimationGroup.current is None
            self.__square.showShadows = showShadows
            for side in self.__sides:
                side.showShadows = showShadows
            self.__resizeHandBoards()

    def __resizeHandBoards(self, dummyResults=None):
        """we are really calling _setRect() too often. But at least it works"""
        for player in self.game.players:
            player.handBoard.computeRect()
        Internal.field.adjustView()

    def __setDrawingOrder(self, dummyResults=None):
        """set drawing order of the wall"""
        levels = {'NW': (2, 3, 1, 0), 'NE':(3, 1, 0, 2), 'SE':(1, 0, 2, 3), 'SW':(0, 2, 3, 1)}
        for idx, side in enumerate(self.__sides):
            side.level = (levels[side.lightSource][idx] + 1) * ZValues.boardLevelFactor

    def __moveDividedTile(self, tile, offset):
        """moves a tile from the divide hole to its new place"""
        board = tile.board
        newOffset = tile.xoffset + offset
        sideLength = len(self.tiles) // 8
        if newOffset >= sideLength:
            sideIdx = self.__sides.index(tile.board)
            board = self.__sides[(sideIdx+1) % 4]
        tile.setBoard(board, newOffset % sideLength, 0, level=2)
        tile.update()

    def _placeLooseTiles(self):
        """place the last 2 tiles on top of kong box"""
        assert len(self.kongBox) % 2 == 0
        afterCurrentAnimationDo(self.__placeLooseTiles2)

    def __placeLooseTiles2(self, dummyResult):
        """place the last 2 tiles on top of kong box, no animation is active"""
        placeCount = len(self.kongBox) // 2
        if placeCount >= 4:
            first = min(placeCount-1, 5)
            second = max(first-2, 1)
            self.__moveDividedTile(self.kongBox[-1], second)
            self.__moveDividedTile(self.kongBox[-2], first)

    def divide(self):
        """divides a wall, building a living and and a dead end"""
        with Animated(False):
            Wall.divide(self)
            for tile in self.tiles:
                # update graphics because tiles having been
                # in kongbox in a previous game
                # might not be there anymore. This gets rid
                # of the cross on them.
                tile.update()
            # move last two tiles onto the dead end:
            return animate().addCallback(self.__placeLooseTiles2)

    def decorate(self):
        """show player info on the wall"""
        for player in self.game.players:
            self.decoratePlayer(player)

    def decoratePlayer(self, player):
        """show player info on the wall"""
        side = player.front
        sideCenter = side.center()
        name = side.nameLabel
        name.setText(' - '.join([player.localName, unicode(player.handTotalForWall())]))
        name.resetTransform()
        if side.rotation() == 180:
            rotateCenter(name, 180)
        nameRect = QRectF()
        nameRect.setSize(name.mapToParent(name.boundingRect()).boundingRect().size())
        name.setPos(sideCenter - nameRect.center())
        player.colorizeName()
        side.windTile.setWind(player.wind, self.game.roundsFinished)
        side.windTile.resetTransform()
        side.windTile.setPos(sideCenter.x()*1.63, sideCenter.y()-side.windTile.rect().height()/2.5)
        side.nameLabel.show()
        side.windTile.show()
