#include <qstringlist.h>

#include <kdebug.h>
#include <klocale.h>
#include <kinstance.h>

#include <kaction.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <kdirlister.h>
#include <konq_dirpart.h>
#include <kparts/browserextension.h>

#include "dirfilterplugin.h"

DirFilterPlugin::DirFilterPlugin( QObject* parent, const char* name )
                :KParts::Plugin( parent, name )
{
  m_pFilterMenu = new KActionMenu( i18n("&View Filter"), "queue",
                                   actionCollection(), "filterdir" );
  m_pFilterMenu->setDelayed( false );
  connect( m_pFilterMenu->popupMenu(), SIGNAL( aboutToShow() ),
           SLOT( slotAboutToShow() ) );
  connect( m_pFilterMenu->popupMenu(), SIGNAL( activated( int ) ),
           SLOT( slotItemSelected( int ) ) );

  KonqDirPart* part = dynamic_cast<KonqDirPart*>(parent);
  if ( part )
  {
    connect( part, SIGNAL( itemRemoved(const KFileItem*) ),
                   SLOT( slotDeleteItem(const KFileItem*) ) );
    connect( part, SIGNAL( itemsAdded(const KFileItemList&) ),
                   SLOT( slotNewItems(const KFileItemList&) ) );
    connect( part, SIGNAL( itemsFilteredByMime(const KFileItemList&) ),
                   SLOT( slotNewItems(const KFileItemList&) ) );
  }
  loadSettings();
}

DirFilterPlugin::~DirFilterPlugin()
{
  saveSettings();
  delete m_pFilterMenu;
}

void DirFilterPlugin::slotAboutToShow()
{
  KonqDirPart* part = dynamic_cast<KonqDirPart*>( parent() );
  if (!part) return;

  QString inodes, label;
  unsigned short inodes_cnt = 0;
  uint count = 0;
  uint enableReset = 0;

  m_pFilterMenu->popupMenu()->clear();
  m_pFilterMenu->popupMenu()->insertTitle(i18n("Only show items of type:"), count++);
  MimeInfoConstIterator it = m_pMimeInfo.begin();
  for ( ; it!= m_pMimeInfo.end() ; ++it )
  {
     kdDebug() << "Mime-type: " << it.key() << endl;
     if( it.key().startsWith("inode") )
    {
      inodes += it.key() + ' ';
      if ( !inodes_cnt )
        inodes_cnt = count++;
      else
        count++;
      continue;
    }
    label = m_bShowCount ? QString("%1  (%2 %3)").arg(it.data().mimeComment).arg(it.data().count).arg(it.data().count==1?"item":"items"):it.data().mimeComment;
    m_pFilterMenu->popupMenu()->insertItem( SmallIconSet(it.data().iconName), label, count++ );
    if ( it.data().useAsFilter )
    {
      m_pFilterMenu->popupMenu()->setItemChecked( count-1, true );
      enableReset += count-1;
    }
  }
  if ( !inodes.isEmpty() )
  {
    QStringList inode_list = QStringList::split( ' ', inodes );
    QStringList::Iterator it = inode_list.begin();
    m_pFilterMenu->popupMenu()->insertSeparator( count++ );
    MimeInfo info;
    for( ; it != inode_list.end(); ++it )
    {
      info = m_pMimeInfo[(*it)];
      if ( m_bShowCount )
        label = QString("%1  (%2 %3)").arg(info.mimeComment).arg(info.count).arg(info.count==1?i18n("item"):i18n("items"));
      else
        label=info.mimeComment;
      m_pFilterMenu->popupMenu()->insertItem( SmallIconSet(info.iconName), label, inodes_cnt++ );
      if ( info.useAsFilter )
      {
        m_pFilterMenu->popupMenu()->setItemChecked( inodes_cnt-1, true );
        enableReset += inodes_cnt-1;
      }
    }
  }
  m_pFilterMenu->popupMenu()->insertSeparator( count++ );
  m_pFilterMenu->popupMenu()->insertItem( i18n("Use Multiple Filters"), this, SLOT(slotMultipleFilters()), 0, count );
  m_pFilterMenu->popupMenu()->setItemEnabled( count, !enableReset );
  m_pFilterMenu->popupMenu()->setItemChecked( count++, m_bUseMultipleFilters );
  label = m_bShowCount ? i18n("Hide Details") : i18n("Show Details");
  m_pFilterMenu->popupMenu()->insertItem( label, this, SLOT(slotShowCount()), 0, count++ );
  m_pFilterMenu->popupMenu()->insertItem( i18n("Reset"), this, SLOT(slotReset()), 0, count );
  m_pFilterMenu->popupMenu()->setItemEnabled( count, enableReset );
}

void DirFilterPlugin::slotItemSelected( int id )
{
  int count = m_pFilterMenu->popupMenu()->count();
  if ( id == 0 || id > count-5 ) return;

  // A new UA string was selected...
  KonqDirPart* part = dynamic_cast<KonqDirPart*>(parent());
  if (!part) return;

  count = 1;
  MimeInfoIterator it = m_pMimeInfo.begin();
  while( count++ < id && it != m_pMimeInfo.end() ) ++it;
  if ( it != m_pMimeInfo.end() )
  {
    if ( it.data().useAsFilter )
    {
      it.data().useAsFilter = false;
      QString filters = part->mimeFilter();
      int pos = filters.find( it.key() );
      if ( pos != -1 )
      {
        filters.remove( pos, it.key().length() );
        part->setMimeFilter( filters.stripWhiteSpace() );
      }
    }
    else
    {
      it.data().useAsFilter = true;
      if ( m_bUseMultipleFilters )
      {
        part->setMimeFilter( part->mimeFilter() + ' ' + it.key() );
      }
      else
      {
        MimeInfoIterator oit = m_pMimeInfo.find( part->mimeFilter() );
        if ( oit != m_pMimeInfo.end() )
          oit.data().useAsFilter = false;
        part->setMimeFilter( it.key().stripWhiteSpace() );
      }
    }
    part->openURL( part->url() );
  }
}

void DirFilterPlugin::slotNewItems( const KFileItemList& list )
{
  KonqDirPart* part = dynamic_cast<KonqDirPart*>(parent());

  if( list.count() == 0 || !part )
    return;

  m_pFilterMenu->setEnabled( part->nameFilter().isEmpty() );

  if ( part && !part->nameFilter().isEmpty() )
    return;

  QString curl = part->url().url();
  if ( m_pURL.isEmpty() || curl != m_pURL )
  {
    m_pURL = curl;
    m_pMimeInfo.clear();
    m_pItemList.clear();
  }

  QString mime_type, url;
  KFileItemListIterator it(list);
  QStringList filter = QStringList::split(' ', part->mimeFilter());
  for ( ; it.current() ; ++it )
  {
    mime_type = it.current()->mimetype();
    if ( mime_type.isEmpty() ) continue;
    url = it.current()->url().url();
    if ( !m_pMimeInfo.contains(mime_type) )
    {
      MimeInfo info;
      info.count++;
      info.useAsFilter = (!filter.isEmpty() && filter.contains(mime_type));
      info.mimeComment = it.current()->mimeComment();
      info.iconName = it.current()->iconName();
      m_pMimeInfo.insert( mime_type, info );
      m_pItemList.append( url );
    }
    else
    {
      if ( m_pItemList.contains(url) == 0 )
      {
        m_pMimeInfo[mime_type].count++;
        m_pItemList.append( url );
      }
    }
  }
}

void DirFilterPlugin::slotDeleteItem( const KFileItem* item )
{
  QString mime_type = item->mimetype();
  if ( item && m_pMimeInfo.contains( mime_type ) )
  {
    MimeInfo info = m_pMimeInfo[mime_type];
    if ( info.count == 1 )
    {
      m_pMimeInfo.remove( mime_type );
      m_pItemList.remove( item->url().url() );
      if ( info.useAsFilter )
      {
        KonqDirPart* part = dynamic_cast<KonqDirPart*>( parent() );
        if ( part )
        {
          QString filters = part->mimeFilter();
          int pos = filters.find( mime_type );
          if ( pos != -1 )
          {
            filters.remove( pos, mime_type.length() );
            part->setMimeFilter( filters.stripWhiteSpace() );
            part->openURL( part->url() );
          }
        }
      }
    }
    else
      m_pMimeInfo[mime_type].count--;
  }
}

void DirFilterPlugin::slotShowCount()
{
  m_bShowCount = !m_bShowCount;
  QString label = m_bShowCount ? i18n( "Hide Details" ) : i18n( "Show Details" );
  m_pFilterMenu->popupMenu()->changeItem( m_pFilterMenu->popupMenu()->count()-2, label );
}

void DirFilterPlugin::slotReset()
{
  KonqDirPart* part = dynamic_cast<KonqDirPart*>( parent() );
  if ( !part ) return;

  part->setMimeFilter( QString::null );
  part->openURL( part->url() );
  MimeInfoIterator it = m_pMimeInfo.begin();
  for ( ; it != m_pMimeInfo.end(); ++it )
    it.data().useAsFilter = false;
}

void DirFilterPlugin::slotMultipleFilters()
{
  m_bUseMultipleFilters = !m_bUseMultipleFilters;
}

void DirFilterPlugin::saveSettings()
{
  KConfig cfg ( "dirfilterrc", false, false );
  cfg.setGroup( "General" );
  cfg.writeEntry( "ShowDetails", m_bShowCount );
  cfg.writeEntry( "UseMultipleFilters", m_bUseMultipleFilters );
}

void DirFilterPlugin::loadSettings()
{
  KConfig cfg ( "dirfilterrc", false, false );
  cfg.setGroup( "General" );
  m_bShowCount = cfg.readBoolEntry( "ShowDetails", false );
  m_bUseMultipleFilters = cfg.readBoolEntry( "UseMultipleFilters", true );
}


DirFilterPluginFactory::DirFilterPluginFactory( QObject* parent, const char* name )
                       :KLibFactory( parent, name )
{
  s_instance = new KInstance("dirfilterplugin");
}

DirFilterPluginFactory::~DirFilterPluginFactory()
{
  delete s_instance;
  s_instance = 0L;
}

QObject* DirFilterPluginFactory::createObject( QObject* parent, const char* name, const char*, const QStringList & )
{
  return new DirFilterPlugin( parent, name );
}

extern "C"
{
  void* init_libdirfilterplugin()
  {
    KGlobal::locale()->insertCatalogue("dirfilterplugin"); 
    return new DirFilterPluginFactory;
  }
}
KInstance* DirFilterPluginFactory::s_instance = 0L;

#include "dirfilterplugin.moc"
